/*
 * bandcampDirect
 * Copyright (C) 2019 Fynn Godau
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 * Not affiliated with bandcamp, Incorporated.
 */

package godau.fynn.bandcampdirect.view;

import android.app.Activity;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.view.View;
import android.widget.ImageButton;
import android.widget.TextView;
import android.widget.Toast;
import godau.fynn.bandcampdirect.R;
import godau.fynn.bandcampdirect.activity.AlbumActivity;
import godau.fynn.bandcampdirect.model.Album;
import godau.fynn.bandcampdirect.service.MusicService;

import static godau.fynn.bandcampdirect.service.MusicService.*;

public class PlayerView {

    private Activity context;

    private Album album;

    private int trackPosition = 0;

    private ImageButton playButton;
    private ImageButton skipButton;
    private ImageButton backtrackButton;
    private TextView trackText;

    private int state = STATE_STOPPED;

    private static final int STATE_STOPPED = 0;
    private static final int STATE_PLAYING = 1;
    private static final int STATE_PAUSED = 2;

    public PlayerView(Activity context, Album album) {
        this.context = context;

        this.album = album;

        init();
    }

    public PlayerView(Activity context, Album album, int trackPosition) {
        this.context = context;

        this.album = album;

        setTrackPosition(trackPosition);

        init();
    }

    private void init() {

        context.registerReceiver(broadcastReceiver, new IntentFilter(context.getPackageName() + ".PLAYBACK_CONTROL"));

        playButton = context.findViewById(R.id.play);
        skipButton = context.findViewById(R.id.skip);
        backtrackButton = context.findViewById(R.id.backtrack);
        trackText = context.findViewById(R.id.trackName);

        playButton.setOnClickListener(view -> {
            switch (state) {

                case STATE_STOPPED:
                    AlbumActivity.play(context, album, trackPosition);
                    updateDisplay();
                    break;

                case STATE_PLAYING:
                    broadcast(ACTION_PAUSE);
                    break;

                case STATE_PAUSED:
                    broadcast(ACTION_PLAY);
                    break;

            }

        });

        skipButton.setOnClickListener(v -> broadcast(ACTION_SKIP_TRACK));
        backtrackButton.setOnClickListener(v -> broadcast(ACTION_BACK_TRACK));

    }

    private void updateDisplay() {

        trackText.setText(album.getTrack(trackPosition).getTitle());

        switch (state) {

            case STATE_STOPPED:
                trackText.setText("");
            case STATE_PAUSED:
                playButton.setImageDrawable(context.getDrawable(R.drawable.ic_play));
                break;
            case STATE_PLAYING:
                playButton.setImageDrawable(context.getDrawable(R.drawable.ic_pause));
                break;
        }
    }

    public void setTrackPosition(int trackPosition) {
        this.trackPosition = trackPosition;
    }

    public void broadcast(int action) {
        Intent intent = new Intent(context.getPackageName() + ".PLAYBACK_CONTROL");
        intent.putExtra(EXTRA_ACTION, action);

        context.sendBroadcast(intent);
    }

    public static void broadcast(Context context, int action) {
        Intent intent = new Intent(context.getPackageName() + ".PLAYBACK_CONTROL");
        intent.putExtra(EXTRA_ACTION, action);

        context.sendBroadcast(intent);
    }

    public static void broadcast(Context context, int action, int position) {
        Intent intent = new Intent(context.getPackageName() + ".PLAYBACK_CONTROL");
        intent.putExtra(EXTRA_ACTION, action);
        intent.putExtra(EXTRA_POSITION, position);

        context.sendBroadcast(intent);
    }


    public void onDestroy() {
        try {
            context.unregisterReceiver(broadcastReceiver);
        } catch (IllegalArgumentException e) {
            // Receiver has not been registered…?
            e.printStackTrace();
        }
    }

    private BroadcastReceiver broadcastReceiver = new BroadcastReceiver() {
        @Override
        public void onReceive(Context context, Intent intent) {
            int action = intent.getIntExtra(EXTRA_ACTION, ACTION_UNDEFINED);

            if (intent.hasExtra(EXTRA_POSITION)) {
                trackPosition = intent.getIntExtra(EXTRA_POSITION, 0);
            }

            switch (action) {
                case ACTION_PAUSE:
                    state = STATE_PAUSED;
                    break;

                case ACTION_PLAY:
                    state = STATE_PLAYING;
                    break;

                case ACTION_SKIP_TRACK:
                    if (!MusicService.isLastTrack(trackPosition, album.getTracks().size())) {
                        trackPosition++;
                        state = STATE_PLAYING;
                    }
                    break;

                case ACTION_BACK_TRACK:
                    if (trackPosition > 0) {
                        trackPosition--;
                        state = STATE_PLAYING;
                    }
                    break;

                case ACTION_END:
                    state = STATE_STOPPED;
            }

            updateDisplay();
        }
    };
}
