/*
 * campfire, formerly known as bandcampDirect
 * Copyright (C) 2020 Fynn Godau
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 * Not affiliated with bandcamp, Incorporated.
 */

package godau.fynn.bandcampdirect.adapter;

import android.content.Intent;
import android.net.Uri;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.recyclerview.widget.RecyclerView;

import godau.fynn.bandcampdirect.R;
import godau.fynn.bandcampdirect.activity.AlbumActivity;
import godau.fynn.bandcampdirect.model.Album;
import godau.fynn.bandcampdirect.model.MetadataDisplayable;
import godau.fynn.bandcampdirect.model.RowViewDisplayable;
import godau.fynn.bandcampdirect.model.Track;
import godau.fynn.bandcampdirect.view.RowView;
import godau.fynn.typedrecyclerview.TypedRecyclerViewAdapter;

import static android.view.View.GONE;

public class AlbumAdapter extends TypedRecyclerViewAdapter<Object> {

    private final Album album;

    private final TypeHandler rowViewHandler = new RowViewHandler(), infoHandler = new InfoHandler();

    /**
     * Constructs an AlbumAdapter that displays an Album
     */
    public AlbumAdapter(Album a) {
        super();

        this.album = a;

        content.addAll(a.getTracks());

        if (a.getAbout() != null) {
            content.add(new MetadataDisplayable.Info(a.getAbout()));
        }

    }

    /**
     * Constructs an AlbumAdapter that displays a Track
     */
    public AlbumAdapter(Album a, Track t) {
        super();

        this.album = a;

        if (t.hasLyrics()) {
            content.add(new MetadataDisplayable.Lyrics(t.getLyrics()));
        }


        if (t.getInfo() != null) {
            content.add(new MetadataDisplayable.Info(t.getInfo()));
        }
    }

    @Override
    protected TypeHandler getItemHandler(Object item, int position) {
        if (item instanceof MetadataDisplayable.Text)
            return infoHandler;
        else return rowViewHandler;
    }

    private class RowViewHandler extends TypeHandler<RowView, RowViewDisplayable> {

        @Override
        public RowView createViewHolder(@NonNull ViewGroup parent) {
            return new RowView(context);
        }

        @Override
        public void bindViewHolder(@NonNull RowView rowView, RowViewDisplayable item, int position) {
            Track track = (Track) content.get(position);
            rowView.bind(track.getTitle(), track.isPaid(),
                    track.hasExtras(album) ? RowView.Action.OPEN : RowView.Action.NONE
            );

            rowView.setOnItemClickListener(v -> AlbumActivity.play(context, album, position));

            rowView.setOnButtonClickListener(v -> {
                Intent intent = new Intent(context, AlbumActivity.class);
                intent.setData(Uri.parse(album.getTrackUrl(album.getTrack(position))));

                intent.putExtra(AlbumActivity.EXTRA_ALBUM, album);
                intent.putExtra(AlbumActivity.EXTRA_TRACK, position);
                context.startActivity(intent);
            });

            RowAdapter.loadImage(rowView, track, track.getCover());
        }
    }

    private class InfoHandler extends TypeHandler<InfoHandler.ViewHolder, MetadataDisplayable.Text> {

        @Override
        public InfoHandler.ViewHolder createViewHolder(@NonNull ViewGroup parent) {
            return new InfoHandler.ViewHolder(LayoutInflater.from(context).inflate(R.layout.row_info, parent, false)) {
            };
        }

        @Override
        public void bindViewHolder(@NonNull InfoHandler.ViewHolder holder, MetadataDisplayable.Text item, int position) {

            if (item instanceof MetadataDisplayable.Lyrics
                    && item.getContentText().toLowerCase().startsWith("lyrics")) {
                holder.title.setVisibility(GONE);
            } else {
                holder.title.setVisibility(View.VISIBLE);
                holder.title.setText(item.getHeaderText());
            }

            holder.info.setText(item.getContentText());

            if (position == 0)
                holder.divider.setVisibility(GONE);
            else
                holder.divider.setVisibility(View.VISIBLE);
        }

        public class ViewHolder extends RecyclerView.ViewHolder {
            TextView title;
            TextView info;
            View divider;

            public ViewHolder(@NonNull View itemView) {
                super(itemView);

                title = itemView.findViewById(R.id.title);
                info = itemView.findViewById(R.id.info);
                divider = itemView.findViewById(R.id.divider);
            }
        }
    }
}
