/*
 * campfire, formerly known as bandcampDirect
 * Copyright (C) 2020 Fynn Godau
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 * Not affiliated with bandcamp, Incorporated.
 */

package godau.fynn.bandcampdirect.adapter;

import android.content.Intent;
import android.net.Uri;
import android.os.Handler;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.recyclerview.widget.RecyclerView;

import godau.fynn.bandcampdirect.R;
import godau.fynn.bandcampdirect.activity.AlbumActivity;
import godau.fynn.bandcampdirect.bandcamp.Bandcamp;
import godau.fynn.bandcampdirect.model.Artist;
import godau.fynn.bandcampdirect.model.MetadataDisplayable;
import godau.fynn.bandcampdirect.model.RowViewDisplayable;
import godau.fynn.bandcampdirect.view.RowView;
import godau.fynn.typedrecyclerview.TypedRecyclerViewAdapter;

public class ArtistAdapter extends TypedRecyclerViewAdapter<Object> {

    private final Artist artist;

    private final TypeHandler rowViewHandler = new RowViewHandler(), infoHandler = new InfoHandler(), headerHandler = new HeaderHandler();

    public ArtistAdapter(Artist artist) {
        super();

        this.artist = artist;
        String info = artist.getBiography();

        content.addAll(artist.getDiscography());

        if (info != null)
            content.add(new MetadataDisplayable.Info(info));

        if (artist.getShows().size() > 0) {
            content.add(new MetadataDisplayable.ShowHeader());
            content.addAll(artist.getShows());
        }

        if (artist.getSites().size() > 0) {
            content.add(new MetadataDisplayable.SiteHeader());
            content.addAll(artist.getSites());
        }
    }

    @Override
    protected TypeHandler getItemHandler(Object item, int position) {
        if (item instanceof MetadataDisplayable.Text)
            return infoHandler;
        else if (item instanceof MetadataDisplayable.Header)
            return headerHandler;
        else return rowViewHandler;
    }

    private class RowViewHandler extends TypeHandler<RowView, RowViewDisplayable> {

        @Override
        public RowView createViewHolder(@NonNull ViewGroup parent) {
            return new RowView(context);
        }

        @Override
        public void bindViewHolder(@NonNull RowView rowView, RowViewDisplayable item, int position) {
            rowView.bind(item.getTitle(), false, item.getAction());

            rowView.setOnButtonClickListener(v -> {

                if (item instanceof Artist.Discograph) {
                    ((Artist.Discograph) item).getUrl(new Bandcamp(null), new Handler(msg -> {
                        String url = msg.getData().getString("url");

                        Intent intent = new Intent(context, AlbumActivity.class);
                        intent.setData(Uri.parse(url));
                        context.startActivity(intent);

                        return false;
                    }), null);
                } else {
                    Intent intent = new Intent(Intent.ACTION_VIEW, Uri.parse(item.getUrl()));
                    context.startActivity(intent);
                }
            });

            RowAdapter.loadImage(rowView, item, item.getCover());
        }
    }

    public class InfoHandler extends TypeHandler<InfoHandler.ViewHolder, MetadataDisplayable> {

        @Override
        public ViewHolder createViewHolder(@NonNull ViewGroup parent) {
            return new ViewHolder(LayoutInflater.from(context).inflate(R.layout.row_info, parent, false));
        }

        @Override
        public void bindViewHolder(@NonNull InfoHandler.ViewHolder holder, MetadataDisplayable item, int position) {
            holder.info.setText(artist.getBiography());
        }

        public class ViewHolder extends RecyclerView.ViewHolder {
            TextView info;

            public ViewHolder(@NonNull View itemView) {
                super(itemView);
                info = itemView.findViewById(R.id.info);
            }
        }
    }

    public class HeaderHandler extends TypeHandler<HeaderHandler.ViewHolder, MetadataDisplayable.Header> {

        @Override
        public ViewHolder createViewHolder(@NonNull ViewGroup parent) {
            return new HeaderHandler.ViewHolder(LayoutInflater.from(context).inflate(R.layout.row_header, parent, false));
        }

        @Override
        public void bindViewHolder(@NonNull ArtistAdapter.HeaderHandler.ViewHolder holder, MetadataDisplayable.Header item, int position) {
            holder.title.setText(item.getHeaderText());
        }

        public class ViewHolder extends RecyclerView.ViewHolder {
            TextView title;

            public ViewHolder(@NonNull View itemView) {
                super(itemView);
                title = itemView.findViewById(R.id.title);
            }
        }
    }
}
