/*************************************************************************/
/*  key_mapping_x11.cpp                                                  */
/*************************************************************************/
/*                       This file is part of:                           */
/*                           GODOT ENGINE                                */
/*                    http://www.godotengine.org                         */
/*************************************************************************/
/* Copyright (c) 2007-2017 Juan Linietsky, Ariel Manzur.                 */
/*                                                                       */
/* Permission is hereby granted, free of charge, to any person obtaining */
/* a copy of this software and associated documentation files (the       */
/* "Software"), to deal in the Software without restriction, including   */
/* without limitation the rights to use, copy, modify, merge, publish,   */
/* distribute, sublicense, and/or sell copies of the Software, and to    */
/* permit persons to whom the Software is furnished to do so, subject to */
/* the following conditions:                                             */
/*                                                                       */
/* The above copyright notice and this permission notice shall be        */
/* included in all copies or substantial portions of the Software.       */
/*                                                                       */
/* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,       */
/* EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF    */
/* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.*/
/* IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY  */
/* CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,  */
/* TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE     */
/* SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.                */
/*************************************************************************/
#include "key_mapping_x11.h"


/***** SCAN CODE CONVERSION ******/

struct _XTranslatePair {

	KeySym keysym;
	unsigned int keycode;
};

static _XTranslatePair _xkeysym_to_keycode[]={
  // misc keys

	{ XK_Escape,                  KEY_ESCAPE   },
	{ XK_Tab,                     KEY_TAB   },
	{ XK_ISO_Left_Tab,            KEY_BACKTAB   },
	{ XK_BackSpace,               KEY_BACKSPACE   },
	{ XK_Return,                  KEY_RETURN   },
	{ XK_Insert,                  KEY_INSERT   },
	{ XK_Delete,                  KEY_DELETE   },
	{ XK_Clear,                   KEY_DELETE   },
	{ XK_Pause,                   KEY_PAUSE   },
	{ XK_Print,                   KEY_PRINT   },
	{ XK_Home,                    KEY_HOME   },
	{ XK_End,                     KEY_END   },
	{ XK_Left,                    KEY_LEFT   },
	{ XK_Up,                      KEY_UP   },
	{ XK_Right,                   KEY_RIGHT   },
	{ XK_Down,                    KEY_DOWN   },
	{ XK_Prior,                   KEY_PAGEUP   },
	{ XK_Next,                    KEY_PAGEDOWN   },
	{ XK_Shift_L,                 KEY_SHIFT   },
	{ XK_Shift_R,                 KEY_SHIFT   },
	{ XK_Shift_Lock,              KEY_SHIFT   },
	{ XK_Control_L,               KEY_CONTROL   },
	{ XK_Control_R,               KEY_CONTROL   },
	{ XK_Meta_L,                  KEY_META   },
	{ XK_Meta_R,                  KEY_META   },
	{ XK_Alt_L,                   KEY_ALT   },
	{ XK_Alt_R,                   KEY_ALT   },
	{ XK_Caps_Lock,               KEY_CAPSLOCK   },
	{ XK_Num_Lock,                KEY_NUMLOCK   },
	{ XK_Scroll_Lock,             KEY_SCROLLLOCK   },
	{ XK_Super_L,                 KEY_SUPER_L   },
	{ XK_Super_R,                 KEY_SUPER_R   },
	{ XK_Menu,                    KEY_MENU   },
	{ XK_Hyper_L,                 KEY_HYPER_L   },
	{ XK_Hyper_R,                 KEY_HYPER_R   },
	{ XK_Help,                    KEY_HELP   },
	{ XK_KP_Space,                KEY_SPACE   },
	{ XK_KP_Tab,                  KEY_TAB   },
	{ XK_KP_Enter,                KEY_ENTER   },
	{ XK_Home,                 KEY_HOME   },
	{ XK_Left,                 KEY_LEFT   },
	{ XK_Up,                   KEY_UP   },
	{ XK_Right,                KEY_RIGHT   },
	{ XK_Down,                 KEY_DOWN   },
	{ XK_Prior,                KEY_PAGEUP   },
	{ XK_Next,                 KEY_PAGEDOWN   },
	{ XK_End,                  KEY_END   },
	{ XK_Begin,                KEY_CLEAR   },
	{ XK_Insert,               KEY_INSERT   },
	{ XK_Delete,               KEY_DELETE   },
//	{ XK_KP_Equal,                KEY_EQUAL   },
//	{ XK_KP_Separator,            KEY_COMMA   },
	{ XK_KP_Decimal,              KEY_KP_PERIOD   },
	{ XK_KP_Delete,              KEY_KP_PERIOD   },
	{ XK_KP_Enter,		      KEY_KP_ENTER },
	{ XK_KP_Multiply,	      KEY_KP_MULTIPLY},
	{ XK_KP_Divide,		     KEY_KP_DIVIDE},
	{ XK_KP_Subtract,		KEY_KP_SUBTRACT},
	{ XK_KP_Add,		KEY_KP_ADD},
	{ XK_KP_0,		KEY_KP_0},
	{ XK_KP_1,		KEY_KP_1},
	{ XK_KP_2,		KEY_KP_2},
	{ XK_KP_3,		KEY_KP_3},
	{ XK_KP_4,		KEY_KP_4},
	{ XK_KP_5,		KEY_KP_5},
	{ XK_KP_6,		KEY_KP_6},
	{ XK_KP_7,		KEY_KP_7},
	{ XK_KP_8,		KEY_KP_8},
	{ XK_KP_9,		KEY_KP_9},
	// same but with numlock
	{ XK_KP_Insert,		KEY_KP_0},
	{ XK_KP_End,		KEY_KP_1},
	{ XK_KP_Down,		KEY_KP_2},
	{ XK_KP_Page_Down,	KEY_KP_3},
	{ XK_KP_Left,		KEY_KP_4},
	{ XK_KP_Begin,		KEY_KP_5},
	{ XK_KP_Right,		KEY_KP_6},
	{ XK_KP_Home,		KEY_KP_7},
	{ XK_KP_Up,		KEY_KP_8},
	{ XK_KP_Page_Up,		KEY_KP_9},
	{ XK_F1,		KEY_F1},
	{ XK_F2,		KEY_F2},
	{ XK_F3,		KEY_F3},
	{ XK_F4,		KEY_F4},
	{ XK_F5,		KEY_F5},
	{ XK_F6,		KEY_F6},
	{ XK_F7,		KEY_F7},
	{ XK_F8,		KEY_F8},
	{ XK_F9,		KEY_F9},
	{ XK_F10,		KEY_F10},
	{ XK_F11,		KEY_F11},
	{ XK_F12,		KEY_F12},
	{ XK_F13,		KEY_F13},
	{ XK_F14,		KEY_F14},
	{ XK_F15,		KEY_F15},
	{ XK_F16,		KEY_F16},

    // media keys
	{ XF86XK_Back,                KEY_BACK   },
	{ XF86XK_Forward,             KEY_FORWARD   },
	{ XF86XK_Stop,                KEY_STOP   },
	{ XF86XK_Refresh,             KEY_REFRESH   },
	{ XF86XK_Favorites,           KEY_FAVORITES   },
	{ XF86XK_AudioMedia,          KEY_LAUNCHMEDIA   },
	{ XF86XK_OpenURL,             KEY_OPENURL   },
	{ XF86XK_HomePage,            KEY_HOMEPAGE   },
	{ XF86XK_Search,              KEY_SEARCH   },
	{ XF86XK_AudioLowerVolume,    KEY_VOLUMEDOWN   },
	{ XF86XK_AudioMute,           KEY_VOLUMEMUTE   },
	{ XF86XK_AudioRaiseVolume,    KEY_VOLUMEUP   },
	{ XF86XK_AudioPlay,           KEY_MEDIAPLAY   },
	{ XF86XK_AudioStop,           KEY_MEDIASTOP   },
	{ XF86XK_AudioPrev,           KEY_MEDIAPREVIOUS   },
	{ XF86XK_AudioNext,           KEY_MEDIANEXT   },
	{ XF86XK_AudioRecord,         KEY_MEDIARECORD   },

    // launch keys
	{ XF86XK_Mail,                KEY_LAUNCHMAIL   },
	{ XF86XK_MyComputer,          KEY_LAUNCH0   },
	{ XF86XK_Calculator,          KEY_LAUNCH1   },
	{ XF86XK_Standby,             KEY_STANDBY   },

	{ XF86XK_Launch0,             KEY_LAUNCH2   },
	{ XF86XK_Launch1,             KEY_LAUNCH3   },
	{ XF86XK_Launch2,             KEY_LAUNCH4   },
	{ XF86XK_Launch3,             KEY_LAUNCH5   },
	{ XF86XK_Launch4,             KEY_LAUNCH6   },
	{ XF86XK_Launch5,             KEY_LAUNCH7   },
	{ XF86XK_Launch6,             KEY_LAUNCH8   },
	{ XF86XK_Launch7,             KEY_LAUNCH9   },
	{ XF86XK_Launch8,             KEY_LAUNCHA   },
	{ XF86XK_Launch9,             KEY_LAUNCHB   },
	{ XF86XK_LaunchA,             KEY_LAUNCHC   },
	{ XF86XK_LaunchB,             KEY_LAUNCHD   },
	{ XF86XK_LaunchC,             KEY_LAUNCHE   },
	{ XF86XK_LaunchD,             KEY_LAUNCHF   },

	{0,                          0  }
};


unsigned int KeyMappingX11::get_keycode(KeySym p_keysym) {

	// kinda bruteforce.. could optimize.

	if (p_keysym<0x100) // Latin 1, maps 1-1
		return p_keysym;

	// look for special key
	for(int idx=0;_xkeysym_to_keycode[idx].keysym!=0;idx++) {

		if (_xkeysym_to_keycode[idx].keysym==p_keysym)
			return  _xkeysym_to_keycode[idx].keycode;
	}

	return 0;
}

KeySym KeyMappingX11::get_keysym(unsigned int p_code) {

	// kinda bruteforce.. could optimize.

	if (p_code<0x100) // Latin 1, maps 1-1
		return p_code;

	// look for special key
	for(int idx=0;_xkeysym_to_keycode[idx].keysym!=0;idx++) {

		if (_xkeysym_to_keycode[idx].keycode==p_code)
			return  _xkeysym_to_keycode[idx].keysym;
	}

	return 0;
}


/***** UNICODE CONVERSION ******/

// Tables taken from FOX toolkit

struct _XTranslateUnicodePair {

	KeySym keysym;
	unsigned int unicode;
};

enum {

	_KEYSYM_MAX=759
};

static _XTranslateUnicodePair _xkeysym_to_unicode[_KEYSYM_MAX] = {
	{ 0x01A1, 0x0104 },
	{ 0x01A2, 0x02D8 },
	{ 0x01A3, 0x0141 },
	{ 0x01A5, 0x013D },
	{ 0x01A6, 0x015A },
	{ 0x01A9, 0x0160 },
	{ 0x01AA, 0x015E },
	{ 0x01AB, 0x0164 },
	{ 0x01AC, 0x0179 },
	{ 0x01AE, 0x017D },
	{ 0x01AF, 0x017B },
	{ 0x01B1, 0x0105 },
	{ 0x01B2, 0x02DB },
	{ 0x01B3, 0x0142 },
	{ 0x01B5, 0x013E },
	{ 0x01B6, 0x015B },
	{ 0x01B7, 0x02C7 },
	{ 0x01B9, 0x0161 },
	{ 0x01BA, 0x015F },
	{ 0x01BB, 0x0165 },
	{ 0x01BC, 0x017A },
	{ 0x01BD, 0x02DD },
	{ 0x01BE, 0x017E },
	{ 0x01BF, 0x017C },
	{ 0x01C0, 0x0154 },
	{ 0x01C3, 0x0102 },
	{ 0x01C5, 0x0139 },
	{ 0x01C6, 0x0106 },
	{ 0x01C8, 0x010C },
	{ 0x01CA, 0x0118 },
	{ 0x01CC, 0x011A },
	{ 0x01CF, 0x010E },
	{ 0x01D0, 0x0110 },
	{ 0x01D1, 0x0143 },
	{ 0x01D2, 0x0147 },
	{ 0x01D5, 0x0150 },
	{ 0x01D8, 0x0158 },
	{ 0x01D9, 0x016E },
	{ 0x01DB, 0x0170 },
	{ 0x01DE, 0x0162 },
	{ 0x01E0, 0x0155 },
	{ 0x01E3, 0x0103 },
	{ 0x01E5, 0x013A },
	{ 0x01E6, 0x0107 },
	{ 0x01E8, 0x010D },
	{ 0x01EA, 0x0119 },
	{ 0x01EC, 0x011B },
	{ 0x01EF, 0x010F },
	{ 0x01F0, 0x0111 },
	{ 0x01F1, 0x0144 },
	{ 0x01F2, 0x0148 },
	{ 0x01F5, 0x0151 },
	{ 0x01F8, 0x0159 },
	{ 0x01F9, 0x016F },
	{ 0x01FB, 0x0171 },
	{ 0x01FE, 0x0163 },
	{ 0x01FF, 0x02D9 },
	{ 0x02A1, 0x0126 },
	{ 0x02A6, 0x0124 },
	{ 0x02A9, 0x0130 },
	{ 0x02AB, 0x011E },
	{ 0x02AC, 0x0134 },
	{ 0x02B1, 0x0127 },
	{ 0x02B6, 0x0125 },
	{ 0x02B9, 0x0131 },
	{ 0x02BB, 0x011F },
	{ 0x02BC, 0x0135 },
	{ 0x02C5, 0x010A },
	{ 0x02C6, 0x0108 },
	{ 0x02D5, 0x0120 },
	{ 0x02D8, 0x011C },
	{ 0x02DD, 0x016C },
	{ 0x02DE, 0x015C },
	{ 0x02E5, 0x010B },
	{ 0x02E6, 0x0109 },
	{ 0x02F5, 0x0121 },
	{ 0x02F8, 0x011D },
	{ 0x02FD, 0x016D },
	{ 0x02FE, 0x015D },
	{ 0x03A2, 0x0138 },
	{ 0x03A3, 0x0156 },
	{ 0x03A5, 0x0128 },
	{ 0x03A6, 0x013B },
	{ 0x03AA, 0x0112 },
	{ 0x03AB, 0x0122 },
	{ 0x03AC, 0x0166 },
	{ 0x03B3, 0x0157 },
	{ 0x03B5, 0x0129 },
	{ 0x03B6, 0x013C },
	{ 0x03BA, 0x0113 },
	{ 0x03BB, 0x0123 },
	{ 0x03BC, 0x0167 },
	{ 0x03BD, 0x014A },
	{ 0x03BF, 0x014B },
	{ 0x03C0, 0x0100 },
	{ 0x03C7, 0x012E },
	{ 0x03CC, 0x0116 },
	{ 0x03CF, 0x012A },
	{ 0x03D1, 0x0145 },
	{ 0x03D2, 0x014C },
	{ 0x03D3, 0x0136 },
	{ 0x03D9, 0x0172 },
	{ 0x03DD, 0x0168 },
	{ 0x03DE, 0x016A },
	{ 0x03E0, 0x0101 },
	{ 0x03E7, 0x012F },
	{ 0x03EC, 0x0117 },
	{ 0x03EF, 0x012B },
	{ 0x03F1, 0x0146 },
	{ 0x03F2, 0x014D },
	{ 0x03F3, 0x0137 },
	{ 0x03F9, 0x0173 },
	{ 0x03FD, 0x0169 },
	{ 0x03FE, 0x016B },
	{ 0x047E, 0x203E },
	{ 0x04A1, 0x3002 },
	{ 0x04A2, 0x300C },
	{ 0x04A3, 0x300D },
	{ 0x04A4, 0x3001 },
	{ 0x04A5, 0x30FB },
	{ 0x04A6, 0x30F2 },
	{ 0x04A7, 0x30A1 },
	{ 0x04A8, 0x30A3 },
	{ 0x04A9, 0x30A5 },
	{ 0x04AA, 0x30A7 },
	{ 0x04AB, 0x30A9 },
	{ 0x04AC, 0x30E3 },
	{ 0x04AD, 0x30E5 },
	{ 0x04AE, 0x30E7 },
	{ 0x04AF, 0x30C3 },
	{ 0x04B0, 0x30FC },
	{ 0x04B1, 0x30A2 },
	{ 0x04B2, 0x30A4 },
	{ 0x04B3, 0x30A6 },
	{ 0x04B4, 0x30A8 },
	{ 0x04B5, 0x30AA },
	{ 0x04B6, 0x30AB },
	{ 0x04B7, 0x30AD },
	{ 0x04B8, 0x30AF },
	{ 0x04B9, 0x30B1 },
	{ 0x04BA, 0x30B3 },
	{ 0x04BB, 0x30B5 },
	{ 0x04BC, 0x30B7 },
	{ 0x04BD, 0x30B9 },
	{ 0x04BE, 0x30BB },
	{ 0x04BF, 0x30BD },
	{ 0x04C0, 0x30BF },
	{ 0x04C1, 0x30C1 },
	{ 0x04C2, 0x30C4 },
	{ 0x04C3, 0x30C6 },
	{ 0x04C4, 0x30C8 },
	{ 0x04C5, 0x30CA },
	{ 0x04C6, 0x30CB },
	{ 0x04C7, 0x30CC },
	{ 0x04C8, 0x30CD },
	{ 0x04C9, 0x30CE },
	{ 0x04CA, 0x30CF },
	{ 0x04CB, 0x30D2 },
	{ 0x04CC, 0x30D5 },
	{ 0x04CD, 0x30D8 },
	{ 0x04CE, 0x30DB },
	{ 0x04CF, 0x30DE },
	{ 0x04D0, 0x30DF },
	{ 0x04D1, 0x30E0 },
	{ 0x04D2, 0x30E1 },
	{ 0x04D3, 0x30E2 },
	{ 0x04D4, 0x30E4 },
	{ 0x04D5, 0x30E6 },
	{ 0x04D6, 0x30E8 },
	{ 0x04D7, 0x30E9 },
	{ 0x04D8, 0x30EA },
	{ 0x04D9, 0x30EB },
	{ 0x04DA, 0x30EC },
	{ 0x04DB, 0x30ED },
	{ 0x04DC, 0x30EF },
	{ 0x04DD, 0x30F3 },
	{ 0x04DE, 0x309B },
	{ 0x04DF, 0x309C },
	{ 0x05AC, 0x060C },
	{ 0x05BB, 0x061B },
	{ 0x05BF, 0x061F },
	{ 0x05C1, 0x0621 },
	{ 0x05C2, 0x0622 },
	{ 0x05C3, 0x0623 },
	{ 0x05C4, 0x0624 },
	{ 0x05C5, 0x0625 },
	{ 0x05C6, 0x0626 },
	{ 0x05C7, 0x0627 },
	{ 0x05C8, 0x0628 },
	{ 0x05C9, 0x0629 },
	{ 0x05CA, 0x062A },
	{ 0x05CB, 0x062B },
	{ 0x05CC, 0x062C },
	{ 0x05CD, 0x062D },
	{ 0x05CE, 0x062E },
	{ 0x05CF, 0x062F },
	{ 0x05D0, 0x0630 },
	{ 0x05D1, 0x0631 },
	{ 0x05D2, 0x0632 },
	{ 0x05D3, 0x0633 },
	{ 0x05D4, 0x0634 },
	{ 0x05D5, 0x0635 },
	{ 0x05D6, 0x0636 },
	{ 0x05D7, 0x0637 },
	{ 0x05D8, 0x0638 },
	{ 0x05D9, 0x0639 },
	{ 0x05DA, 0x063A },
	{ 0x05E0, 0x0640 },
	{ 0x05E1, 0x0641 },
	{ 0x05E2, 0x0642 },
	{ 0x05E3, 0x0643 },
	{ 0x05E4, 0x0644 },
	{ 0x05E5, 0x0645 },
	{ 0x05E6, 0x0646 },
	{ 0x05E7, 0x0647 },
	{ 0x05E8, 0x0648 },
	{ 0x05E9, 0x0649 },
	{ 0x05EA, 0x064A },
	{ 0x05EB, 0x064B },
	{ 0x05EC, 0x064C },
	{ 0x05ED, 0x064D },
	{ 0x05EE, 0x064E },
	{ 0x05EF, 0x064F },
	{ 0x05F0, 0x0650 },
	{ 0x05F1, 0x0651 },
	{ 0x05F2, 0x0652 },
	{ 0x06A1, 0x0452 },
	{ 0x06A2, 0x0453 },
	{ 0x06A3, 0x0451 },
	{ 0x06A4, 0x0454 },
	{ 0x06A5, 0x0455 },
	{ 0x06A6, 0x0456 },
	{ 0x06A7, 0x0457 },
	{ 0x06A8, 0x0458 },
	{ 0x06A9, 0x0459 },
	{ 0x06AA, 0x045A },
	{ 0x06AB, 0x045B },
	{ 0x06AC, 0x045C },
	{ 0x06AE, 0x045E },
	{ 0x06AF, 0x045F },
	{ 0x06B0, 0x2116 },
	{ 0x06B1, 0x0402 },
	{ 0x06B2, 0x0403 },
	{ 0x06B3, 0x0401 },
	{ 0x06B4, 0x0404 },
	{ 0x06B5, 0x0405 },
	{ 0x06B6, 0x0406 },
	{ 0x06B7, 0x0407 },
	{ 0x06B8, 0x0408 },
	{ 0x06B9, 0x0409 },
	{ 0x06BA, 0x040A },
	{ 0x06BB, 0x040B },
	{ 0x06BC, 0x040C },
	{ 0x06BE, 0x040E },
	{ 0x06BF, 0x040F },
	{ 0x06C0, 0x044E },
	{ 0x06C1, 0x0430 },
	{ 0x06C2, 0x0431 },
	{ 0x06C3, 0x0446 },
	{ 0x06C4, 0x0434 },
	{ 0x06C5, 0x0435 },
	{ 0x06C6, 0x0444 },
	{ 0x06C7, 0x0433 },
	{ 0x06C8, 0x0445 },
	{ 0x06C9, 0x0438 },
	{ 0x06CA, 0x0439 },
	{ 0x06CB, 0x043A },
	{ 0x06CC, 0x043B },
	{ 0x06CD, 0x043C },
	{ 0x06CE, 0x043D },
	{ 0x06CF, 0x043E },
	{ 0x06D0, 0x043F },
	{ 0x06D1, 0x044F },
	{ 0x06D2, 0x0440 },
	{ 0x06D3, 0x0441 },
	{ 0x06D4, 0x0442 },
	{ 0x06D5, 0x0443 },
	{ 0x06D6, 0x0436 },
	{ 0x06D7, 0x0432 },
	{ 0x06D8, 0x044C },
	{ 0x06D9, 0x044B },
	{ 0x06DA, 0x0437 },
	{ 0x06DB, 0x0448 },
	{ 0x06DC, 0x044D },
	{ 0x06DD, 0x0449 },
	{ 0x06DE, 0x0447 },
	{ 0x06DF, 0x044A },
	{ 0x06E0, 0x042E },
	{ 0x06E1, 0x0410 },
	{ 0x06E2, 0x0411 },
	{ 0x06E3, 0x0426 },
	{ 0x06E4, 0x0414 },
	{ 0x06E5, 0x0415 },
	{ 0x06E6, 0x0424 },
	{ 0x06E7, 0x0413 },
	{ 0x06E8, 0x0425 },
	{ 0x06E9, 0x0418 },
	{ 0x06EA, 0x0419 },
	{ 0x06EB, 0x041A },
	{ 0x06EC, 0x041B },
	{ 0x06ED, 0x041C },
	{ 0x06EE, 0x041D },
	{ 0x06EF, 0x041E },
	{ 0x06F0, 0x041F },
	{ 0x06F1, 0x042F },
	{ 0x06F2, 0x0420 },
	{ 0x06F3, 0x0421 },
	{ 0x06F4, 0x0422 },
	{ 0x06F5, 0x0423 },
	{ 0x06F6, 0x0416 },
	{ 0x06F7, 0x0412 },
	{ 0x06F8, 0x042C },
	{ 0x06F9, 0x042B },
	{ 0x06FA, 0x0417 },
	{ 0x06FB, 0x0428 },
	{ 0x06FC, 0x042D },
	{ 0x06FD, 0x0429 },
	{ 0x06FE, 0x0427 },
	{ 0x06FF, 0x042A },
	{ 0x07A1, 0x0386 },
	{ 0x07A2, 0x0388 },
	{ 0x07A3, 0x0389 },
	{ 0x07A4, 0x038A },
	{ 0x07A5, 0x03AA },
	{ 0x07A7, 0x038C },
	{ 0x07A8, 0x038E },
	{ 0x07A9, 0x03AB },
	{ 0x07AB, 0x038F },
	{ 0x07AE, 0x0385 },
	{ 0x07AF, 0x2015 },
	{ 0x07B1, 0x03AC },
	{ 0x07B2, 0x03AD },
	{ 0x07B3, 0x03AE },
	{ 0x07B4, 0x03AF },
	{ 0x07B5, 0x03CA },
	{ 0x07B6, 0x0390 },
	{ 0x07B7, 0x03CC },
	{ 0x07B8, 0x03CD },
	{ 0x07B9, 0x03CB },
	{ 0x07BA, 0x03B0 },
	{ 0x07BB, 0x03CE },
	{ 0x07C1, 0x0391 },
	{ 0x07C2, 0x0392 },
	{ 0x07C3, 0x0393 },
	{ 0x07C4, 0x0394 },
	{ 0x07C5, 0x0395 },
	{ 0x07C6, 0x0396 },
	{ 0x07C7, 0x0397 },
	{ 0x07C8, 0x0398 },
	{ 0x07C9, 0x0399 },
	{ 0x07CA, 0x039A },
	{ 0x07CB, 0x039B },
	{ 0x07CC, 0x039C },
	{ 0x07CD, 0x039D },
	{ 0x07CE, 0x039E },
	{ 0x07CF, 0x039F },
	{ 0x07D0, 0x03A0 },
	{ 0x07D1, 0x03A1 },
	{ 0x07D2, 0x03A3 },
	{ 0x07D4, 0x03A4 },
	{ 0x07D5, 0x03A5 },
	{ 0x07D6, 0x03A6 },
	{ 0x07D7, 0x03A7 },
	{ 0x07D8, 0x03A8 },
	{ 0x07D9, 0x03A9 },
	{ 0x07E1, 0x03B1 },
	{ 0x07E2, 0x03B2 },
	{ 0x07E3, 0x03B3 },
	{ 0x07E4, 0x03B4 },
	{ 0x07E5, 0x03B5 },
	{ 0x07E6, 0x03B6 },
	{ 0x07E7, 0x03B7 },
	{ 0x07E8, 0x03B8 },
	{ 0x07E9, 0x03B9 },
	{ 0x07EA, 0x03BA },
	{ 0x07EB, 0x03BB },
	{ 0x07EC, 0x03BC },
	{ 0x07ED, 0x03BD },
	{ 0x07EE, 0x03BE },
	{ 0x07EF, 0x03BF },
	{ 0x07F0, 0x03C0 },
	{ 0x07F1, 0x03C1 },
	{ 0x07F2, 0x03C3 },
	{ 0x07F3, 0x03C2 },
	{ 0x07F4, 0x03C4 },
	{ 0x07F5, 0x03C5 },
	{ 0x07F6, 0x03C6 },
	{ 0x07F7, 0x03C7 },
	{ 0x07F8, 0x03C8 },
	{ 0x07F9, 0x03C9 },
	{ 0x08A1, 0x23B7 },
	{ 0x08A2, 0x250C },
	{ 0x08A3, 0x2500 },
	{ 0x08A4, 0x2320 },
	{ 0x08A5, 0x2321 },
	{ 0x08A6, 0x2502 },
	{ 0x08A7, 0x23A1 },
	{ 0x08A8, 0x23A3 },
	{ 0x08A9, 0x23A4 },
	{ 0x08AA, 0x23A6 },
	{ 0x08AB, 0x239B },
	{ 0x08AC, 0x239D },
	{ 0x08AD, 0x239E },
	{ 0x08AE, 0x23A0 },
	{ 0x08AF, 0x23A8 },
	{ 0x08B0, 0x23AC },
	{ 0x08BC, 0x2264 },
	{ 0x08BD, 0x2260 },
	{ 0x08BE, 0x2265 },
	{ 0x08BF, 0x222B },
	{ 0x08C0, 0x2234 },
	{ 0x08C1, 0x221D },
	{ 0x08C2, 0x221E },
	{ 0x08C5, 0x2207 },
	{ 0x08C8, 0x223C },
	{ 0x08C9, 0x2243 },
	{ 0x08CD, 0x21D4 },
	{ 0x08CE, 0x21D2 },
	{ 0x08CF, 0x2261 },
	{ 0x08D6, 0x221A },
	{ 0x08DA, 0x2282 },
	{ 0x08DB, 0x2283 },
	{ 0x08DC, 0x2229 },
	{ 0x08DD, 0x222A },
	{ 0x08DE, 0x2227 },
	{ 0x08DF, 0x2228 },
	{ 0x08EF, 0x2202 },
	{ 0x08F6, 0x0192 },
	{ 0x08FB, 0x2190 },
	{ 0x08FC, 0x2191 },
	{ 0x08FD, 0x2192 },
	{ 0x08FE, 0x2193 },
	{ 0x09E0, 0x25C6 },
	{ 0x09E1, 0x2592 },
	{ 0x09E2, 0x2409 },
	{ 0x09E3, 0x240C },
	{ 0x09E4, 0x240D },
	{ 0x09E5, 0x240A },
	{ 0x09E8, 0x2424 },
	{ 0x09E9, 0x240B },
	{ 0x09EA, 0x2518 },
	{ 0x09EB, 0x2510 },
	{ 0x09EC, 0x250C },
	{ 0x09ED, 0x2514 },
	{ 0x09EE, 0x253C },
	{ 0x09EF, 0x23BA },
	{ 0x09F0, 0x23BB },
	{ 0x09F1, 0x2500 },
	{ 0x09F2, 0x23BC },
	{ 0x09F3, 0x23BD },
	{ 0x09F4, 0x251C },
	{ 0x09F5, 0x2524 },
	{ 0x09F6, 0x2534 },
	{ 0x09F7, 0x252C },
	{ 0x09F8, 0x2502 },
	{ 0x0AA1, 0x2003 },
	{ 0x0AA2, 0x2002 },
	{ 0x0AA3, 0x2004 },
	{ 0x0AA4, 0x2005 },
	{ 0x0AA5, 0x2007 },
	{ 0x0AA6, 0x2008 },
	{ 0x0AA7, 0x2009 },
	{ 0x0AA8, 0x200A },
	{ 0x0AA9, 0x2014 },
	{ 0x0AAA, 0x2013 },
	{ 0x0AAE, 0x2026 },
	{ 0x0AAF, 0x2025 },
	{ 0x0AB0, 0x2153 },
	{ 0x0AB1, 0x2154 },
	{ 0x0AB2, 0x2155 },
	{ 0x0AB3, 0x2156 },
	{ 0x0AB4, 0x2157 },
	{ 0x0AB5, 0x2158 },
	{ 0x0AB6, 0x2159 },
	{ 0x0AB7, 0x215A },
	{ 0x0AB8, 0x2105 },
	{ 0x0ABB, 0x2012 },
	{ 0x0ABC, 0x2329 },
	{ 0x0ABE, 0x232A },
	{ 0x0AC3, 0x215B },
	{ 0x0AC4, 0x215C },
	{ 0x0AC5, 0x215D },
	{ 0x0AC6, 0x215E },
	{ 0x0AC9, 0x2122 },
	{ 0x0ACA, 0x2613 },
	{ 0x0ACC, 0x25C1 },
	{ 0x0ACD, 0x25B7 },
	{ 0x0ACE, 0x25CB },
	{ 0x0ACF, 0x25AF },
	{ 0x0AD0, 0x2018 },
	{ 0x0AD1, 0x2019 },
	{ 0x0AD2, 0x201C },
	{ 0x0AD3, 0x201D },
	{ 0x0AD4, 0x211E },
	{ 0x0AD6, 0x2032 },
	{ 0x0AD7, 0x2033 },
	{ 0x0AD9, 0x271D },
	{ 0x0ADB, 0x25AC },
	{ 0x0ADC, 0x25C0 },
	{ 0x0ADD, 0x25B6 },
	{ 0x0ADE, 0x25CF },
	{ 0x0ADF, 0x25AE },
	{ 0x0AE0, 0x25E6 },
	{ 0x0AE1, 0x25AB },
	{ 0x0AE2, 0x25AD },
	{ 0x0AE3, 0x25B3 },
	{ 0x0AE4, 0x25BD },
	{ 0x0AE5, 0x2606 },
	{ 0x0AE6, 0x2022 },
	{ 0x0AE7, 0x25AA },
	{ 0x0AE8, 0x25B2 },
	{ 0x0AE9, 0x25BC },
	{ 0x0AEA, 0x261C },
	{ 0x0AEB, 0x261E },
	{ 0x0AEC, 0x2663 },
	{ 0x0AED, 0x2666 },
	{ 0x0AEE, 0x2665 },
	{ 0x0AF0, 0x2720 },
	{ 0x0AF1, 0x2020 },
	{ 0x0AF2, 0x2021 },
	{ 0x0AF3, 0x2713 },
	{ 0x0AF4, 0x2717 },
	{ 0x0AF5, 0x266F },
	{ 0x0AF6, 0x266D },
	{ 0x0AF7, 0x2642 },
	{ 0x0AF8, 0x2640 },
	{ 0x0AF9, 0x260E },
	{ 0x0AFA, 0x2315 },
	{ 0x0AFB, 0x2117 },
	{ 0x0AFC, 0x2038 },
	{ 0x0AFD, 0x201A },
	{ 0x0AFE, 0x201E },
	{ 0x0BA3, 0x003C },
	{ 0x0BA6, 0x003E },
	{ 0x0BA8, 0x2228 },
	{ 0x0BA9, 0x2227 },
	{ 0x0BC0, 0x00AF },
	{ 0x0BC2, 0x22A5 },
	{ 0x0BC3, 0x2229 },
	{ 0x0BC4, 0x230A },
	{ 0x0BC6, 0x005F },
	{ 0x0BCA, 0x2218 },
	{ 0x0BCC, 0x2395 },
	{ 0x0BCE, 0x22A4 },
	{ 0x0BCF, 0x25CB },
	{ 0x0BD3, 0x2308 },
	{ 0x0BD6, 0x222A },
	{ 0x0BD8, 0x2283 },
	{ 0x0BDA, 0x2282 },
	{ 0x0BDC, 0x22A2 },
	{ 0x0BFC, 0x22A3 },
	{ 0x0CDF, 0x2017 },
	{ 0x0CE0, 0x05D0 },
	{ 0x0CE1, 0x05D1 },
	{ 0x0CE2, 0x05D2 },
	{ 0x0CE3, 0x05D3 },
	{ 0x0CE4, 0x05D4 },
	{ 0x0CE5, 0x05D5 },
	{ 0x0CE6, 0x05D6 },
	{ 0x0CE7, 0x05D7 },
	{ 0x0CE8, 0x05D8 },
	{ 0x0CE9, 0x05D9 },
	{ 0x0CEA, 0x05DA },
	{ 0x0CEB, 0x05DB },
	{ 0x0CEC, 0x05DC },
	{ 0x0CED, 0x05DD },
	{ 0x0CEE, 0x05DE },
	{ 0x0CEF, 0x05DF },
	{ 0x0CF0, 0x05E0 },
	{ 0x0CF1, 0x05E1 },
	{ 0x0CF2, 0x05E2 },
	{ 0x0CF3, 0x05E3 },
	{ 0x0CF4, 0x05E4 },
	{ 0x0CF5, 0x05E5 },
	{ 0x0CF6, 0x05E6 },
	{ 0x0CF7, 0x05E7 },
	{ 0x0CF8, 0x05E8 },
	{ 0x0CF9, 0x05E9 },
	{ 0x0CFA, 0x05EA },
	{ 0x0DA1, 0x0E01 },
	{ 0x0DA2, 0x0E02 },
	{ 0x0DA3, 0x0E03 },
	{ 0x0DA4, 0x0E04 },
	{ 0x0DA5, 0x0E05 },
	{ 0x0DA6, 0x0E06 },
	{ 0x0DA7, 0x0E07 },
	{ 0x0DA8, 0x0E08 },
	{ 0x0DA9, 0x0E09 },
	{ 0x0DAA, 0x0E0A },
	{ 0x0DAB, 0x0E0B },
	{ 0x0DAC, 0x0E0C },
	{ 0x0DAD, 0x0E0D },
	{ 0x0DAE, 0x0E0E },
	{ 0x0DAF, 0x0E0F },
	{ 0x0DB0, 0x0E10 },
	{ 0x0DB1, 0x0E11 },
	{ 0x0DB2, 0x0E12 },
	{ 0x0DB3, 0x0E13 },
	{ 0x0DB4, 0x0E14 },
	{ 0x0DB5, 0x0E15 },
	{ 0x0DB6, 0x0E16 },
	{ 0x0DB7, 0x0E17 },
	{ 0x0DB8, 0x0E18 },
	{ 0x0DB9, 0x0E19 },
	{ 0x0DBA, 0x0E1A },
	{ 0x0DBB, 0x0E1B },
	{ 0x0DBC, 0x0E1C },
	{ 0x0DBD, 0x0E1D },
	{ 0x0DBE, 0x0E1E },
	{ 0x0DBF, 0x0E1F },
	{ 0x0DC0, 0x0E20 },
	{ 0x0DC1, 0x0E21 },
	{ 0x0DC2, 0x0E22 },
	{ 0x0DC3, 0x0E23 },
	{ 0x0DC4, 0x0E24 },
	{ 0x0DC5, 0x0E25 },
	{ 0x0DC6, 0x0E26 },
	{ 0x0DC7, 0x0E27 },
	{ 0x0DC8, 0x0E28 },
	{ 0x0DC9, 0x0E29 },
	{ 0x0DCA, 0x0E2A },
	{ 0x0DCB, 0x0E2B },
	{ 0x0DCC, 0x0E2C },
	{ 0x0DCD, 0x0E2D },
	{ 0x0DCE, 0x0E2E },
	{ 0x0DCF, 0x0E2F },
	{ 0x0DD0, 0x0E30 },
	{ 0x0DD1, 0x0E31 },
	{ 0x0DD2, 0x0E32 },
	{ 0x0DD3, 0x0E33 },
	{ 0x0DD4, 0x0E34 },
	{ 0x0DD5, 0x0E35 },
	{ 0x0DD6, 0x0E36 },
	{ 0x0DD7, 0x0E37 },
	{ 0x0DD8, 0x0E38 },
	{ 0x0DD9, 0x0E39 },
	{ 0x0DDA, 0x0E3A },
	{ 0x0DDF, 0x0E3F },
	{ 0x0DE0, 0x0E40 },
	{ 0x0DE1, 0x0E41 },
	{ 0x0DE2, 0x0E42 },
	{ 0x0DE3, 0x0E43 },
	{ 0x0DE4, 0x0E44 },
	{ 0x0DE5, 0x0E45 },
	{ 0x0DE6, 0x0E46 },
	{ 0x0DE7, 0x0E47 },
	{ 0x0DE8, 0x0E48 },
	{ 0x0DE9, 0x0E49 },
	{ 0x0DEA, 0x0E4A },
	{ 0x0DEB, 0x0E4B },
	{ 0x0DEC, 0x0E4C },
	{ 0x0DED, 0x0E4D },
	{ 0x0DF0, 0x0E50 },
	{ 0x0DF1, 0x0E51 },
	{ 0x0DF2, 0x0E52 },
	{ 0x0DF3, 0x0E53 },
	{ 0x0DF4, 0x0E54 },
	{ 0x0DF5, 0x0E55 },
	{ 0x0DF6, 0x0E56 },
	{ 0x0DF7, 0x0E57 },
	{ 0x0DF8, 0x0E58 },
	{ 0x0DF9, 0x0E59 },
	{ 0x0EA1, 0x3131 },
	{ 0x0EA2, 0x3132 },
	{ 0x0EA3, 0x3133 },
	{ 0x0EA4, 0x3134 },
	{ 0x0EA5, 0x3135 },
	{ 0x0EA6, 0x3136 },
	{ 0x0EA7, 0x3137 },
	{ 0x0EA8, 0x3138 },
	{ 0x0EA9, 0x3139 },
	{ 0x0EAA, 0x313A },
	{ 0x0EAB, 0x313B },
	{ 0x0EAC, 0x313C },
	{ 0x0EAD, 0x313D },
	{ 0x0EAE, 0x313E },
	{ 0x0EAF, 0x313F },
	{ 0x0EB0, 0x3140 },
	{ 0x0EB1, 0x3141 },
	{ 0x0EB2, 0x3142 },
	{ 0x0EB3, 0x3143 },
	{ 0x0EB4, 0x3144 },
	{ 0x0EB5, 0x3145 },
	{ 0x0EB6, 0x3146 },
	{ 0x0EB7, 0x3147 },
	{ 0x0EB8, 0x3148 },
	{ 0x0EB9, 0x3149 },
	{ 0x0EBA, 0x314A },
	{ 0x0EBB, 0x314B },
	{ 0x0EBC, 0x314C },
	{ 0x0EBD, 0x314D },
	{ 0x0EBE, 0x314E },
	{ 0x0EBF, 0x314F },
	{ 0x0EC0, 0x3150 },
	{ 0x0EC1, 0x3151 },
	{ 0x0EC2, 0x3152 },
	{ 0x0EC3, 0x3153 },
	{ 0x0EC4, 0x3154 },
	{ 0x0EC5, 0x3155 },
	{ 0x0EC6, 0x3156 },
	{ 0x0EC7, 0x3157 },
	{ 0x0EC8, 0x3158 },
	{ 0x0EC9, 0x3159 },
	{ 0x0ECA, 0x315A },
	{ 0x0ECB, 0x315B },
	{ 0x0ECC, 0x315C },
	{ 0x0ECD, 0x315D },
	{ 0x0ECE, 0x315E },
	{ 0x0ECF, 0x315F },
	{ 0x0ED0, 0x3160 },
	{ 0x0ED1, 0x3161 },
	{ 0x0ED2, 0x3162 },
	{ 0x0ED3, 0x3163 },
	{ 0x0ED4, 0x11A8 },
	{ 0x0ED5, 0x11A9 },
	{ 0x0ED6, 0x11AA },
	{ 0x0ED7, 0x11AB },
	{ 0x0ED8, 0x11AC },
	{ 0x0ED9, 0x11AD },
	{ 0x0EDA, 0x11AE },
	{ 0x0EDB, 0x11AF },
	{ 0x0EDC, 0x11B0 },
	{ 0x0EDD, 0x11B1 },
	{ 0x0EDE, 0x11B2 },
	{ 0x0EDF, 0x11B3 },
	{ 0x0EE0, 0x11B4 },
	{ 0x0EE1, 0x11B5 },
	{ 0x0EE2, 0x11B6 },
	{ 0x0EE3, 0x11B7 },
	{ 0x0EE4, 0x11B8 },
	{ 0x0EE5, 0x11B9 },
	{ 0x0EE6, 0x11BA },
	{ 0x0EE7, 0x11BB },
	{ 0x0EE8, 0x11BC },
	{ 0x0EE9, 0x11BD },
	{ 0x0EEA, 0x11BE },
	{ 0x0EEB, 0x11BF },
	{ 0x0EEC, 0x11C0 },
	{ 0x0EED, 0x11C1 },
	{ 0x0EEE, 0x11C2 },
	{ 0x0EEF, 0x316D },
	{ 0x0EF0, 0x3171 },
	{ 0x0EF1, 0x3178 },
	{ 0x0EF2, 0x317F },
	{ 0x0EF3, 0x3181 },
	{ 0x0EF4, 0x3184 },
	{ 0x0EF5, 0x3186 },
	{ 0x0EF6, 0x318D },
	{ 0x0EF7, 0x318E },
	{ 0x0EF8, 0x11EB },
	{ 0x0EF9, 0x11F0 },
	{ 0x0EFA, 0x11F9 },
	{ 0x0EFF, 0x20A9 },
	{ 0x13A4, 0x20AC },
	{ 0x13BC, 0x0152 },
	{ 0x13BD, 0x0153 },
	{ 0x13BE, 0x0178 },
	{ 0x20AC, 0x20AC },
};

unsigned int KeyMappingX11::get_unicode_from_keysym(KeySym p_keysym) {

	/* Latin-1 */
	if (p_keysym>=0x20 && p_keysym<=0x7e)
		return p_keysym;
	if (p_keysym>=0xa0 && p_keysym<=0xff)
		return p_keysym;
	// keypad to latin1 is easy
	if (p_keysym>=0xffaa && p_keysym<=0xffb9)
		return p_keysym-0xff80;

	/* Unicode (may be present)*/

	if((p_keysym&0xff000000)==0x01000000)
		return p_keysym&0x00ffffff;

	int middle,low=0,high=_KEYSYM_MAX-1;
	do {
		middle=(high+low)/2;
		if ( _xkeysym_to_unicode[middle].keysym==p_keysym)
			return _xkeysym_to_unicode[middle].unicode;
		if ( _xkeysym_to_unicode[middle].keysym<=p_keysym )
			low=middle+1;
		else
			high=middle-1;
	} while (high>=low);

	return 0;

}

struct _XTranslateUnicodePairReverse {

	unsigned int unicode;
	KeySym keysym;
};

enum {

	_UNICODE_MAX=750
};

static _XTranslateUnicodePairReverse _unicode_to_xkeysym[_UNICODE_MAX] = {
	{ 0x0ABD, 0x002E },
	{ 0x0BA3, 0x003C },
	{ 0x0BA6, 0x003E },
	{ 0x0BC6, 0x005F },
	{ 0x0BC0, 0x00AF },
	{ 0x03C0, 0x0100 },
	{ 0x03E0, 0x0101 },
	{ 0x01C3, 0x0102 },
	{ 0x01E3, 0x0103 },
	{ 0x01A1, 0x0104 },
	{ 0x01B1, 0x0105 },
	{ 0x01C6, 0x0106 },
	{ 0x01E6, 0x0107 },
	{ 0x02C6, 0x0108 },
	{ 0x02E6, 0x0109 },
	{ 0x02C5, 0x010A },
	{ 0x02E5, 0x010B },
	{ 0x01C8, 0x010C },
	{ 0x01E8, 0x010D },
	{ 0x01CF, 0x010E },
	{ 0x01EF, 0x010F },
	{ 0x01D0, 0x0110 },
	{ 0x01F0, 0x0111 },
	{ 0x03AA, 0x0112 },
	{ 0x03BA, 0x0113 },
	{ 0x03CC, 0x0116 },
	{ 0x03EC, 0x0117 },
	{ 0x01CA, 0x0118 },
	{ 0x01EA, 0x0119 },
	{ 0x01CC, 0x011A },
	{ 0x01EC, 0x011B },
	{ 0x02D8, 0x011C },
	{ 0x02F8, 0x011D },
	{ 0x02AB, 0x011E },
	{ 0x02BB, 0x011F },
	{ 0x02D5, 0x0120 },
	{ 0x02F5, 0x0121 },
	{ 0x03AB, 0x0122 },
	{ 0x03BB, 0x0123 },
	{ 0x02A6, 0x0124 },
	{ 0x02B6, 0x0125 },
	{ 0x02A1, 0x0126 },
	{ 0x02B1, 0x0127 },
	{ 0x03A5, 0x0128 },
	{ 0x03B5, 0x0129 },
	{ 0x03CF, 0x012A },
	{ 0x03EF, 0x012B },
	{ 0x03C7, 0x012E },
	{ 0x03E7, 0x012F },
	{ 0x02A9, 0x0130 },
	{ 0x02B9, 0x0131 },
	{ 0x02AC, 0x0134 },
	{ 0x02BC, 0x0135 },
	{ 0x03D3, 0x0136 },
	{ 0x03F3, 0x0137 },
	{ 0x03A2, 0x0138 },
	{ 0x01C5, 0x0139 },
	{ 0x01E5, 0x013A },
	{ 0x03A6, 0x013B },
	{ 0x03B6, 0x013C },
	{ 0x01A5, 0x013D },
	{ 0x01B5, 0x013E },
	{ 0x01A3, 0x0141 },
	{ 0x01B3, 0x0142 },
	{ 0x01D1, 0x0143 },
	{ 0x01F1, 0x0144 },
	{ 0x03D1, 0x0145 },
	{ 0x03F1, 0x0146 },
	{ 0x01D2, 0x0147 },
	{ 0x01F2, 0x0148 },
	{ 0x03BD, 0x014A },
	{ 0x03BF, 0x014B },
	{ 0x03D2, 0x014C },
	{ 0x03F2, 0x014D },
	{ 0x01D5, 0x0150 },
	{ 0x01F5, 0x0151 },
	{ 0x13BC, 0x0152 },
	{ 0x13BD, 0x0153 },
	{ 0x01C0, 0x0154 },
	{ 0x01E0, 0x0155 },
	{ 0x03A3, 0x0156 },
	{ 0x03B3, 0x0157 },
	{ 0x01D8, 0x0158 },
	{ 0x01F8, 0x0159 },
	{ 0x01A6, 0x015A },
	{ 0x01B6, 0x015B },
	{ 0x02DE, 0x015C },
	{ 0x02FE, 0x015D },
	{ 0x01AA, 0x015E },
	{ 0x01BA, 0x015F },
	{ 0x01A9, 0x0160 },
	{ 0x01B9, 0x0161 },
	{ 0x01DE, 0x0162 },
	{ 0x01FE, 0x0163 },
	{ 0x01AB, 0x0164 },
	{ 0x01BB, 0x0165 },
	{ 0x03AC, 0x0166 },
	{ 0x03BC, 0x0167 },
	{ 0x03DD, 0x0168 },
	{ 0x03FD, 0x0169 },
	{ 0x03DE, 0x016A },
	{ 0x03FE, 0x016B },
	{ 0x02DD, 0x016C },
	{ 0x02FD, 0x016D },
	{ 0x01D9, 0x016E },
	{ 0x01F9, 0x016F },
	{ 0x01DB, 0x0170 },
	{ 0x01FB, 0x0171 },
	{ 0x03D9, 0x0172 },
	{ 0x03F9, 0x0173 },
	{ 0x13BE, 0x0178 },
	{ 0x01AC, 0x0179 },
	{ 0x01BC, 0x017A },
	{ 0x01AF, 0x017B },
	{ 0x01BF, 0x017C },
	{ 0x01AE, 0x017D },
	{ 0x01BE, 0x017E },
	{ 0x08F6, 0x0192 },
	{ 0x01B7, 0x02C7 },
	{ 0x01A2, 0x02D8 },
	{ 0x01FF, 0x02D9 },
	{ 0x01B2, 0x02DB },
	{ 0x01BD, 0x02DD },
	{ 0x07AE, 0x0385 },
	{ 0x07A1, 0x0386 },
	{ 0x07A2, 0x0388 },
	{ 0x07A3, 0x0389 },
	{ 0x07A4, 0x038A },
	{ 0x07A7, 0x038C },
	{ 0x07A8, 0x038E },
	{ 0x07AB, 0x038F },
	{ 0x07B6, 0x0390 },
	{ 0x07C1, 0x0391 },
	{ 0x07C2, 0x0392 },
	{ 0x07C3, 0x0393 },
	{ 0x07C4, 0x0394 },
	{ 0x07C5, 0x0395 },
	{ 0x07C6, 0x0396 },
	{ 0x07C7, 0x0397 },
	{ 0x07C8, 0x0398 },
	{ 0x07C9, 0x0399 },
	{ 0x07CA, 0x039A },
	{ 0x07CB, 0x039B },
	{ 0x07CC, 0x039C },
	{ 0x07CD, 0x039D },
	{ 0x07CE, 0x039E },
	{ 0x07CF, 0x039F },
	{ 0x07D0, 0x03A0 },
	{ 0x07D1, 0x03A1 },
	{ 0x07D2, 0x03A3 },
	{ 0x07D4, 0x03A4 },
	{ 0x07D5, 0x03A5 },
	{ 0x07D6, 0x03A6 },
	{ 0x07D7, 0x03A7 },
	{ 0x07D8, 0x03A8 },
	{ 0x07D9, 0x03A9 },
	{ 0x07A5, 0x03AA },
	{ 0x07A9, 0x03AB },
	{ 0x07B1, 0x03AC },
	{ 0x07B2, 0x03AD },
	{ 0x07B3, 0x03AE },
	{ 0x07B4, 0x03AF },
	{ 0x07BA, 0x03B0 },
	{ 0x07E1, 0x03B1 },
	{ 0x07E2, 0x03B2 },
	{ 0x07E3, 0x03B3 },
	{ 0x07E4, 0x03B4 },
	{ 0x07E5, 0x03B5 },
	{ 0x07E6, 0x03B6 },
	{ 0x07E7, 0x03B7 },
	{ 0x07E8, 0x03B8 },
	{ 0x07E9, 0x03B9 },
	{ 0x07EA, 0x03BA },
	{ 0x07EB, 0x03BB },
	{ 0x07EC, 0x03BC },
	{ 0x07ED, 0x03BD },
	{ 0x07EE, 0x03BE },
	{ 0x07EF, 0x03BF },
	{ 0x07F0, 0x03C0 },
	{ 0x07F1, 0x03C1 },
	{ 0x07F3, 0x03C2 },
	{ 0x07F2, 0x03C3 },
	{ 0x07F4, 0x03C4 },
	{ 0x07F5, 0x03C5 },
	{ 0x07F6, 0x03C6 },
	{ 0x07F7, 0x03C7 },
	{ 0x07F8, 0x03C8 },
	{ 0x07F9, 0x03C9 },
	{ 0x07B5, 0x03CA },
	{ 0x07B9, 0x03CB },
	{ 0x07B7, 0x03CC },
	{ 0x07B8, 0x03CD },
	{ 0x07BB, 0x03CE },
	{ 0x06B3, 0x0401 },
	{ 0x06B1, 0x0402 },
	{ 0x06B2, 0x0403 },
	{ 0x06B4, 0x0404 },
	{ 0x06B5, 0x0405 },
	{ 0x06B6, 0x0406 },
	{ 0x06B7, 0x0407 },
	{ 0x06B8, 0x0408 },
	{ 0x06B9, 0x0409 },
	{ 0x06BA, 0x040A },
	{ 0x06BB, 0x040B },
	{ 0x06BC, 0x040C },
	{ 0x06BE, 0x040E },
	{ 0x06BF, 0x040F },
	{ 0x06E1, 0x0410 },
	{ 0x06E2, 0x0411 },
	{ 0x06F7, 0x0412 },
	{ 0x06E7, 0x0413 },
	{ 0x06E4, 0x0414 },
	{ 0x06E5, 0x0415 },
	{ 0x06F6, 0x0416 },
	{ 0x06FA, 0x0417 },
	{ 0x06E9, 0x0418 },
	{ 0x06EA, 0x0419 },
	{ 0x06EB, 0x041A },
	{ 0x06EC, 0x041B },
	{ 0x06ED, 0x041C },
	{ 0x06EE, 0x041D },
	{ 0x06EF, 0x041E },
	{ 0x06F0, 0x041F },
	{ 0x06F2, 0x0420 },
	{ 0x06F3, 0x0421 },
	{ 0x06F4, 0x0422 },
	{ 0x06F5, 0x0423 },
	{ 0x06E6, 0x0424 },
	{ 0x06E8, 0x0425 },
	{ 0x06E3, 0x0426 },
	{ 0x06FE, 0x0427 },
	{ 0x06FB, 0x0428 },
	{ 0x06FD, 0x0429 },
	{ 0x06FF, 0x042A },
	{ 0x06F9, 0x042B },
	{ 0x06F8, 0x042C },
	{ 0x06FC, 0x042D },
	{ 0x06E0, 0x042E },
	{ 0x06F1, 0x042F },
	{ 0x06C1, 0x0430 },
	{ 0x06C2, 0x0431 },
	{ 0x06D7, 0x0432 },
	{ 0x06C7, 0x0433 },
	{ 0x06C4, 0x0434 },
	{ 0x06C5, 0x0435 },
	{ 0x06D6, 0x0436 },
	{ 0x06DA, 0x0437 },
	{ 0x06C9, 0x0438 },
	{ 0x06CA, 0x0439 },
	{ 0x06CB, 0x043A },
	{ 0x06CC, 0x043B },
	{ 0x06CD, 0x043C },
	{ 0x06CE, 0x043D },
	{ 0x06CF, 0x043E },
	{ 0x06D0, 0x043F },
	{ 0x06D2, 0x0440 },
	{ 0x06D3, 0x0441 },
	{ 0x06D4, 0x0442 },
	{ 0x06D5, 0x0443 },
	{ 0x06C6, 0x0444 },
	{ 0x06C8, 0x0445 },
	{ 0x06C3, 0x0446 },
	{ 0x06DE, 0x0447 },
	{ 0x06DB, 0x0448 },
	{ 0x06DD, 0x0449 },
	{ 0x06DF, 0x044A },
	{ 0x06D9, 0x044B },
	{ 0x06D8, 0x044C },
	{ 0x06DC, 0x044D },
	{ 0x06C0, 0x044E },
	{ 0x06D1, 0x044F },
	{ 0x06A3, 0x0451 },
	{ 0x06A1, 0x0452 },
	{ 0x06A2, 0x0453 },
	{ 0x06A4, 0x0454 },
	{ 0x06A5, 0x0455 },
	{ 0x06A6, 0x0456 },
	{ 0x06A7, 0x0457 },
	{ 0x06A8, 0x0458 },
	{ 0x06A9, 0x0459 },
	{ 0x06AA, 0x045A },
	{ 0x06AB, 0x045B },
	{ 0x06AC, 0x045C },
	{ 0x06AE, 0x045E },
	{ 0x06AF, 0x045F },
	{ 0x0CE0, 0x05D0 },
	{ 0x0CE1, 0x05D1 },
	{ 0x0CE2, 0x05D2 },
	{ 0x0CE3, 0x05D3 },
	{ 0x0CE4, 0x05D4 },
	{ 0x0CE5, 0x05D5 },
	{ 0x0CE6, 0x05D6 },
	{ 0x0CE7, 0x05D7 },
	{ 0x0CE8, 0x05D8 },
	{ 0x0CE9, 0x05D9 },
	{ 0x0CEA, 0x05DA },
	{ 0x0CEB, 0x05DB },
	{ 0x0CEC, 0x05DC },
	{ 0x0CED, 0x05DD },
	{ 0x0CEE, 0x05DE },
	{ 0x0CEF, 0x05DF },
	{ 0x0CF0, 0x05E0 },
	{ 0x0CF1, 0x05E1 },
	{ 0x0CF2, 0x05E2 },
	{ 0x0CF3, 0x05E3 },
	{ 0x0CF4, 0x05E4 },
	{ 0x0CF5, 0x05E5 },
	{ 0x0CF6, 0x05E6 },
	{ 0x0CF7, 0x05E7 },
	{ 0x0CF8, 0x05E8 },
	{ 0x0CF9, 0x05E9 },
	{ 0x0CFA, 0x05EA },
	{ 0x05AC, 0x060C },
	{ 0x05BB, 0x061B },
	{ 0x05BF, 0x061F },
	{ 0x05C1, 0x0621 },
	{ 0x05C2, 0x0622 },
	{ 0x05C3, 0x0623 },
	{ 0x05C4, 0x0624 },
	{ 0x05C5, 0x0625 },
	{ 0x05C6, 0x0626 },
	{ 0x05C7, 0x0627 },
	{ 0x05C8, 0x0628 },
	{ 0x05C9, 0x0629 },
	{ 0x05CA, 0x062A },
	{ 0x05CB, 0x062B },
	{ 0x05CC, 0x062C },
	{ 0x05CD, 0x062D },
	{ 0x05CE, 0x062E },
	{ 0x05CF, 0x062F },
	{ 0x05D0, 0x0630 },
	{ 0x05D1, 0x0631 },
	{ 0x05D2, 0x0632 },
	{ 0x05D3, 0x0633 },
	{ 0x05D4, 0x0634 },
	{ 0x05D5, 0x0635 },
	{ 0x05D6, 0x0636 },
	{ 0x05D7, 0x0637 },
	{ 0x05D8, 0x0638 },
	{ 0x05D9, 0x0639 },
	{ 0x05DA, 0x063A },
	{ 0x05E0, 0x0640 },
	{ 0x05E1, 0x0641 },
	{ 0x05E2, 0x0642 },
	{ 0x05E3, 0x0643 },
	{ 0x05E4, 0x0644 },
	{ 0x05E5, 0x0645 },
	{ 0x05E6, 0x0646 },
	{ 0x05E7, 0x0647 },
	{ 0x05E8, 0x0648 },
	{ 0x05E9, 0x0649 },
	{ 0x05EA, 0x064A },
	{ 0x05EB, 0x064B },
	{ 0x05EC, 0x064C },
	{ 0x05ED, 0x064D },
	{ 0x05EE, 0x064E },
	{ 0x05EF, 0x064F },
	{ 0x05F0, 0x0650 },
	{ 0x05F1, 0x0651 },
	{ 0x05F2, 0x0652 },
	{ 0x0DA1, 0x0E01 },
	{ 0x0DA2, 0x0E02 },
	{ 0x0DA3, 0x0E03 },
	{ 0x0DA4, 0x0E04 },
	{ 0x0DA5, 0x0E05 },
	{ 0x0DA6, 0x0E06 },
	{ 0x0DA7, 0x0E07 },
	{ 0x0DA8, 0x0E08 },
	{ 0x0DA9, 0x0E09 },
	{ 0x0DAA, 0x0E0A },
	{ 0x0DAB, 0x0E0B },
	{ 0x0DAC, 0x0E0C },
	{ 0x0DAD, 0x0E0D },
	{ 0x0DAE, 0x0E0E },
	{ 0x0DAF, 0x0E0F },
	{ 0x0DB0, 0x0E10 },
	{ 0x0DB1, 0x0E11 },
	{ 0x0DB2, 0x0E12 },
	{ 0x0DB3, 0x0E13 },
	{ 0x0DB4, 0x0E14 },
	{ 0x0DB5, 0x0E15 },
	{ 0x0DB6, 0x0E16 },
	{ 0x0DB7, 0x0E17 },
	{ 0x0DB8, 0x0E18 },
	{ 0x0DB9, 0x0E19 },
	{ 0x0DBA, 0x0E1A },
	{ 0x0DBB, 0x0E1B },
	{ 0x0DBC, 0x0E1C },
	{ 0x0DBD, 0x0E1D },
	{ 0x0DBE, 0x0E1E },
	{ 0x0DBF, 0x0E1F },
	{ 0x0DC0, 0x0E20 },
	{ 0x0DC1, 0x0E21 },
	{ 0x0DC2, 0x0E22 },
	{ 0x0DC3, 0x0E23 },
	{ 0x0DC4, 0x0E24 },
	{ 0x0DC5, 0x0E25 },
	{ 0x0DC6, 0x0E26 },
	{ 0x0DC7, 0x0E27 },
	{ 0x0DC8, 0x0E28 },
	{ 0x0DC9, 0x0E29 },
	{ 0x0DCA, 0x0E2A },
	{ 0x0DCB, 0x0E2B },
	{ 0x0DCC, 0x0E2C },
	{ 0x0DCD, 0x0E2D },
	{ 0x0DCE, 0x0E2E },
	{ 0x0DCF, 0x0E2F },
	{ 0x0DD0, 0x0E30 },
	{ 0x0DD1, 0x0E31 },
	{ 0x0DD2, 0x0E32 },
	{ 0x0DD3, 0x0E33 },
	{ 0x0DD4, 0x0E34 },
	{ 0x0DD5, 0x0E35 },
	{ 0x0DD6, 0x0E36 },
	{ 0x0DD7, 0x0E37 },
	{ 0x0DD8, 0x0E38 },
	{ 0x0DD9, 0x0E39 },
	{ 0x0DDA, 0x0E3A },
	{ 0x0DDF, 0x0E3F },
	{ 0x0DE0, 0x0E40 },
	{ 0x0DE1, 0x0E41 },
	{ 0x0DE2, 0x0E42 },
	{ 0x0DE3, 0x0E43 },
	{ 0x0DE4, 0x0E44 },
	{ 0x0DE5, 0x0E45 },
	{ 0x0DE6, 0x0E46 },
	{ 0x0DE7, 0x0E47 },
	{ 0x0DE8, 0x0E48 },
	{ 0x0DE9, 0x0E49 },
	{ 0x0DEA, 0x0E4A },
	{ 0x0DEB, 0x0E4B },
	{ 0x0DEC, 0x0E4C },
	{ 0x0DED, 0x0E4D },
	{ 0x0DF0, 0x0E50 },
	{ 0x0DF1, 0x0E51 },
	{ 0x0DF2, 0x0E52 },
	{ 0x0DF3, 0x0E53 },
	{ 0x0DF4, 0x0E54 },
	{ 0x0DF5, 0x0E55 },
	{ 0x0DF6, 0x0E56 },
	{ 0x0DF7, 0x0E57 },
	{ 0x0DF8, 0x0E58 },
	{ 0x0DF9, 0x0E59 },
	{ 0x0ED4, 0x11A8 },
	{ 0x0ED5, 0x11A9 },
	{ 0x0ED6, 0x11AA },
	{ 0x0ED7, 0x11AB },
	{ 0x0ED8, 0x11AC },
	{ 0x0ED9, 0x11AD },
	{ 0x0EDA, 0x11AE },
	{ 0x0EDB, 0x11AF },
	{ 0x0EDC, 0x11B0 },
	{ 0x0EDD, 0x11B1 },
	{ 0x0EDE, 0x11B2 },
	{ 0x0EDF, 0x11B3 },
	{ 0x0EE0, 0x11B4 },
	{ 0x0EE1, 0x11B5 },
	{ 0x0EE2, 0x11B6 },
	{ 0x0EE3, 0x11B7 },
	{ 0x0EE4, 0x11B8 },
	{ 0x0EE5, 0x11B9 },
	{ 0x0EE6, 0x11BA },
	{ 0x0EE7, 0x11BB },
	{ 0x0EE8, 0x11BC },
	{ 0x0EE9, 0x11BD },
	{ 0x0EEA, 0x11BE },
	{ 0x0EEB, 0x11BF },
	{ 0x0EEC, 0x11C0 },
	{ 0x0EED, 0x11C1 },
	{ 0x0EEE, 0x11C2 },
	{ 0x0EF8, 0x11EB },
	{ 0x0EFA, 0x11F9 },
	{ 0x0AA2, 0x2002 },
	{ 0x0AA1, 0x2003 },
	{ 0x0AA3, 0x2004 },
	{ 0x0AA4, 0x2005 },
	{ 0x0AA5, 0x2007 },
	{ 0x0AA6, 0x2008 },
	{ 0x0AA7, 0x2009 },
	{ 0x0AA8, 0x200A },
	{ 0x0ABB, 0x2012 },
	{ 0x0AAA, 0x2013 },
	{ 0x0AA9, 0x2014 },
	{ 0x07AF, 0x2015 },
	{ 0x0CDF, 0x2017 },
	{ 0x0AD0, 0x2018 },
	{ 0x0AD1, 0x2019 },
	{ 0x0AFD, 0x201A },
	{ 0x0AD2, 0x201C },
	{ 0x0AD3, 0x201D },
	{ 0x0AFE, 0x201E },
	{ 0x0AF1, 0x2020 },
	{ 0x0AF2, 0x2021 },
	{ 0x0AE6, 0x2022 },
	{ 0x0AAE, 0x2026 },
	{ 0x0AD6, 0x2032 },
	{ 0x0AD7, 0x2033 },
	{ 0x0AFC, 0x2038 },
	{ 0x047E, 0x203E },
	{ 0x20A0, 0x20A0 },
	{ 0x20A1, 0x20A1 },
	{ 0x20A2, 0x20A2 },
	{ 0x20A3, 0x20A3 },
	{ 0x20A4, 0x20A4 },
	{ 0x20A5, 0x20A5 },
	{ 0x20A6, 0x20A6 },
	{ 0x20A7, 0x20A7 },
	{ 0x20A8, 0x20A8 },
	{ 0x0EFF, 0x20A9 },
	{ 0x20A9, 0x20A9 },
	{ 0x20AA, 0x20AA },
	{ 0x20AB, 0x20AB },
	{ 0x20AC, 0x20AC },
	{ 0x0AB8, 0x2105 },
	{ 0x06B0, 0x2116 },
	{ 0x0AFB, 0x2117 },
	{ 0x0AD4, 0x211E },
	{ 0x0AC9, 0x2122 },
	{ 0x0AB0, 0x2153 },
	{ 0x0AB1, 0x2154 },
	{ 0x0AB2, 0x2155 },
	{ 0x0AB3, 0x2156 },
	{ 0x0AB4, 0x2157 },
	{ 0x0AB5, 0x2158 },
	{ 0x0AB6, 0x2159 },
	{ 0x0AB7, 0x215A },
	{ 0x0AC3, 0x215B },
	{ 0x0AC4, 0x215C },
	{ 0x0AC5, 0x215D },
	{ 0x0AC6, 0x215E },
	{ 0x08FB, 0x2190 },
	{ 0x08FC, 0x2191 },
	{ 0x08FD, 0x2192 },
	{ 0x08FE, 0x2193 },
	{ 0x08CE, 0x21D2 },
	{ 0x08CD, 0x21D4 },
	{ 0x08EF, 0x2202 },
	{ 0x08C5, 0x2207 },
	{ 0x0BCA, 0x2218 },
	{ 0x08D6, 0x221A },
	{ 0x08C1, 0x221D },
	{ 0x08C2, 0x221E },
	{ 0x08DE, 0x2227 },
	{ 0x0BA9, 0x2227 },
	{ 0x08DF, 0x2228 },
	{ 0x0BA8, 0x2228 },
	{ 0x08DC, 0x2229 },
	{ 0x0BC3, 0x2229 },
	{ 0x08DD, 0x222A },
	{ 0x0BD6, 0x222A },
	{ 0x08BF, 0x222B },
	{ 0x08C0, 0x2234 },
	{ 0x08C8, 0x2245 },
	{ 0x08BD, 0x2260 },
	{ 0x08CF, 0x2261 },
	{ 0x08BC, 0x2264 },
	{ 0x08BE, 0x2265 },
	{ 0x08DA, 0x2282 },
	{ 0x0BDA, 0x2282 },
	{ 0x08DB, 0x2283 },
	{ 0x0BD8, 0x2283 },
	{ 0x0BFC, 0x22A2 },
	{ 0x0BDC, 0x22A3 },
	{ 0x0BC2, 0x22A4 },
	{ 0x0BCE, 0x22A5 },
	{ 0x0BD3, 0x2308 },
	{ 0x0BC4, 0x230A },
	{ 0x0AFA, 0x2315 },
	{ 0x08A4, 0x2320 },
	{ 0x08A5, 0x2321 },
	{ 0x0ABC, 0x2329 },
	{ 0x0ABE, 0x232A },
	{ 0x0BCC, 0x2395 },
	{ 0x09E2, 0x2409 },
	{ 0x09E5, 0x240A },
	{ 0x09E9, 0x240B },
	{ 0x09E3, 0x240C },
	{ 0x09E4, 0x240D },
	{ 0x09DF, 0x2422 },
	{ 0x09E8, 0x2424 },
	{ 0x09F1, 0x2500 },
	{ 0x08A6, 0x2502 },
	{ 0x09F8, 0x2502 },
	{ 0x09EC, 0x250C },
	{ 0x09EB, 0x2510 },
	{ 0x09ED, 0x2514 },
	{ 0x09EA, 0x2518 },
	{ 0x09F4, 0x251C },
	{ 0x09F5, 0x2524 },
	{ 0x09F7, 0x252C },
	{ 0x09F6, 0x2534 },
	{ 0x09EE, 0x253C },
	{ 0x09E1, 0x2592 },
	{ 0x0ADF, 0x25A0 },
	{ 0x0ACF, 0x25A1 },
	{ 0x0AE7, 0x25AA },
	{ 0x0AE1, 0x25AB },
	{ 0x0ADB, 0x25AC },
	{ 0x0AE2, 0x25AD },
	{ 0x0AE8, 0x25B2 },
	{ 0x0AE3, 0x25B3 },
	{ 0x0ADD, 0x25B6 },
	{ 0x0ACD, 0x25B7 },
	{ 0x0AE9, 0x25BC },
	{ 0x0AE4, 0x25BD },
	{ 0x0ADC, 0x25C0 },
	{ 0x0ACC, 0x25C1 },
	{ 0x09E0, 0x25C6 },
	{ 0x0ACE, 0x25CB },
	{ 0x0BCF, 0x25CB },
	{ 0x0ADE, 0x25CF },
	{ 0x0AE0, 0x25E6 },
	{ 0x0AE5, 0x2606 },
	{ 0x0AF9, 0x260E },
	{ 0x0ACA, 0x2613 },
	{ 0x0AEA, 0x261C },
	{ 0x0AEB, 0x261E },
	{ 0x0AF8, 0x2640 },
	{ 0x0AF7, 0x2642 },
	{ 0x0AEC, 0x2663 },
	{ 0x0AEE, 0x2665 },
	{ 0x0AED, 0x2666 },
	{ 0x0AF6, 0x266D },
	{ 0x0AF5, 0x266F },
	{ 0x0AF3, 0x2713 },
	{ 0x0AF4, 0x2717 },
	{ 0x0AD9, 0x271D },
	{ 0x0AF0, 0x2720 },
	{ 0x04A4, 0x3001 },
	{ 0x04A1, 0x3002 },
	{ 0x04A2, 0x300C },
	{ 0x04A3, 0x300D },
	{ 0x04DE, 0x309B },
	{ 0x04DF, 0x309C },
	{ 0x04A7, 0x30A1 },
	{ 0x04B1, 0x30A2 },
	{ 0x04A8, 0x30A3 },
	{ 0x04B2, 0x30A4 },
	{ 0x04A9, 0x30A5 },
	{ 0x04B3, 0x30A6 },
	{ 0x04AA, 0x30A7 },
	{ 0x04B4, 0x30A8 },
	{ 0x04AB, 0x30A9 },
	{ 0x04B5, 0x30AA },
	{ 0x04B6, 0x30AB },
	{ 0x04B7, 0x30AD },
	{ 0x04B8, 0x30AF },
	{ 0x04B9, 0x30B1 },
	{ 0x04BA, 0x30B3 },
	{ 0x04BB, 0x30B5 },
	{ 0x04BC, 0x30B7 },
	{ 0x04BD, 0x30B9 },
	{ 0x04BE, 0x30BB },
	{ 0x04BF, 0x30BD },
	{ 0x04C0, 0x30BF },
	{ 0x04C1, 0x30C1 },
	{ 0x04AF, 0x30C3 },
	{ 0x04C2, 0x30C4 },
	{ 0x04C3, 0x30C6 },
	{ 0x04C4, 0x30C8 },
	{ 0x04C5, 0x30CA },
	{ 0x04C6, 0x30CB },
	{ 0x04C7, 0x30CC },
	{ 0x04C8, 0x30CD },
	{ 0x04C9, 0x30CE },
	{ 0x04CA, 0x30CF },
	{ 0x04CB, 0x30D2 },
	{ 0x04CC, 0x30D5 },
	{ 0x04CD, 0x30D8 },
	{ 0x04CE, 0x30DB },
	{ 0x04CF, 0x30DE },
	{ 0x04D0, 0x30DF },
	{ 0x04D1, 0x30E0 },
	{ 0x04D2, 0x30E1 },
	{ 0x04D3, 0x30E2 },
	{ 0x04AC, 0x30E3 },
	{ 0x04D4, 0x30E4 },
	{ 0x04AD, 0x30E5 },
	{ 0x04D5, 0x30E6 },
	{ 0x04AE, 0x30E7 },
	{ 0x04D6, 0x30E8 },
	{ 0x04D7, 0x30E9 },
	{ 0x04D8, 0x30EA },
	{ 0x04D9, 0x30EB },
	{ 0x04DA, 0x30EC },
	{ 0x04DB, 0x30ED },
	{ 0x04DC, 0x30EF },
	{ 0x04A6, 0x30F2 },
	{ 0x04DD, 0x30F3 },
	{ 0x04A5, 0x30FB },
	{ 0x04B0, 0x30FC },
	{ 0x0EA1, 0x3131 },
	{ 0x0EA2, 0x3132 },
	{ 0x0EA3, 0x3133 },
	{ 0x0EA4, 0x3134 },
	{ 0x0EA5, 0x3135 },
	{ 0x0EA6, 0x3136 },
	{ 0x0EA7, 0x3137 },
	{ 0x0EA8, 0x3138 },
	{ 0x0EA9, 0x3139 },
	{ 0x0EAA, 0x313A },
	{ 0x0EAB, 0x313B },
	{ 0x0EAC, 0x313C },
	{ 0x0EAD, 0x313D },
	{ 0x0EAE, 0x313E },
	{ 0x0EAF, 0x313F },
	{ 0x0EB0, 0x3140 },
	{ 0x0EB1, 0x3141 },
	{ 0x0EB2, 0x3142 },
	{ 0x0EB3, 0x3143 },
	{ 0x0EB4, 0x3144 },
	{ 0x0EB5, 0x3145 },
	{ 0x0EB6, 0x3146 },
	{ 0x0EB7, 0x3147 },
	{ 0x0EB8, 0x3148 },
	{ 0x0EB9, 0x3149 },
	{ 0x0EBA, 0x314A },
	{ 0x0EBB, 0x314B },
	{ 0x0EBC, 0x314C },
	{ 0x0EBD, 0x314D },
	{ 0x0EBE, 0x314E },
	{ 0x0EBF, 0x314F },
	{ 0x0EC0, 0x3150 },
	{ 0x0EC1, 0x3151 },
	{ 0x0EC2, 0x3152 },
	{ 0x0EC3, 0x3153 },
	{ 0x0EC4, 0x3154 },
	{ 0x0EC5, 0x3155 },
	{ 0x0EC6, 0x3156 },
	{ 0x0EC7, 0x3157 },
	{ 0x0EC8, 0x3158 },
	{ 0x0EC9, 0x3159 },
	{ 0x0ECA, 0x315A },
	{ 0x0ECB, 0x315B },
	{ 0x0ECC, 0x315C },
	{ 0x0ECD, 0x315D },
	{ 0x0ECE, 0x315E },
	{ 0x0ECF, 0x315F },
	{ 0x0ED0, 0x3160 },
	{ 0x0ED1, 0x3161 },
	{ 0x0ED2, 0x3162 },
	{ 0x0ED3, 0x3163 },
	{ 0x0EEF, 0x316D },
	{ 0x0EF0, 0x3171 },
	{ 0x0EF1, 0x3178 },
	{ 0x0EF2, 0x317F },
	{ 0x0EF4, 0x3184 },
	{ 0x0EF5, 0x3186 },
	{ 0x0EF6, 0x318D },
	{ 0x0EF7, 0x318E }
};

KeySym KeyMappingX11::get_keysym_from_unicode(unsigned int p_unicode) {

	/* Latin 1 */

	if (p_unicode>=0x20 && p_unicode<=0x7e)
		return p_unicode;

	if (p_unicode>=0xa0 && p_unicode<=0xff)
		return p_unicode;

	int middle,low=0,high=_UNICODE_MAX-1;
	do {
		middle=(high+low)/2;
		if ( _unicode_to_xkeysym[middle].keysym==p_unicode)
			return _unicode_to_xkeysym[middle].keysym;
		if ( _unicode_to_xkeysym[middle].keysym<=p_unicode )
			low=middle+1;
		else
			high=middle-1;
	} while (high>=low);

	// if not found, let's hope X understands it as unicode
	return p_unicode|0x01000000;
}
