package krasilnikov.alexey.cryptopass.v14;

import android.annotation.TargetApi;
import android.content.AsyncTaskLoader;
import android.content.Context;
import android.database.Cursor;
import android.os.Build;

import javax.inject.Inject;

import krasilnikov.alexey.cryptopass.Data;
import krasilnikov.alexey.cryptopass.data.BookmarksStorage;

@TargetApi(Build.VERSION_CODES.ICE_CREAM_SANDWICH)
class BookmarkCursorLoader extends AsyncTaskLoader<Cursor> {
    private final ForceLoadContentObserver mObserver = new ForceLoadContentObserver();
    private final BookmarksStorage mStorage;

    @Inject
    public BookmarkCursorLoader(Context context, BookmarksStorage storage) {
        super(context);
        mStorage = storage;
    }

    private Cursor mCursor;

    /* Runs on a worker thread */
    @Override
    public Cursor loadInBackground() {
        Cursor cursor = mStorage.queryBookmarks(Data.BOOKMARKS_PROJECTION);
        if (cursor != null) {
            try {
                // Ensure the cursor window is filled.
                cursor.getCount();
            } catch (RuntimeException ex) {
                cursor.close();
                throw ex;
            }
        }
        return cursor;
    }

    /* Runs on the UI thread */
    @Override
    public void deliverResult(Cursor cursor) {
        if (isReset()) {
            // An async query came in while the loader is stopped
            if (cursor != null) {
                cursor.close();
            }
            return;
        }
        Cursor oldCursor = mCursor;
        mCursor = cursor;

        if (isStarted()) {
            super.deliverResult(cursor);
        }

        if (oldCursor != null && oldCursor != cursor && !oldCursor.isClosed()) {
            oldCursor.close();
        }
    }

    /**
     * Starts an asynchronous load of the contacts list data. When the result is ready the callbacks
     * will be called on the UI thread. If a previous load has been completed and is still valid
     * the result may be passed to the callbacks immediately.
     *
     * Must be called from the UI thread
     */
    @Override
    protected void onStartLoading() {
        if (mCursor != null) {
            deliverResult(mCursor);
        }
        if (takeContentChanged() || mCursor == null) {
            forceLoad();
        }
        mStorage.registerObserver(mObserver);
    }

    /**
     * Must be called from the UI thread
     */
    @Override
    protected void onStopLoading() {
        mStorage.unregisterObserver(mObserver);
        // Attempt to cancel the current load task if possible.
        cancelLoad();
    }

    @Override
    public void onCanceled(Cursor cursor) {
        if (cursor != null && !cursor.isClosed()) {
            cursor.close();
        }
    }

    @Override
    protected void onReset() {
        super.onReset();

        if (mCursor != null && !mCursor.isClosed()) {
            mCursor.close();
        }
        mCursor = null;
    }

}
