/*
 * AndFHEM - Open Source Android application to control a FHEM home automation
 * server.
 *
 * Copyright (c) 2012, Matthias Klass or third-party contributors as
 * indicated by the @author tags or express copyright attribution
 * statements applied by the authors.  All third-party contributions are
 * distributed under license by Red Hat Inc.
 *
 * This copyrighted material is made available to anyone wishing to use, modify,
 * copy, or redistribute it subject to the terms and conditions of the GNU GENERAL PUBLICLICENSE, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU GENERAL PUBLIC LICENSE
 * for more details.
 *
 * You should have received a copy of the GNU GENERAL PUBLIC LICENSE
 * along with this distribution; if not, write to:
 *   Free Software Foundation, Inc.
 *   51 Franklin Street, Fifth Floor
 */

package li.klass.fhem.billing.playstore;


import android.app.Activity;
import android.app.PendingIntent;
import android.app.PendingIntent.CanceledException;
import android.content.Intent;
import android.content.IntentSender;
import android.os.Handler;
import android.util.Log;
import li.klass.fhem.util.Reject;

import java.lang.reflect.Method;

import static li.klass.fhem.billing.BillingConstants.PurchaseState;
import static li.klass.fhem.billing.BillingConstants.ResponseCode;

/**
 * An interface for observing changes related to purchases. The main application
 * extends this class and registers an instance of that derived class with
 * {@link PlayStoreResponseHandler}. The main application implements the callbacks
 * {@link #onBillingSupported(boolean)} and
 * {@link #onPurchaseStateChange}. These methods
 * are used to update the UI.
 */
public abstract class PlayStorePurchaseObserver {
    private static final String TAG = PlayStorePurchaseObserver.class.getName();
    private final Handler handler;
    private Method startIntentSender;
    private Object[] startIntentSenderArgs = new Object[5];
    private static final Class[] START_INTENT_SENDER_SIG = new Class[] {
            IntentSender.class, Intent.class, int.class, int.class, int.class
    };
    protected Activity activity;

    public PlayStorePurchaseObserver(Handler handler) {
        this.handler = handler;
    }

    /**
     * This is the callback that is invoked when Android Market responds to the
     * {@link PlayStoreBillingService#checkBillingSupported()} request.
     * @param supported true if in-app billing is supported.
     */
    public abstract void onBillingSupported(boolean supported);

    /**
     * This is the callback that is invoked when an item is purchased,
     * refunded, or canceled.  It is the callback invoked in response to
     * calling {@link PlayStoreBillingService#requestPurchase(String, String)}.  It may also
     * be invoked asynchronously when a purchase is made on another device
     * (if the purchase was for a Market-managed item), or if the purchase
     * was refunded, or the charge was canceled.  This handles the UI
     * update.  The database update is handled in
     * {@link PlayStoreResponseHandler#purchaseResponse}.
     * @param purchaseState the purchase state of the item
     * @param itemId a string identifying the item (the "SKU")
     * @param quantity the current quantity of this item after the purchase
     * @param purchaseTime the time the product was purchased, in
     * milliseconds since the epoch (Jan 1, 1970)
     */
    public abstract void onPurchaseStateChange(PurchaseState purchaseState,
                                               String itemId, int quantity, long purchaseTime, String developerPayload);

    /**
     * This is called when we receive a response code from Market for a
     * RequestPurchase request that we made.  This is NOT used for any
     * purchase state changes.  All purchase state changes are received in
     *
     * {@link #onPurchaseStateChange}
     * This is used for reporting various errors, or if the user backed out
     * and didn't purchase the item.  The possible response codes are:
     *   RESULT_OK means that the order was sent successfully to the server.
     *       The onPurchaseStateChange() will be invoked later (with a
     *       purchase state of PURCHASED or CANCELED) when the order is
     *       charged or canceled.  This response code can also happen if an
     *       order for a Market-managed item was already sent to the server.
     *   RESULT_USER_CANCELED means that the user didn't buy the item.
     *   RESULT_SERVICE_UNAVAILABLE means that we couldn't connect to the
     *       Android Market server (for example if the data connection is down).
     *   RESULT_BILLING_UNAVAILABLE means that in-app billing is not
     *       supported yet.
     *   RESULT_ITEM_UNAVAILABLE means that the item this app offered for
     *       sale does not exist (or is not published) in the server-side
     *       catalog.
     *   RESULT_ERROR is used for any other errors (such as a server error).
     */
    public abstract void onRequestPurchaseResponse(PlayStoreBillingService.RequestPurchase request,
                                                   ResponseCode responseCode);

    /**
     * This is called when we receive a response code from Android Market for a
     * RestoreTransactions request that we made.  A response code of
     * RESULT_OK means that the request was successfully sent to the server.
     */
    public abstract void onRestoreTransactionsResponse(PlayStoreBillingService.RestoreTransactions request,
                                                       ResponseCode responseCode);

    public void bindActivity(Activity activity) {
        Reject.ifNull(activity);

        this.activity = activity;
        initCompatibilityLayer();
    }

    public void initCompatibilityLayer() {
        try {
            startIntentSender = activity.getClass().getMethod("startIntentSender",
                    START_INTENT_SENDER_SIG);
        } catch (SecurityException e) {
            startIntentSender = null;
        } catch (NoSuchMethodException e) {
            startIntentSender = null;
        }
    }

    void startBuyPageActivity(PendingIntent pendingIntent, Intent intent) {
        if (activity == null) {
            throw new IllegalStateException("cannot start buy page activity without having an activity attached");
        }

        if (startIntentSender != null) {
            // This is on Android 2.0 and beyond.  The in-app buy page activity
            // must be on the activity stack of the application.
            try {
                // This implements the method call:
                // activity.startIntentSender(pendingIntent.getIntentSender(),
                //     intent, 0, 0, 0);
                startIntentSenderArgs[0] = pendingIntent.getIntentSender();
                startIntentSenderArgs[1] = intent;
                startIntentSenderArgs[2] = 0;
                startIntentSenderArgs[3] = 0;
                startIntentSenderArgs[4] = 0;
                startIntentSender.invoke(activity, startIntentSenderArgs);
            } catch (Exception e) {
                Log.e(TAG, "error starting activity", e);
            }
        } else {
            // This is on Android version 1.6. The in-app buy page activity must be on its
            // own separate activity stack instead of on the activity stack of
            // the application.
            try {
                pendingIntent.send(activity, 0 /* code */, intent);
            } catch (CanceledException e) {
                Log.e(TAG, "error starting activity", e);
            }
        }
    }

    /**
     * Updates the UI after the database has been updated.  This method runs
     * in a background thread so it has to post a Runnable to run on the UI
     * thread.
     * @param purchaseState the purchase state of the item
     * @param itemId a string identifying the item
     * @param quantity the quantity of items in this purchase
     */
    void postPurchaseStateChange(final PurchaseState purchaseState, final String itemId,
                                 final int quantity, final long purchaseTime, final String developerPayload) {
        Log.i(TAG, "purchase state change (itemId: " + itemId + ", target state: " + purchaseState.name());
        handler.post(new Runnable() {
            public void run() {
                onPurchaseStateChange(purchaseState, itemId, quantity, purchaseTime, developerPayload);
            }
        });
    }
}