/**
 *  ReGalAndroid, a gallery client for Android, supporting G2, G3, etc...
 *  URLs: https://github.com/anthonydahanne/ReGalAndroid , http://blog.dahanne.net
 *  Copyright (c) 2010 Anthony Dahanne
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package net.dahanne.android.regalandroid.utils;

import net.dahanne.gallery.commons.model.Album;
import net.dahanne.gallery.commons.utils.Serialization;
import net.dahanne.gallery.commons.utils.SerializationException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.SQLException;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;

public class DBHelper {

	public static final String DB_NAME = "regalandroid_context";
	public static final String DB_TABLE = "regalandroid_context_table";
	public static final int DB_VERSION = 3;

	private static final String[] COLS = new String[] { "_id",
			"current_position", "current_album" };

	private SQLiteDatabase db;
	private final DBOpenHelper dbOpenHelper;
	private final Logger logger = LoggerFactory.getLogger(DBHelper.class);

	//
	// inner classes
	//

	public static class ReGalAndroidContext {

		public long id;
		public int currentPosition;
		public Album currentAlbum;

		public ReGalAndroidContext() {
		}

		public ReGalAndroidContext(final long id, final int currentPosition,
				final Album currentAlbum) {
			this.id = id;
			this.currentPosition = currentPosition;
			this.currentAlbum = currentAlbum;
		}

		@Override
		public String toString() {
			return id + " " + currentPosition + " " + currentAlbum;
		}
	}

	private static class DBOpenHelper extends SQLiteOpenHelper {

		private static final String DB_CREATE = "CREATE TABLE "
				+ DBHelper.DB_TABLE
				+ " (_id INTEGER PRIMARY KEY, current_position INTEGER, current_album BLOB );";

		public DBOpenHelper(final Context context) {
			super(context, DBHelper.DB_NAME, null, DBHelper.DB_VERSION);
		}

		@Override
		public void onCreate(final SQLiteDatabase db) {
			try {
				db.execSQL(DBOpenHelper.DB_CREATE);
			} catch (SQLException e) {
				// Log.e(Constants.LOGTAG, DBHelper.CLASSNAME, e);
			}
		}

		@Override
		public void onOpen(final SQLiteDatabase db) {
			super.onOpen(db);
		}

		@Override
		public void onUpgrade(final SQLiteDatabase db, final int oldVersion,
				final int newVersion) {
			db.execSQL("DROP TABLE IF EXISTS " + DBHelper.DB_TABLE);
			onCreate(db);
		}
	}

	//
	// end inner classes
	//

	public DBHelper(final Context context) {
		dbOpenHelper = new DBOpenHelper(context);
		establishDb();
	}

	private void establishDb() {
		if (db == null) {
			db = dbOpenHelper.getWritableDatabase();
		}
	}

	public void cleanup() {
		if (db != null) {
			db.close();
			db = null;
		}
	}

	public void insert(final ReGalAndroidContext regalAndroidContext) {
		ContentValues values = new ContentValues();
		values.put("current_position", regalAndroidContext.currentPosition);
		if(regalAndroidContext.currentAlbum!=null){
			try {
				values.put("current_album", Serialization.serializeAlbum(regalAndroidContext.currentAlbum));
			} catch (SerializationException e) {
				logger.debug(e.getMessage());
			}
		}

		db.insert(DBHelper.DB_TABLE, null, values);

		cleanup();
	}

	public void update(final ReGalAndroidContext regalAndroidContext) {
		ContentValues values = new ContentValues();
		try {
		values.put("current_position", regalAndroidContext.currentPosition);
			values.put("current_album", Serialization.serializeAlbum(regalAndroidContext.currentAlbum));
		} catch (SerializationException e) {
			logger.debug(e.getMessage());
		}
		db.update(DBHelper.DB_TABLE, values, "_id=" + regalAndroidContext.id, null);
		cleanup();
	}

	public void delete(final long id) {
		db.delete(DBHelper.DB_TABLE, "_id=" + id, null);
	}

	public void deleteAll() {
		db.delete(DBHelper.DB_TABLE, null, null);
	}


	public ReGalAndroidContext getLast() {
		Cursor c = null;
		ReGalAndroidContext regalAndroidContext = null;
		try {
			c = db.query(true, DBHelper.DB_TABLE, DBHelper.COLS, null, null,
					null, null, "_id DESC", null);
			if (c.getCount() > 0) {
				c.moveToFirst();
				regalAndroidContext = new ReGalAndroidContext();
				regalAndroidContext.id = c.getLong(0);
				regalAndroidContext.currentPosition = c.getInt(1);
				byte[] blob = c
						.getBlob(2);
				if(blob!=null){
					regalAndroidContext.currentAlbum = Serialization.unserializeAlbum(blob);
				}

			}
		} catch (SQLException e) {
			logger.debug(e.getMessage());
		} catch (SerializationException e) {
			logger.debug(e.getMessage());
		} finally {
			c.close();
			cleanup();
		}
		return regalAndroidContext;
	}

}