package net.sourceforge.wifiremoteplay;

import android.content.SharedPreferences;
import android.net.Uri;
import android.os.AsyncTask;
import android.os.Build;
import android.os.Bundle;
import android.os.SystemClock;
import android.preference.PreferenceManager;
import android.annotation.SuppressLint;
import android.annotation.TargetApi;
import android.app.Activity;
import android.app.AlertDialog;
import android.content.Intent;
import android.text.Html;
import android.util.Log;
import android.util.Xml;
import android.view.KeyEvent;
import android.view.Menu;
import android.view.MenuItem;
import android.view.View;
import android.view.WindowManager.LayoutParams;
import android.widget.SeekBar;
import android.widget.Toast;
import android.widget.SeekBar.OnSeekBarChangeListener;
import android.widget.TextView;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.net.Authenticator;
import java.net.PasswordAuthentication;
import java.net.URL;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Locale;
import java.util.Timer;
import java.util.TimerTask;

import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;

class MyDebug {
	static final boolean LOG = false;
}

@SuppressLint("SimpleDateFormat")
public class MainActivity extends Activity {
	private static final String TAG = "MainActivity";

	private boolean app_is_active = true;
	private Timer requestStatusTimer = null;
	private Timer refreshTimeLabelTimer = null;
	
	private boolean waiting_for_status = false;

	private boolean is_paused = true;
	private boolean have_time = false;
	private Calendar time = GregorianCalendar.getInstance(); // creates a new calendar instance
	private boolean have_saved_last_time = false;
	private Calendar saved_last_time = GregorianCalendar.getInstance(); // creates a new calendar instance
	private long time_last_updated = SystemClock.elapsedRealtime();

    private boolean is_muted = false; // for VLC
    private int saved_volume = 0; // for VLC unmuting

	@Override
	protected void onCreate(Bundle savedInstanceState) {
		if( MyDebug.LOG )
			Log.d(TAG, "onCreate");
		super.onCreate(savedInstanceState);
		setContentView(R.layout.activity_main);
		PreferenceManager.setDefaultValues(this, R.xml.preferences, false);

        final SeekBar progress = (SeekBar)findViewById(R.id.progress);
        progress.setOnSeekBarChangeListener(new OnSeekBarChangeListener() {
	        @Override       
	        public void onStopTrackingTouch(SeekBar seekBar) {      
	        	int value = seekBar.getProgress();
	    		if( MyDebug.LOG )
	    			Log.d(TAG, "onStopTrackingTouch: " + value);
		        moveToTime(value);
	        }

	        @Override       
	        public void onStartTrackingTouch(SeekBar seekBar) {     
	        }

	        @Override       
	        public void onProgressChanged(SeekBar seekBar, int progress, boolean fromUser) {     
	        }
	    });       
        if( !MainApp.IPDefined(this) ) {
    		if( MyDebug.LOG )
    			Log.d(TAG, "startup - ip address not defined, so open preferences");
	        Intent intent = new Intent(this, MyPreferenceActivity.class);
	        this.startActivity(intent);
		}
	}

	private void cancelTimers() {
		if( requestStatusTimer != null ) {
			requestStatusTimer.cancel();
			requestStatusTimer.purge();
			requestStatusTimer = null;
		}
		if( refreshTimeLabelTimer != null ) {
			refreshTimeLabelTimer.cancel();
			refreshTimeLabelTimer.purge();
			refreshTimeLabelTimer = null;
		}
	}
	
	private void setupTimers() {
		cancelTimers(); // cancel any already setup, just in case!

		waiting_for_status = false; // reset, so we force being able to send a new request (needed if, e.g., settings have changed)
		class RequestStatusTimerTask extends TimerTask {
			public void run() {
				/*if( MyDebug.LOG )
					Log.d(TAG, "RequestStatusTimerTask.run");*/
				MainActivity.this.requestStatus();
			}
		}
		requestStatusTimer = new Timer();
		requestStatusTimer.scheduleAtFixedRate(new RequestStatusTimerTask(), 0, 5000);
		//requestStatus();

		class RefreshTimeLabelTimerTask extends TimerTask {
			public void run() {
				/*if( MyDebug.LOG )
					Log.d(TAG, "RefreshTimeLabelTimerTask.run");*/
				MainActivity.this.refreshTimeLabel();
			}
		}
		refreshTimeLabelTimer = new Timer();
		refreshTimeLabelTimer.scheduleAtFixedRate(new RefreshTimeLabelTimerTask(), 0, 100);
	}

	@Override
	public boolean onCreateOptionsMenu(Menu menu) {
		// Inflate the menu; this adds items to the action bar if it is present.
		getMenuInflater().inflate(R.menu.main, menu);
		return true;
	}

	@Override
	public void onPause() {
		super.onPause(); // always call superclass first
		if( MyDebug.LOG )
			Log.d(TAG, "onPause()");
		this.app_is_active = false;
		this.cancelTimers();
	}
	
	@Override
	public void onResume() {
		super.onResume(); // always call superclass first
		if( MyDebug.LOG )
			Log.d(TAG, "onResume()");
		this.app_is_active = true;

		SharedPreferences sharedPreferences = PreferenceManager.getDefaultSharedPreferences(this);

		// keep screen active - see http://stackoverflow.com/questions/2131948/force-screen-on
		if( sharedPreferences.getBoolean("preference_keep_display_on", true) ) {
			if( MyDebug.LOG )
				Log.d(TAG, "do keep screen on");
			getWindow().addFlags(LayoutParams.FLAG_KEEP_SCREEN_ON);
		}
		else {
			if( MyDebug.LOG )
				Log.d(TAG, "don't keep screen on");
			getWindow().clearFlags(LayoutParams.FLAG_KEEP_SCREEN_ON);
		}

		String name;
        if( !MainApp.IPDefined(MainActivity.this) ) {
			name = "Please set the PC's IP Address from the Settings menu";
		}
        else if( !MainApp.portValid(MainActivity.this) ) {
			name = "Invalid MPC/VLC port - please set from the Settings menu";
		}
		else {
	        if( MainApp.getPlayer(this) == MainApp.PLAYER_VLC ) {
				final String vlc_password = MainApp.getVLCPassword(this);
				// Not sure if it's worth sending an empty password (VLC doesn't seem to allow access if the password isn't set), but might as well send in all cases
				if( MyDebug.LOG )
	    			Log.d(TAG, "startup - set VLC password: " + vlc_password);
				Authenticator.setDefault(new Authenticator() {
					protected PasswordAuthentication getPasswordAuthentication() {
						return new PasswordAuthentication("", vlc_password.toCharArray());
					 }
				});
	        }
			name = "Waiting to retrieve status";
	        int player = MainApp.getPlayer(MainActivity.this);
		    if( player == MainApp.PLAYER_MPC ) {
		    	name += " from Media Player Classic";
		    }
		    else if( player == MainApp.PLAYER_VLC ) {
		    	name += " from VLC";
		    }
		    name += "...";
		}
		final ScrollingTextView nameWidget = (ScrollingTextView)findViewById(R.id.name_widget);
		nameWidget.setText(name);
		// nameWidget initialisation should be done before calling setupTimers!

		this.setupTimers();

	}
	
    @Override
	public boolean onOptionsItemSelected(MenuItem item) {
		switch( item.getItemId() ) {
		case R.id.action_settings:
		{
			if( MyDebug.LOG )
				Log.d(TAG, "options: settings");
	        Intent intent = new Intent(this, MyPreferenceActivity.class);
	        this.startActivity(intent);
			return true;
		}
		case R.id.action_togglefullscreen:
		{
			if( MyDebug.LOG )
				Log.d(TAG, "options: togglefullscreen");
	        int player = MainApp.getPlayer(this);
		    if( player == MainApp.PLAYER_MPC ) {
		    	MainApp.sendMPCCommand(this, 830);
		    }
		    else if( player == MainApp.PLAYER_VLC ) {
		    	MainApp.sendVLCCommand(this, "fullscreen");
		    }
			return true;
		}
		case R.id.action_openfilebrowser:
		{
			if( MyDebug.LOG ) {
				Log.d(TAG, "options: openfilebrowser");
				Log.d(TAG, "clickedDVD");
			}
	        int player = MainApp.getPlayer(this);
		    if( player == MainApp.PLAYER_MPC ) {
		        Intent intent = new Intent(this, FileBrowserActivity.class);
		        this.startActivity(intent);
		    }
		    else {
	    	    Toast.makeText(getApplicationContext(), "File Browser only available with MPC", Toast.LENGTH_SHORT).show();
		    }
			return true;
		}
		case R.id.action_onlinehelp:
		{
			if( MyDebug.LOG )
				Log.d(TAG, "options: onlinehelp");
	        Intent browserIntent = new Intent(Intent.ACTION_VIEW, Uri.parse("http://wifiremoteplay.sourceforge.net/"));
	        startActivity(browserIntent);
	        return true;
		}
		case R.id.action_makeadonation:
		{
			if( MyDebug.LOG )
				Log.d(TAG, "options: makeadonation");
	        Intent browserIntent = new Intent(Intent.ACTION_VIEW, Uri.parse("https://play.google.com/store/apps/details?id=harman.mark.donation"));
	        startActivity(browserIntent);
			return true;
		}
		case R.id.action_webinterface:
		{
			if( MyDebug.LOG )
				Log.d(TAG, "options: webinterface");
	        int player = MainApp.getPlayer(this);
	        if( player == MainApp.PLAYER_MPC ) {
	            String url = "http://" + MainApp.getIPAddress(this) + ":" + MainApp.getMPCPort(this) + "/controls.html";
	    		if( MyDebug.LOG )
	    			Log.d(TAG, "open: " + url);
		        Intent browserIntent = new Intent(Intent.ACTION_VIEW, Uri.parse(url));
		        startActivity(browserIntent);
	        }
	        else if( player == MainApp.PLAYER_VLC ) {
	            String url = "http://" + MainApp.getIPAddress(this) + ":" + MainApp.getVLCPort(this) + "/";
	    		if( MyDebug.LOG )
	    			Log.d(TAG, "open: " + url);
		        Intent browserIntent = new Intent(Intent.ACTION_VIEW, Uri.parse(url));
		        startActivity(browserIntent);
	        }
			return true;
		}
		case R.id.action_about:
		{
			if( MyDebug.LOG )
				Log.d(TAG, "options: about");
	        AlertDialog.Builder alertDialog = new AlertDialog.Builder(this);
            alertDialog.setTitle("Wifi Remote Play v1.12");
            alertDialog.setMessage("by Mark Harman\n\nPlease donate if you like this app :)\n");
            alertDialog.setPositiveButton("OK", null);
            alertDialog.show();
			return true;
		}
		default:
			return super.onOptionsItemSelected(item);
		}
	}

    @SuppressLint("SimpleDateFormat")
	private Calendar convertSecondsToCalendar(int seconds) {
        SimpleDateFormat sdf = new SimpleDateFormat("hh:mm:ss");
		try {
			Date date0 = sdf.parse("00:00:00");
	        Calendar time0 = GregorianCalendar.getInstance(); // creates a new calendar instance
	        time0.setTime(date0);
	        time0.add(Calendar.SECOND, seconds);
	        return time0;
		}
		catch(ParseException e) {
			if( MyDebug.LOG )
				Log.d(TAG, "onStopTrackingTouch: failed to parse time0");
	        e.printStackTrace();
	        return null;
		}
    }
    
    private int convertCalenderToSeconds(Calendar time1) {
        SimpleDateFormat sdf = new SimpleDateFormat("hh:mm:ss");
		try {
			Date date0 = sdf.parse("00:00:00");
	        Calendar time0 = GregorianCalendar.getInstance(); // creates a new calendar instance
	        time0.setTime(date0);
	        return (int)((time1.getTimeInMillis() - time0.getTimeInMillis())/1000);
		}
		catch(ParseException e) {
			if( MyDebug.LOG )
				Log.d(TAG, "failed to parse time0");
	        e.printStackTrace();
		}
		return 0;
    }
    
    private String formatTime(Calendar time1) {
    	// formats time1 as hh:mm:ss
    	return String.format(Locale.US, "%02d", time1.get(Calendar.HOUR)) + ":" + String.format(Locale.US, "%02d", time1.get(Calendar.MINUTE)) + ":" + String.format(Locale.US, "%02d", time1.get(Calendar.SECOND));
    }

    private void moveToTime(int seconds) {
    	Calendar time0 = convertSecondsToCalendar(seconds);
    	if( time0 != null ) {
	        if( have_time ) {
	        	time = time0;
	        	refreshTimeLabel();
	        }
	        have_saved_last_time = false; // status checker may get confused, so need to reset
	        int player = MainApp.getPlayer(this);
	        if( player == MainApp.PLAYER_MPC ) {
	        	MainApp.sendMPCCommand(this, -1, true, "position", formatTime(time0));
	        }
	        else if( player == MainApp.PLAYER_VLC ) {
	        	MainApp.sendVLCCommand(this, "seek&val=" + seconds + "s");
	        }
    	}
    }

    public void clickedPrev(View view) {
		if( MyDebug.LOG )
			Log.d(TAG, "clickedPrev");
	    if( have_time ) {
	        have_time = false;
	        refreshTimeLabel();
	    }
	    have_saved_last_time = false; // status checker may get confused, so need to reset
        int player = MainApp.getPlayer(this);
	    if( player == MainApp.PLAYER_MPC ) {
	    	MainApp.sendMPCCommand(this, 921);
	    }
	    else if( player == MainApp.PLAYER_VLC ) {
	    	MainApp.sendVLCCommand(this, "pl_previous");
	    }
	}

	public void clickedNext(View view) {
		if( MyDebug.LOG )
			Log.d(TAG, "clickedNext");
	    if( have_time ) {
	        have_time = false;
	        refreshTimeLabel();
	    }
	    have_saved_last_time = false; // status checker may get confused, so need to reset
        int player = MainApp.getPlayer(this);
	    if( player == MainApp.PLAYER_MPC ) {
	        //sendMPCCommand(920);
	    	MainApp.sendMPCCommand(this, 922);
	    }
	    else if( player == MainApp.PLAYER_VLC ) {
	    	MainApp.sendVLCCommand(this, "pl_next");
	    }
	}

	public void clickedBwd(View view) {
		if( MyDebug.LOG )
			Log.d(TAG, "clickedBwd");
	    if( have_time ) {
	    	time.add(Calendar.SECOND, -5);
	        refreshTimeLabel();
	    }
	    have_saved_last_time = false; // status checker may get confused, so need to reset
        int player = MainApp.getPlayer(this);
	    if( player == MainApp.PLAYER_MPC ) {
	    	MainApp.sendMPCCommand(this, 901);
	    }
	    else if( player == MainApp.PLAYER_VLC ) {
	    	MainApp.sendVLCCommand(this, "seek&val=-5s");
	    }
	}

	public void clickedFwd(View view) {
		if( MyDebug.LOG )
			Log.d(TAG, "clickedFwd");
	    if( have_time ) {
	    	time.add(Calendar.SECOND, 5);
	        refreshTimeLabel();
	    }
	    have_saved_last_time = false; // status checker may get confused, so need to reset
        int player = MainApp.getPlayer(this);
	    if( player == MainApp.PLAYER_MPC ) {
	    	MainApp.sendMPCCommand(this, 902);
	    }
	    else if( player == MainApp.PLAYER_VLC ) {
	    	MainApp.sendVLCCommand(this, "seek&val=+5s");
	    }
	}

	public void clickedPlay(View view) {
		if( MyDebug.LOG )
			Log.d(TAG, "clickedPlay");

        is_paused = !is_paused;
	    have_saved_last_time = false; // status checker may get confused, so need to reset

        int player = MainApp.getPlayer(this);
	    if( player == MainApp.PLAYER_MPC ) {
	    	MainApp.sendMPCCommand(this, 889);
	    }
	    else if( player == MainApp.PLAYER_VLC ) {
	    	MainApp.sendVLCCommand(this, "pl_pause");
	    }
	}

	public void clickedVolumeMinus(View view) {
		if( MyDebug.LOG )
			Log.d(TAG, "clickedVolumeMinus");
        int player = MainApp.getPlayer(this);
	    if( player == MainApp.PLAYER_MPC ) {
	    	MainApp.sendMPCCommand(this, 908);
	    }
	    else if( player == MainApp.PLAYER_VLC ) {
	    	MainApp.sendVLCCommand(this, "volume&val=-20");
	    }
	}

	public void clickedVolumePlus(View view) {
		if( MyDebug.LOG )
			Log.d(TAG, "clickedVolumePlus");
        int player = MainApp.getPlayer(this);
	    if( player == MainApp.PLAYER_MPC ) {
	    	MainApp.sendMPCCommand(this, 907);
	    }
	    else if( player == MainApp.PLAYER_VLC ) {
	    	MainApp.sendVLCCommand(this, "volume&val=+20");
	    }
	}

	public void clickedVolumeMute(View view) {
		if( MyDebug.LOG )
			Log.d(TAG, "clickedVolumeMute");
        int player = MainApp.getPlayer(this);
	    if( player == MainApp.PLAYER_MPC ) {
	    	MainApp.sendMPCCommand(this, 909);
	    }
	    else if( player == MainApp.PLAYER_VLC ) {
	        if( is_muted ) {
	        	MainApp.sendVLCCommand(this, "volume&val=" + saved_volume);
	        }
	        else {
	            MainApp.sendVLCCommand(this, "volume&val=0");
	        }
	        is_muted = !is_muted;
	    }
	}

	public void clickedDVD(View view) {
		if( MyDebug.LOG )
			Log.d(TAG, "clickedDVD");
        int player = MainApp.getPlayer(this);
	    if( player == MainApp.PLAYER_MPC ) {
	        Intent intent = new Intent(this, DVDActivity.class);
	        this.startActivity(intent);
	    }
	    else {
    	    Toast.makeText(getApplicationContext(), "DVD Control only available with MPC", Toast.LENGTH_SHORT).show();
	    }
	}
	
	@TargetApi(Build.VERSION_CODES.HONEYCOMB)
	private void requestStatus() {
    	if( !MainApp.IPDefined(this) ) {
    		return;
    	}
    	if( !MainApp.portValid(this) ) {
    		return;
    	}
		if( !this.app_is_active ) {
			return;
		}

		if( MyDebug.LOG )
			Log.d(TAG, "requestStatus: " + MainApp.getIPAddress(this));
	    if( waiting_for_status ) {
	        // already sent a request
			if( MyDebug.LOG )
				Log.d(TAG, "already sent a request");
	        return;
	    }
	    waiting_for_status = true;
		if( MyDebug.LOG )
			Log.d(TAG, "send status request");
        String url_string = "";
        int player = MainApp.getPlayer(this);
	    if( player == MainApp.PLAYER_MPC ) {
	    	url_string = "http://" + MainApp.getIPAddress(this) + ":" + MainApp.getMPCPort(this) + "/status.html";
	    }
	    else if( player == MainApp.PLAYER_VLC ) {
	    	url_string = "http://" + MainApp.getIPAddress(this) + ":" + MainApp.getVLCPort(this) + "/requests/status.xml";
	    }

        if( Build.VERSION.SDK_INT >= Build.VERSION_CODES.HONEYCOMB ) {
        	// AsyncTask must run on UI thread on Android 3.x at least
        	final String url_string_final = url_string;
    		runOnUiThread(new Runnable() {
    		    public void run() {
    	        	new RequestStatusTask().executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR, url_string_final);
    		    }
    		});
        }
        else {
        	// AsyncTask must run on UI thread on Android 2.x
        	final String url_string_final = url_string;
    		runOnUiThread(new Runnable() {
    		    public void run() {
    	    	    new RequestStatusTask().execute(url_string_final);
    		    }
    		});
        }
	}

	private class RequestStatusTask extends AsyncTask<String, Integer, Integer> {
	    protected Integer doInBackground(String... urls) {
            // only care about 1st parameter
	    	String url_string = urls[0];
	 	    URL url;
		    HttpURLConnection urlConnection = null;
			InputStream inputStream = null;
            try {
		    	url = new URL(url_string);
		    	urlConnection = (HttpURLConnection)url.openConnection();

	    	    inputStream = new BufferedInputStream(urlConnection.getInputStream());

	    	    ByteArrayOutputStream buffer = new ByteArrayOutputStream();

	    	    int nRead;
	    	    byte[] data = new byte[16384];
	    	    while ((nRead = inputStream.read(data, 0, data.length)) != -1) {
	    	      buffer.write(data, 0, nRead);
	    	    }

	    	    buffer.flush();

	    	    byte [] result = buffer.toByteArray();
	    	    parseStatus(result);
            }
		    catch(MalformedURLException e) {
				if( MyDebug.LOG )
					Log.d(TAG, "failed to create URL: MalformedURLException: " + url_string);
		        e.printStackTrace();
		    }
		    catch(IOException e) {
				if( MyDebug.LOG )
					Log.d(TAG, "failed to send http request or parse response: IOException: " + url_string);
		        e.printStackTrace();
		    }
		    catch(Exception e) {
	        	// Android bug? See http://stackoverflow.com/questions/15557355/android-httpurlconnection-getinputstream-throws-nullpointerexception
				if( MyDebug.LOG )
					Log.d(TAG, "failed to create URL or send http request or parse response: unknown exception: " + url_string);
				e.printStackTrace();
		    }
		    finally {
				if( inputStream != null ) {
					try {
						inputStream.close();
					}
					catch(IOException e) {
						e.printStackTrace();
					}
				}
		    	if( urlConnection != null )
		    		urlConnection.disconnect();
			}
	    	
            return 0;
        }

	    protected void onProgressUpdate(Integer... progress) {
	    }

	    protected void onPostExecute(Integer result) {
	    }
	}

	static class StatusInfo {
		String name = "";
		String time = "";
		String length = "";
		int progress = 0;
		int progress_max = 0;
	}

	private void updateStatusWidgets(final StatusInfo statusInfo) {
		final ScrollingTextView nameWidget = (ScrollingTextView)findViewById(R.id.name_widget);
		final TextView timeWidget = (TextView)findViewById(R.id.time_widget);
		final TextView lengthWidget = (TextView)findViewById(R.id.length_widget);
		final SeekBar progressWidget = (SeekBar)findViewById(R.id.progress);
		runOnUiThread(new Runnable() {
		    public void run() {
	    		nameWidget.setText(statusInfo.name);
				timeWidget.setText(statusInfo.time);
				lengthWidget.setText(statusInfo.length);
				progressWidget.setProgress(statusInfo.progress);
				progressWidget.setMax(statusInfo.progress_max);
		    }
		});
	}

	@SuppressLint("SimpleDateFormat")
	private void parseStatus(byte [] data) {
	    if( !waiting_for_status ) {
	        return;
	    }
	    waiting_for_status = false;

		if( MyDebug.LOG ) {
			Log.d(TAG, "parseStatus received " + data.length + " bytes");
			/*try {
				Log.d(TAG, "data:" + new String(data, "UTF-8"));
			}
			catch(UnsupportedEncodingException e) {
				// don't care, only logging
			}*/
	        /*for(int i=0;i<data.length;i++){
	        	Log.d(TAG, "" + data[i]);
	        }*/
		}
        int player = MainApp.getPlayer(this);
		if( player == MainApp.PLAYER_MPC ) {
	        // n.b., data may be empty, e.g. if MPC not running!

			final StatusInfo statusInfo = new StatusInfo();

			// ignore OnStatus(
			final int offset = 9;
			byte [] info_utf8 = new byte[data.length - offset];
			System.arraycopy(data, offset, info_utf8, 0, data.length - 9);
			try {
				String info_s = new String(info_utf8, "UTF-8");
				StringBuilder info = new StringBuilder(info_s);
				if( info.length() > 0 ) {
					info.deleteCharAt(info.length()-1); // chop off closing bracket
				}
		        // find first ' character not as a \'
				boolean found = false;
				int index = 0;
				char quote = '\'';
		        // MPC 1.6.0.4014 and earlier uses ', 1.6.3.5818 and later uses " ...
				if( index < info.length() && info.charAt(index) == quote )
		            index++; // ignore first '
		        else if( index < info.length() && info.charAt(index) == '\"' ) {
		            quote = '\"';
		            index++; // ignore first "
		        }
		        int name_starts_at = index;
		        while( !found ) {
		            index = info.indexOf(String.valueOf(quote), index);
		            if( index == -1 ) {
		        		if( MyDebug.LOG )
		        			Log.d(TAG, "can't find name string");
						updateStatusWidgets(statusInfo);
		                return;
		            }
		            if( index > 0 && info.charAt(index-1) == '\\' ) {
		                // not a real ', part of the name string, so carry on searching
		                index++;
		            }
		            else {
		                found = true;
		            }
		        }
				if( MyDebug.LOG ) {
					Log.d(TAG, "found? " + found);
					Log.d(TAG, "index = " + index);
				}
				String name_s = info.substring(name_starts_at, index);
				StringBuffer name = new StringBuffer(name_s);
		        int ignore_mpc = name.lastIndexOf(" - Media Player Classic Home Cinema - v");
		        if( ignore_mpc != -1 ) {
		            // for earlier versions of MPC, e.g., v1.4.2677.0
		            name.setLength(ignore_mpc);
		        }
		        name_s = name.toString();
		        name_s = name_s.replace("\\'", "'"); // replace \' with '
		        name = new StringBuffer(name_s);
				if( MyDebug.LOG )
					Log.d(TAG, "name: " + name_s);
		        statusInfo.name = name_s;
		        
		        String remainder = info.substring(index);
		        // Once More, With Feeling:
		        // we have to do the filename separately and chop it off, to avoid problems caused by filenames with commas
		        String [] infoList = remainder.split(",");
				if( MyDebug.LOG ) {
			        for(int i=0;i<infoList.length;i++) {
				        Log.d(TAG, "infoList[" + i + "]: " + infoList[i]);
			        }
				}
		        if( infoList.length <= 3 ) {
					updateStatusWidgets(statusInfo);
		        	return;
		        }
		        String time_s = infoList[3];
		        StringBuilder time_sb = new StringBuilder(time_s);
		        if( time_sb.length() >= 3 ) {
		        	time_sb.delete(0, 2); // get rid of space and first quote
		        	time_sb.deleteCharAt(time_sb.length()-1); // get rid of last quote
		        }
				if( MyDebug.LOG )
					Log.d(TAG, "time_sb: " + time_sb);

		        SimpleDateFormat sdf = new SimpleDateFormat("hh:mm:ss");
				try {
					Date date = sdf.parse(time_sb.toString());
			        Calendar new_time = GregorianCalendar.getInstance(); // creates a new calendar instance
			        new_time.setTime(date);   // assigns calendar to given date 
					if( MyDebug.LOG )
						Log.d(TAG, "new_time to string: " + new_time.toString());
			        if( have_saved_last_time ) {
			    		if( MyDebug.LOG )
			    			Log.d(TAG, "    compare to saved_last_time to string: " + saved_last_time.toString());
			            if( new_time.equals(saved_last_time) ) {
			                // probably paused?
			                is_paused = true;
			        		if( MyDebug.LOG )
			        			Log.d(TAG, "### paused?");
			            }
			            else {
			                // assume not paused
			                is_paused = false;
			        		if( MyDebug.LOG )
			        			Log.d(TAG, "### playing?");
			            }
			        }
			        time = new_time;
			        have_time = true;
			        saved_last_time = (Calendar)time.clone();
			        have_saved_last_time = true;
			        time_last_updated = SystemClock.elapsedRealtime();

			        statusInfo.time = formatTime(time);
					statusInfo.progress = convertCalenderToSeconds(time);
				}
				catch(ParseException e) {
					if( MyDebug.LOG )
						Log.d(TAG, "failed to parse time: " + time_sb);
					e.printStackTrace();
				}

		        if( infoList.length <= 5 ) {
					updateStatusWidgets(statusInfo);
		            return;
		        }
		        String length_s = infoList[5];
		        final StringBuilder length_sb = new StringBuilder(length_s);
		        if( length_sb.length() >= 3 ) {
		        	length_sb.delete(0, 2); // get rid of space and first quote
		        	length_sb.deleteCharAt(length_sb.length()-1); // get rid of last quote
		        }
				if( MyDebug.LOG )
					Log.d(TAG, "length: " + length_sb);
		        statusInfo.length = "/" + length_sb.toString();

				try {
					Date date = sdf.parse(length_sb.toString());
			        Calendar length_time = GregorianCalendar.getInstance(); // creates a new calendar instance
			        length_time.setTime(date);   // assigns calendar to given date 
					if( MyDebug.LOG )
						Log.d(TAG, "length to string: " + length_time.toString());
			        final int length_seconds = convertCalenderToSeconds(length_time);
					if( MyDebug.LOG )
						Log.d(TAG, "length in seconds: " + length_seconds);
			        statusInfo.progress_max = length_seconds;
				}
				catch(ParseException e) {
					if( MyDebug.LOG )
						Log.d(TAG, "failed to parse time: " + time_sb);
					e.printStackTrace();
				}
			}
			catch(UnsupportedEncodingException e) {
				if( MyDebug.LOG )
					Log.d(TAG, "UnsupportedEncodingException");
				e.printStackTrace();
				updateStatusWidgets(statusInfo);
				return;
			}

			updateStatusWidgets(statusInfo);
		}
		else if( player == MainApp.PLAYER_VLC ) {
			final StatusInfo statusInfo = new StatusInfo();

			ByteArrayInputStream inputstream = new ByteArrayInputStream(data);
			InputStreamReader reader = new InputStreamReader(inputstream);
			XmlPullParser parser = Xml.newPullParser();
			try {
				parser.setInput(reader);
				parser.setFeature(XmlPullParser.FEATURE_PROCESS_NAMESPACES, false);
				if( MyDebug.LOG )
					Log.d(TAG, "parse VLC XML...");

				int eventType = parser.getEventType();
				while( eventType != XmlPullParser.END_DOCUMENT ) {
					if( eventType == XmlPullParser.START_DOCUMENT ) {
				    }
					else if( eventType == XmlPullParser.START_TAG ) {
		                if( parser.getName().equals("volume") ) {
		                	String element = parser.nextText();
		            		if( MyDebug.LOG )
		            			Log.d(TAG, "volume = " + element);
							try {
								int volume = Integer.parseInt(element);
			                    if( volume > 0 ) {
			                        saved_volume = volume;
			                    }
							}
			                catch(NumberFormatException e) {
			            		if( MyDebug.LOG )
			            			Log.d(TAG, "failed to parse volume: " + element);
			            		e.printStackTrace();
			                }
		                }
		                else if( parser.getName().equals("state") ) {
		                	String element = parser.nextText();
		            		if( MyDebug.LOG )
		            			Log.d(TAG, "state = " + element);
							is_paused = !element.equals("playing");
		                }
		                else if( parser.getName().equals("length") ) {
		                	String element = parser.nextText();
		            		if( MyDebug.LOG )
		            			Log.d(TAG, "length = " + element);
							try {
								final int length_seconds = Integer.parseInt(element);
						    	final Calendar time0 = convertSecondsToCalendar(length_seconds);
						    	if( time0 != null ) {
						    		statusInfo.length = "/ " + formatTime(time0);
						    	}
					    		statusInfo.progress_max = length_seconds;
							}
			                catch(NumberFormatException e) {
			            		if( MyDebug.LOG )
			            			Log.d(TAG, "failed to parse length: " + element);
			            		e.printStackTrace();
			                }
		                }
		                else if( parser.getName().equals("time") ) {
		                	String element = parser.nextText();
		            		if( MyDebug.LOG )
		            			Log.d(TAG, "time = " + element);
							try {
								int time_seconds = Integer.parseInt(element);
						    	Calendar new_time = convertSecondsToCalendar(time_seconds);
						    	if( new_time != null ) {
							        time = new_time;
							        have_time = true;
							        saved_last_time = (Calendar)time.clone();
							        have_saved_last_time = true;
							        time_last_updated = SystemClock.elapsedRealtime();
	
									statusInfo.time = formatTime(time);
									statusInfo.progress = time_seconds;
						    	}
							}
			                catch(NumberFormatException e) {
			            		if( MyDebug.LOG )
			            			Log.d(TAG, "failed to parse time: " + element);
			            		e.printStackTrace();
			                }
		                }
		                else if( parser.getName().equals("information") ) {
		                    parseStatusInformation(statusInfo, parser);
		                }
					}
					else if( eventType == XmlPullParser.END_TAG ) {
					}
					else if( eventType == XmlPullParser.TEXT ) {
					}
					eventType = parser.next();
				}
			}
			catch(XmlPullParserException e) {
				if( MyDebug.LOG )
					Log.d(TAG, "XmlPullParserException");
				e.printStackTrace();
			}
			catch(IOException e) {
				if( MyDebug.LOG )
					Log.d(TAG, "IOException");
				e.printStackTrace();
			}

			updateStatusWidgets(statusInfo);
		}
	}
	
	private void parseStatusInformation(StatusInfo statusInfo, XmlPullParser parser) throws XmlPullParserException, IOException {
	    // parses status from VLC in the <information> ... </information>
		if( MyDebug.LOG )
			Log.d(TAG, "reading VLC information...");
		int eventType = parser.getEventType();
		while( eventType != XmlPullParser.END_DOCUMENT ) {
			if( eventType == XmlPullParser.START_DOCUMENT ) {
		    }
			else if( eventType == XmlPullParser.START_TAG ) {
                if( parser.getName().equals("meta-information") ) {
                	parseStatusMetaInformation(statusInfo, parser);
                }
                else if( parser.getName().equals("category") ) {
                	if( parser.getAttributeValue(null, "name").equals("meta") ) {
	                    parseStatusMetaCategory(statusInfo, parser);
                	}
                }
			}
			else if( eventType == XmlPullParser.END_TAG ) {
                if( parser.getName().equals("information") ) {
                	break;
                }
			}
			eventType = parser.next();
		}
	}

	private void parseStatusMetaInformation(StatusInfo statusInfo, XmlPullParser parser) throws XmlPullParserException, IOException {
	    // parses status from VLC in the <meta-information> ... </meta-information> [v1 API]
		if( MyDebug.LOG )
			Log.d(TAG, "reading meta-information... [v1 API]");

	    String text_title = "", text_artist = "";
		int eventType = parser.getEventType();
		while( eventType != XmlPullParser.END_DOCUMENT ) {
			if( eventType == XmlPullParser.START_DOCUMENT ) {
		    }
			else if( eventType == XmlPullParser.START_TAG ) {
                if( parser.getName().equals("title") ) {
                	String element = parser.nextText();
            		if( MyDebug.LOG )
            			Log.d(TAG, "title = " + element);
                    // need to decode HTML "&" encoding etc
                    //noinspection deprecation
                    text_title = Html.fromHtml(element).toString();
                }
                else if( parser.getName().equals("artist") ) {
                	String element = parser.nextText();
            		if( MyDebug.LOG )
            			Log.d(TAG, "artist = " + element);
                    // need to decode HTML "&" encoding etc
                    //noinspection deprecation
                    text_artist = Html.fromHtml(element).toString();
                }
			}
			else if( eventType == XmlPullParser.END_TAG ) {
				if( parser.getName().equals("meta-information") ) {
                	break;
                }
			}
			eventType = parser.next();
		}

		if( text_title.length() > 0 || text_artist.length() > 0 ) {
		    String text_name = "";
	        if( text_artist.length() > 0 ) {
	            text_name += text_artist;
	        }
	        if( text_title.length() > 0 ) {
	            if( text_name.length() > 0 ) {
	                text_name += " - ";
	            }
	            text_name += text_title;
	        }
	        statusInfo.name = text_name;
	    }
	}

	private void parseStatusMetaCategory(StatusInfo statusInfo, XmlPullParser parser) throws XmlPullParserException, IOException {
	    // parses status from VLC in the <category name="meta"> ... </category> [v2 API]
		if( MyDebug.LOG )
			Log.d(TAG, "reading category meta... [v2 API]");
	    // It seems that if the file is tagged, "filename" isn't the actual filename!
	    // So we prefer looking at title/artist instead, if available
	    String text_title = "", text_artist = "", text_filename = "";
		int eventType = parser.getEventType();
		while( eventType != XmlPullParser.END_DOCUMENT ) {
			if( eventType == XmlPullParser.START_DOCUMENT ) {
		    }
			else if( eventType == XmlPullParser.START_TAG ) {
                if( parser.getName().equals("info") ) {
                	if( parser.getAttributeValue(null, "name").equals("title") ) {
	                	String element = parser.nextText();
	            		if( MyDebug.LOG )
	            			Log.d(TAG, "title = " + element);
                        // need to decode HTML "&" encoding etc
                        //noinspection deprecation
                        text_title = Html.fromHtml(element).toString();
                	}
                	else if( parser.getAttributeValue(null, "name").equals("artist") ) {
	                	String element = parser.nextText();
	            		if( MyDebug.LOG )
	            			Log.d(TAG, "artist = " + element);
                        // need to decode HTML "&" encoding etc
                        //noinspection deprecation
                        text_artist = Html.fromHtml(element).toString();
                	}
                	else if( parser.getAttributeValue(null, "name").equals("filename") ) {
	                	String element = parser.nextText();
	            		if( MyDebug.LOG )
	            			Log.d(TAG, "filename = " + element);
                        // need to decode HTML "&" encoding etc
                        //noinspection deprecation
                        text_filename = Html.fromHtml(element).toString();
                	}
                }
			}
			else if( eventType == XmlPullParser.END_TAG ) {
                if( parser.getName().equals("category") ) {
                	break;
                }
			}
			eventType = parser.next();
		}
		
		if( text_title.length() > 0 || text_artist.length() > 0 ) {
		    String text_name = "";
	        if( text_artist.length() > 0 ) {
	            text_name += text_artist;
	        }
	        if( text_title.length() > 0 ) {
	            if( text_name.length() > 0 ) {
	                text_name += " - ";
	            }
	            text_name += text_title;
	        }
	        statusInfo.name = text_name;
	    }
	    else {
	        statusInfo.name = text_filename;
	    }
	}

	private void refreshTimeLabel() {
		if( !this.app_is_active ) {
			return;
		}
		/*if( MyDebug.LOG )
			Log.d(TAG, "refreshTimeLabel");*/
		runOnUiThread(new Runnable() {
		    public void run() {
			    if( have_time ) {
					/*if( MyDebug.LOG )
						Log.d(TAG, "is_paused? " + is_paused);*/
			        if( !is_paused ) {
			            long elapsed = (SystemClock.elapsedRealtime() - time_last_updated);
			            time_last_updated = SystemClock.elapsedRealtime();
				    	time.add(Calendar.MILLISECOND, (int)elapsed);
						/*if( MyDebug.LOG )
							Log.d(TAG, "elapsed:" + elapsed);*/
			        }
					TextView timeWidget = (TextView)findViewById(R.id.time_widget);
					timeWidget.setText( formatTime(time) );
					SeekBar progress = (SeekBar)findViewById(R.id.progress);
			        int time_seconds = convertCalenderToSeconds(time);
					progress.setProgress(time_seconds);
			    }
			    else {
					TextView timeWidget = (TextView)findViewById(R.id.time_widget);
					timeWidget.setText("");
					SeekBar progress = (SeekBar)findViewById(R.id.progress);
					progress.setProgress(0);
			    }
		    }
		});
	}

    @Override
    public boolean dispatchKeyEvent(KeyEvent event) {
        if( event.getAction() == KeyEvent.ACTION_DOWN ) {
            int keyCode = event.getKeyCode();
            switch( keyCode ) {
            case KeyEvent.KEYCODE_VOLUME_UP:
            	clickedVolumePlus(null);
            	return true;
            case KeyEvent.KEYCODE_VOLUME_DOWN:
            	clickedVolumeMinus(null);
            	return true;
            }
        }
        return super.dispatchKeyEvent(event);
    }
}
