package net.sourceforge.wifiremoteplay;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;

import android.annotation.TargetApi;
import android.content.Context;
import android.content.SharedPreferences;
import android.os.AsyncTask;
import android.os.Build;
import android.preference.PreferenceManager;
import android.util.Log;

class MainApp {
	private static final String TAG = "MainApp";

	public static final int PLAYER_UNKNOWN = -1;
	public static final int PLAYER_MPC = 0;
	public static final int PLAYER_VLC = 1;

	public static int getPlayer(Context context) {
		SharedPreferences sharedPreferences = PreferenceManager.getDefaultSharedPreferences(context);
		String player_s = sharedPreferences.getString("preference_player", "");
		if( player_s.equals("mpc") )
			return PLAYER_MPC;
		else if( player_s.equals("vlc") )
			return PLAYER_VLC;
		if( MyDebug.LOG )
			Log.d(TAG, "getPlayer(): unknown player!: " + player_s);
		return PLAYER_UNKNOWN;
	}
	
    public static boolean IPDefined(Context context) {
    	String ip_address = getIPAddress(context);
		return !(ip_address.length() < 4 || ip_address.contains("..") || ip_address.charAt(0) == '.' || ip_address.charAt(ip_address.length() - 1) == '.');
	}

    public static String getIPAddress(Context context) {
		SharedPreferences sharedPreferences = PreferenceManager.getDefaultSharedPreferences(context);
		return sharedPreferences.getString("preference_ip_address", "");
	}
    
    public static boolean portValid(Context context) {
    	int player = getPlayer(context);
    	if( player == PLAYER_MPC )
    		return getMPCPort(context) != -1;
    	else if( player == PLAYER_VLC )
    		return getVLCPort(context) != -1;
    	return false;
    }
    
    public static int getMPCPort(Context context) {
		SharedPreferences sharedPreferences = PreferenceManager.getDefaultSharedPreferences(context);
		String mpc_port_s = sharedPreferences.getString("preference_mpc_port", "0");
		try {
			int mpc_port = mpc_port_s == null ? -1 : Integer.parseInt(mpc_port_s);
			if( MyDebug.LOG )
				Log.d(TAG, "getMPCPort() returns " + mpc_port);
			return mpc_port;
		}
		catch(NumberFormatException e) {
			// can happen if, e.g., number bigger than can be fit into 32-bit int
			if( MyDebug.LOG )
				Log.d(TAG, "getMPCPort: numberformatexception on: " + mpc_port_s);
			// don't need to print stack trace
			return -1;
		}
    }

    public static int getVLCPort(Context context) {
		SharedPreferences sharedPreferences = PreferenceManager.getDefaultSharedPreferences(context);
		String vlc_port_s = sharedPreferences.getString("preference_vlc_port", "0");
		try {
			int vlc_port = vlc_port_s == null ? -1 : Integer.parseInt(vlc_port_s);
			if( MyDebug.LOG )
				Log.d(TAG, "getVLCPort() returns " + vlc_port);
			return vlc_port;
		}
		catch(NumberFormatException e) {
			// can happen if, e.g., number bigger than can be fit into 32-bit int
			if( MyDebug.LOG )
				Log.d(TAG, "getVLCPort: numberformatexception on: " + vlc_port_s);
			// don't need to print stack trace
			return -1;
		}
    }
    
    public static String getVLCPassword(Context context) {
		SharedPreferences sharedPreferences = PreferenceManager.getDefaultSharedPreferences(context);
		return sharedPreferences.getString("preference_vlc_password", "");
    }

	public static void sendMPCCommand(Context context, int command) {
	    sendMPCCommand(context, command, false, "", "");
	}

	@TargetApi(Build.VERSION_CODES.HONEYCOMB)
	public static void sendMPCCommand(final Context context, final int command, final boolean has_extra, final String extra_name, final String extra_value) {
		if( MyDebug.LOG )
			Log.d(TAG, "SendMPCCommand " + command);

    	class SendMPCCommandTask extends AsyncTask<Void, Void, Void> {
    	    protected Void doInBackground(Void...dummy) {
    	    	if( !IPDefined(context) ) {
    	    		if( MyDebug.LOG )
    	    			Log.d(TAG, "SendMPCCommandTask - ip not defined");
    		        return null;
    	    	}
    	    	if( !portValid(context) ) {
    	    		if( MyDebug.LOG )
    	    			Log.d(TAG, "SendMPCCommandTask - port not valid");
    		        return null;
    	    	}
    			if( MyDebug.LOG )
    				Log.d(TAG, "SendMPCCommandTask received command " + command);

				HttpURLConnection urlConnection = null;
				try {
					URL url = new URL("http://" + getIPAddress(context) + ":" + getMPCPort(context) + "/command.html");
					urlConnection = (HttpURLConnection)url.openConnection();
					urlConnection.setRequestMethod("POST");
					String parameters = "wm_command=" + command;
					if( has_extra ) {
						parameters += "&" + extra_name + "=" + extra_value;
					}
					if( MyDebug.LOG )
						Log.d(TAG, "SendMPCCommandTask parameters: " + parameters);
					urlConnection.setDoOutput(true);

					DataOutputStream dataStream = new DataOutputStream(urlConnection.getOutputStream());
					dataStream.writeBytes(parameters);

					dataStream.flush();
					dataStream.close();

					int responseCode = urlConnection.getResponseCode();
					if( MyDebug.LOG )
						Log.d(TAG, "SendMPCCommandTask received responseCode: " + responseCode);
				}
				catch(MalformedURLException e) {
					if( MyDebug.LOG )
						Log.d(TAG, "MalformedURLException while trying to send MPC command");
					e.printStackTrace();
				}
				catch(IOException e) {
					if( MyDebug.LOG )
						Log.d(TAG, "IOException while trying to send MPC command");
					e.printStackTrace();
				}
				catch(Exception e) {
					// Android bug? See http://stackoverflow.com/questions/15557355/android-httpurlconnection-getinputstream-throws-nullpointerexception
					if( MyDebug.LOG )
						Log.d(TAG, "failed to create URL or send http request or parse response: unknown exception");
					e.printStackTrace();
				}
				finally {
					if( urlConnection != null ) {
						urlConnection.disconnect();
					}
				}
		        return null;
            }

		}

        // want to run threads in parallel
        if( Build.VERSION.SDK_INT >= Build.VERSION_CODES.HONEYCOMB ) {
        	new SendMPCCommandTask().executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR);
        }
        else {
    	    new SendMPCCommandTask().execute();
        }
	}

	@TargetApi(Build.VERSION_CODES.HONEYCOMB)
	public static void sendVLCCommand(final Context context, final String command) {
		if( MyDebug.LOG )
			Log.d(TAG, "sendVLCCommand " + command);

    	class SendVLCCommandTask extends AsyncTask<Void, Void, Void> {
    	    protected Void doInBackground(Void...dummy) {
    	    	if( !IPDefined(context) ) {
    	    		if( MyDebug.LOG )
    	    			Log.d(TAG, "SendVLCCommandTask - ip not defined");
    		        return null;
    	    	}
    	    	if( !portValid(context) ) {
    	    		if( MyDebug.LOG )
    	    			Log.d(TAG, "SendVLCCommandTask - port not valid");
    		        return null;
    	    	}
    			if( MyDebug.LOG )
    				Log.d(TAG, "SendVLCCommandTask received command " + command);

    			HttpURLConnection urlConnection = null;
    	        try {
    	        	URL url = new URL("http://" + getIPAddress(context) + ":" + getVLCPort(context) + "/requests/status.xml?command=" + command);
    	        	urlConnection = (HttpURLConnection)url.openConnection();
    	        	InputStream in = new BufferedInputStream(urlConnection.getInputStream());
    	        	BufferedReader rd = new BufferedReader(new InputStreamReader(in), 4096);
    	        	while( rd.readLine() != null ) {
    	        	}
    	        	rd.close();

    	        }
    	        catch(MalformedURLException e) {
    	    		if( MyDebug.LOG )
    	    			Log.d(TAG, "MalformedURLException while trying to send VLC command");
    	    		e.printStackTrace();
    	        }
    	        catch(IOException e) {
    	    		if( MyDebug.LOG )
    	    			Log.d(TAG, "IOException while trying to send VLC command");
    	            e.printStackTrace();
    	        }
    		    catch(Exception e) {
    	        	// Android bug? See http://stackoverflow.com/questions/15557355/android-httpurlconnection-getinputstream-throws-nullpointerexception
    				if( MyDebug.LOG )
    					Log.d(TAG, "failed to create URL or send http request or parse response: unknown exception");
    				e.printStackTrace();
    		    }
    	        finally {
    	        	if( urlConnection != null ) {
    	        		urlConnection.disconnect();
    	        	}
    	        }
		        return null;
            }
    	}

        // want to run threads in parallel
        if( Build.VERSION.SDK_INT >= Build.VERSION_CODES.HONEYCOMB ) {
        	new SendVLCCommandTask().executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR);
        }
        else {
    	    new SendVLCCommandTask().execute();
        }
	}
}
