package net.sourceforge.wifiremoteplay;

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Rect;
import android.os.SystemClock;
import android.util.AttributeSet;
import android.util.DisplayMetrics;
import android.util.Log;
import android.view.View;
import android.view.WindowManager;

public class ScrollingTextView extends View {
	private static final String TAG = "ScrollingTextView";

	private final Paint p = new Paint();
	private final Rect textBounds = new Rect();

	// attributes
	private String text = "";

	private boolean have_last_time = false;
	private long last_time = 0;
	private final static float initial_step_c = -32; // start with slight offset;
	private float step = initial_step_c;

    public ScrollingTextView(Context context, AttributeSet attrs) {
        super(context, attrs);
		if( MyDebug.LOG )
			Log.d(TAG, "ScrollingTextView constructor");

        last_time = SystemClock.elapsedRealtime();

		p.setFlags(Paint.ANTI_ALIAS_FLAG);

		TypedArray a = context.getTheme().obtainStyledAttributes(
            attrs,
            R.styleable.ScrollingTextView,
            0, 0);

        try {
            text = a.getString(R.styleable.ScrollingTextView_text);
    		p.setColor( a.getColor(R.styleable.ScrollingTextView_textColor, Color.WHITE) );
    		DisplayMetrics dm = new DisplayMetrics();
    		WindowManager wm = (WindowManager)context.getSystemService(Context.WINDOW_SERVICE);
    		wm.getDefaultDisplay().getMetrics(dm);
    		float defaultPixelSize = 18.0f * dm.scaledDensity;
    		p.setTextSize( a.getDimension(R.styleable.ScrollingTextView_textSize, defaultPixelSize) );
        }
        finally {
            a.recycle();
        }
    }

    private String getText() {
    	return text;
    }
    
    void setText(String text) {
    	this.text = text;
    	this.invalidate();
    }

    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
		/*if( MyDebug.LOG )
			Log.d(TAG, "onMeasure");*/
        String text = this.getText();
		p.getTextBounds(text, 0, text.length(), textBounds);
        int required_width = MeasureSpec.getSize(widthMeasureSpec);
        int required_height = textBounds.bottom - textBounds.top;
		/*if( MyDebug.LOG )
			Log.d(TAG, "request size " + required_width + " x " + required_height);*/
        this.setMeasuredDimension(required_width, required_height);
    }

    @Override
    protected void onDraw(Canvas canvas) {
		/*if( MyDebug.LOG ) {
			Log.d(TAG, "onDraw");
			Log.d(TAG, "width " + canvas.getWidth() + " height " + canvas.getHeight());
		}*/
        String text = this.getText();
		p.getTextBounds(text, 0, text.length(), textBounds);
		/*if( MyDebug.LOG ) {
			Log.d(TAG, "text bounds top " + textBounds.top + " bottom " + textBounds.bottom);
			Log.d(TAG, "text bounds left " + textBounds.left + " right " + textBounds.right);
		}*/
		int text_wid = textBounds.right;
		int x_pos = 0;

		if( text_wid > canvas.getWidth() ) {
			// need to scroll
			long time_now = SystemClock.elapsedRealtime();
			int elapsed_time = 0;
			if( have_last_time ) {
				elapsed_time = (int)(time_now - last_time);
			}
			/*if( MyDebug.LOG )
				Log.d(TAG, "elapsed_time = " + elapsed_time);*/
	        last_time = time_now;
	        have_last_time = true;
	        final float ms_for_full_width = 10000;
	        final float pixels_per_ms = canvas.getWidth() / ms_for_full_width;
	        step += (pixels_per_ms * elapsed_time);
			x_pos = - (int)step;
		    if( x_pos < - text_wid ) {
		        // wraparound
				if( MyDebug.LOG )
					Log.d(TAG, "wraparound: " + x_pos + ", " + text_wid);
		        step = - canvas.getWidth();
		    }
		}
		else {
			have_last_time = false;
			step = initial_step_c;
		}

		// ensure still works on Android 2.3.3 if we change this!
		canvas.drawText(text, x_pos, - textBounds.top, p);

    	this.invalidate(); // force continual refresh
    	
    }
}
