/*
 * Copyright 2010 Rodrigo Damazio <rodrigo@damazio.org>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.damazio.notifier.notification;

/**
 * Interface that defines a method of sending notifications.
 *
 * @author rdamazio
 */
interface NotificationMethod {
  
  /**
   * Callback to be called after a notification is sent, or its sending
   * fails permanently.
   */
  interface NotificationCallback {
    /**
     * Indicates that the given notification's sending is done (either successfully or with a
     * failure).
     *
     * @param failureReason the exception that caused the failure, if there was one
     */
    void notificationDone(Object target, Throwable failureReason);
  }

  /**
   * Sends a notification to the desktop application.
   * This should never be called if {@link #isEnabled} returns false.
   *
   * @param payload the notification contents to send
   * @param target the target to send to (returned from {@link #getTargets})
   * @param callback callback which is called
   * @param isForeground true if the user actively requested sending this notification,
   *        false if it's being generated by an event
   */
  void sendNotification(byte[] payload, Object target, NotificationCallback callback,
      boolean isForeground);

  /**
   * Returns a set of targets that notifications should be sent to.
   * {@link #sendNotification} will be called once (in parallel) for each target returned.
   */
  Iterable<? extends Object> getTargets();

  /**
   * @return the name of this notification method
   */
  String getName();

  /**
   * @return whether this method is enabled and should be called
   */
  boolean isEnabled();
}
