/*
 * This file is part of EasyRPG Player.
 *
 * EasyRPG Player is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * EasyRPG Player is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with EasyRPG Player. If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef EP_GAME_PICTURE_H
#define EP_GAME_PICTURE_H

// Headers
#include <string>
#include "async_handler.h"
#include "rpg_save.h"
#include "sprite.h"

class Sprite;
class Scene;

/**
 * Pictures class.
 */
class Game_Pictures {
public:
	Game_Pictures() = default;

	void SetSaveData(std::vector<RPG::SavePicture> save);
	std::vector<RPG::SavePicture> GetSaveData() const;

	void InitGraphics();

	static int GetDefaultNumberOfPictures();

	struct Params {
		int position_x;
		int position_y;
		int magnify;
		int top_trans;
		int bottom_trans;
		int red;
		int green;
		int blue;
		int saturation;
		int effect_mode;
		int effect_power;
	};
	struct ShowParams : Params {
		std::string name;
		// RPG Maker 2k3 1.12
		int spritesheet_cols = 1;
		int spritesheet_rows = 1;
		int spritesheet_frame = 0;
		int spritesheet_speed = 0;
		int map_layer = 7;
		int battle_layer = 0;
		// erase_on_map_change | affected_by_flash | affected_by_shake
		int flags = 1 | 32 | 64;
		bool spritesheet_play_once = false;
		bool use_transparent_color = false;
		bool fixed_to_map = false;
	};

	struct MoveParams : Params {
		int duration;
	};

	void Show(int id, const ShowParams& params);
	void Move(int id, const MoveParams& params);
	void Erase(int id);

	void Update(bool is_battle);
	void UpdateGraphics(bool is_battle);

	void OnMapChange();
	void OnBattleEnd();

private:
	struct Picture {
		Picture(int id) { data.ID = id; }
		Picture(RPG::SavePicture data);

		RPG::SavePicture data;
		std::unique_ptr<Sprite> sprite;
		FileRequestBinding request_id;
		int last_spritesheet_frame = 0;
		bool needs_update = false;

		void Update(bool is_battle);
		void UpdateGraphics(bool is_battle);

		bool IsOnMap() const;
		bool IsOnBattle() const;
		int NumSpriteSheetFrames() const;

		void SetNonEffectParams(const Params& params, bool set_positions);
		void SyncCurrentToFinish();

		bool Show(const ShowParams& params);
		void Move(const MoveParams& params);
		void Erase();

		void OnPictureSpriteReady();
	};

	Picture& GetPicture(int id);
	Picture* GetPicturePtr(int id);
	void RequestPictureSprite(Picture& pic);
	void OnPictureSpriteReady(int id);

	std::vector<Picture> pictures;
	int frame_counter = 0;
};

inline bool Game_Pictures::Picture::IsOnMap() const {
	return data.map_layer > 0;
}

inline bool Game_Pictures::Picture::IsOnBattle() const {
	return data.battle_layer > 0;
}

#endif
