/*
 * This file is part of EasyRPG Player.
 *
 * EasyRPG Player is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * EasyRPG Player is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with EasyRPG Player. If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef EP_INPUT_SOURCE_H
#define EP_INPUT_SOURCE_H

#include <bitset>
#include <fstream>
#include <memory>
#include "input_buttons.h"

namespace Input {
	/**
	 * A source for button presses.
	 */
	class Source {
	public:
		Source() = default;

		virtual ~Source() = default;

		/** Called once each logical frame to update pressed_buttons. */
		virtual void Update() = 0;

		/** Called once each physical frame when no logical frames occured to update pressed_buttons for system buttons. */
		virtual void UpdateSystem() = 0;

		const std::bitset<BUTTON_COUNT>& GetPressedButtons() const {
			return pressed_buttons;
		}
		std::bitset<BUTTON_COUNT> GetPressedNonSystemButtons() const {
			auto pressed = pressed_buttons;
			for(unsigned i = 0; i < BUTTON_COUNT; ++i) {
				if (IsSystemButton(static_cast<InputButton>(i))) {
					pressed[i] = false;
				}
			}
			return pressed;
		}

	protected:
		std::bitset<BUTTON_COUNT> pressed_buttons;
	};

	/**
	 * Source that maps key presses from the UI to button
	 * presses.
	 */
	class UiSource : public Source {
	public:
		UiSource() = default;
		~UiSource() override = default;

		void Update() override;
		void UpdateSystem() override;

		// NOTE: buttons/dir_buttons/InitButtons could be moved here
	private:
		void DoUpdate(bool system_only);
	};

	/**
	 * Source that replays button presses from a log file.
	 */
	class LogSource : public Source {
	public:
		LogSource(const char* log_path);
		~LogSource() override = default;

		void Update() override;
		void UpdateSystem() override;

		operator bool() const { return bool(log_file); }
	private:
		std::ifstream log_file;
	};

	extern std::unique_ptr<Source> source;
}

#endif
