/*
camera.cpp
Copyright (C) 2010-2013 celeron55, Perttu Ahola <celeron55@gmail.com>
*/

/*
This file is part of Freeminer.

Freeminer is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Freeminer  is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Freeminer.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "camera.h"
#include "debug.h"
#include "client.h"
#include "map.h"
#include "clientmap.h"     // MapDrawControl
#include "player.h"
#include <cmath>
#include "settings.h"
#include "wieldmesh.h"
#include "noise.h"         // easeCurve
#include "gamedef.h"
#include "sound.h"
#include "event.h"
#include "profiler.h"
#include "util/numeric.h"
#include "util/mathconstants.h"
#include "constants.h"
#include "fontengine.h"

#define CAMERA_OFFSET_STEP 1000

#include "nodedef.h"
#include "game.h" // CameraModes

#include "nodedef.h"
#include "log_types.h"

Camera::Camera(scene::ISceneManager* smgr, MapDrawControl& draw_control,
		IGameDef *gamedef):
	m_playernode(NULL),
	m_headnode(NULL),
	m_cameranode(NULL),

	m_wieldmgr(NULL),
	m_wieldlight(0),
	m_wieldlight_add(0),
	m_wieldnode(NULL),

	m_draw_control(draw_control),
	m_gamedef(gamedef),

	m_camera_position(0,0,0),
	m_camera_direction(0,0,0),
	m_camera_offset(0,0,0),

	m_aspect(1.0),
	m_fov_x(1.0),
	m_fov_y(1.0),

	m_view_bobbing_anim(0),
	m_view_bobbing_state(0),
	m_view_bobbing_speed(0),
	m_view_bobbing_fall(0),

	m_digging_anim(0),
	m_digging_button(-1),

	m_wield_change_timer(0.125),
	m_wield_item_next(),

	m_camera_mode(CAMERA_MODE_FIRST)
{
	//dstream<<FUNCTION_NAME<<std::endl;

	m_driver = smgr->getVideoDriver();
	// note: making the camera node a child of the player node
	// would lead to unexpected behaviour, so we don't do that.
	m_playernode = smgr->addEmptySceneNode(smgr->getRootSceneNode());
	m_headnode = smgr->addEmptySceneNode(m_playernode);
	m_cameranode = smgr->addCameraSceneNode(smgr->getRootSceneNode());
	m_cameranode->bindTargetAndRotation(true);

	// This needs to be in its own scene manager. It is drawn after
	// all other 3D scene nodes and before the GUI.
	m_wieldmgr = smgr->createNewSceneManager();
	m_wieldmgr->addCameraSceneNode();
	m_wieldnode = new WieldMeshSceneNode(m_wieldmgr->getRootSceneNode(), m_wieldmgr, -1, false);
	m_wieldnode->setItem(ItemStack(), m_gamedef);
	m_wieldnode->drop(); // m_wieldmgr grabbed it

	/* TODO: Add a callback function so these can be updated when a setting
	 *       changes.  At this point in time it doesn't matter (e.g. /set
	 *       is documented to change server settings only)
	 *
	 * TODO: Local caching of settings is not optimal and should at some stage
	 *       be updated to use a global settings object for getting thse values
	 *       (as opposed to the this local caching). This can be addressed in
	 *       a later release.
	 */

	m_cache_movement_fov        = g_settings->getBool("movement_fov");

	m_cache_fall_bobbing_amount = g_settings->getFloat("fall_bobbing_amount");
	m_cache_view_bobbing_amount = g_settings->getFloat("view_bobbing_amount");
	m_cache_wanted_fps          = g_settings->getFloat("wanted_fps");
	m_cache_fov                 = g_settings->getFloat("fov");
	m_cache_view_bobbing        = g_settings->getBool("view_bobbing");
	m_nametags.clear();
}

Camera::~Camera()
{
	m_wieldmgr->drop();
}

bool Camera::successfullyCreated(std::string &error_message)
{
	if (!m_playernode) {
		error_message = "Failed to create the player scene node";
	} else if (!m_headnode) {
		error_message = "Failed to create the head scene node";
	} else if (!m_cameranode) {
		error_message = "Failed to create the camera scene node";
	} else if (!m_wieldmgr) {
		error_message = "Failed to create the wielded item scene manager";
	} else if (!m_wieldnode) {
		error_message = "Failed to create the wielded item scene node";
	} else {
		error_message.clear();
	}
	return error_message.empty();
}

// Returns the fractional part of x
inline f32 my_modf(f32 x)
{
	double dummy;
	return modf(x, &dummy);
}

void Camera::step(f32 dtime)
{
	if(m_view_bobbing_fall > 0)
	{
		m_view_bobbing_fall -= 3 * dtime;
		if(m_view_bobbing_fall <= 0)
			m_view_bobbing_fall = -1; // Mark the effect as finished
	}

	bool was_under_zero = m_wield_change_timer < 0;
	m_wield_change_timer = MYMIN(m_wield_change_timer + dtime, 0.125);

	if (m_wield_change_timer >= 0 && was_under_zero)
		m_wieldnode->setItem(m_wield_item_next, m_gamedef);

	if (m_view_bobbing_state != 0)
	{
		//f32 offset = dtime * m_view_bobbing_speed * 0.035;
		f32 offset = dtime * m_view_bobbing_speed * 0.030;
		if (m_view_bobbing_state == 2) {
			// Animation is getting turned off
			if (m_view_bobbing_anim < 0.25) {
				m_view_bobbing_anim -= offset;
			} else if (m_view_bobbing_anim > 0.75) {
				m_view_bobbing_anim += offset;
			}

			if (m_view_bobbing_anim < 0.5) {
				m_view_bobbing_anim += offset;
				if (m_view_bobbing_anim > 0.5)
					m_view_bobbing_anim = 0.5;
			} else {
				m_view_bobbing_anim -= offset;
				if (m_view_bobbing_anim < 0.5)
					m_view_bobbing_anim = 0.5;
			}

			if (m_view_bobbing_anim <= 0 || m_view_bobbing_anim >= 1 ||
					fabs(m_view_bobbing_anim - 0.5) < 0.01) {
				m_view_bobbing_anim = 0;
				m_view_bobbing_state = 0;
			}
		}
		else {
			float was = m_view_bobbing_anim;
			m_view_bobbing_anim = my_modf(m_view_bobbing_anim + offset);
			bool step = (was == 0 ||
					(was < 0.5f && m_view_bobbing_anim >= 0.5f) ||
					(was > 0.5f && m_view_bobbing_anim <= 0.5f));
			if(step) {
				MtEvent *e = new SimpleTriggerEvent("ViewBobbingStep");
				m_gamedef->event()->put(e);
			}
		}
	}

	if (m_digging_button != -1)
	{
		f32 offset = dtime * 3.5;
		float m_digging_anim_was = m_digging_anim;
		m_digging_anim += offset;
		if (m_digging_anim >= 1)
		{
			m_digging_anim = 0;
			m_digging_button = -1;
		}
		float lim = 0.15;
		if(m_digging_anim_was < lim && m_digging_anim >= lim)
		{
			if(m_digging_button == 0)
			{
				MtEvent *e = new SimpleTriggerEvent("CameraPunchLeft");
				m_gamedef->event()->put(e);
			} else if(m_digging_button == 1) {
				MtEvent *e = new SimpleTriggerEvent("CameraPunchRight");
				m_gamedef->event()->put(e);
			}
		}
	}
}

void Camera::update(LocalPlayer* player, f32 frametime, f32 busytime,
		f32 tool_reload_ratio, ClientEnvironment &c_env)
{
	// Get player position
	// Smooth the movement when walking up stairs
	v3f old_player_position = m_playernode->getPosition();
	v3f player_position = player->getPosition();
	if (player->isAttached && player->parent)
		player_position = player->parent->getPosition();
	//if(player->touching_ground && player_position.Y > old_player_position.Y)
	if(player->touching_ground &&
			player_position.Y > old_player_position.Y)
	{
		f32 oldy = old_player_position.Y;
		f32 newy = player_position.Y;
		f32 t = exp(-23*frametime);
		player_position.Y = oldy * t + newy * (1-t);
	}

	// Set player node transformation
	m_playernode->setPosition(player_position);
	m_playernode->setRotation(v3f(0, -1 * player->getYaw(), 0));
	m_playernode->updateAbsolutePosition();

	// Get camera tilt timer (hurt animation)
	float cameratilt = fabs(fabs(player->hurt_tilt_timer-0.75)-0.75);

	// Fall bobbing animation
	float fall_bobbing = 0;
	if(player->camera_impact >= 1 && m_camera_mode < CAMERA_MODE_THIRD)
	{
		if(m_view_bobbing_fall == -1) // Effect took place and has finished
			player->camera_impact = m_view_bobbing_fall = 0;
		else if(m_view_bobbing_fall == 0) // Initialize effect
			m_view_bobbing_fall = 1;

		// Convert 0 -> 1 to 0 -> 1 -> 0
		fall_bobbing = m_view_bobbing_fall < 0.5 ? m_view_bobbing_fall * 2 : -(m_view_bobbing_fall - 0.5) * 2 + 1;
		// Smoothen and invert the above
		fall_bobbing = sin(fall_bobbing * 0.5 * M_PI) * -1;
		// Amplify according to the intensity of the impact
		if (player->camera_impact)
		fall_bobbing *= (1 - rangelim(50 / player->camera_impact, 0, 1)) * 5;

		fall_bobbing *= m_cache_fall_bobbing_amount;
	}

	// Calculate players eye offset for different camera modes
	v3f PlayerEyeOffset = player->getEyeOffset();
	if (m_camera_mode == CAMERA_MODE_FIRST)
		PlayerEyeOffset += player->eye_offset_first;
	else
		PlayerEyeOffset += player->eye_offset_third;

	// Set head node transformation
	m_headnode->setPosition(PlayerEyeOffset+v3f(0,cameratilt*-player->hurt_tilt_strength+fall_bobbing,0));
	m_headnode->setRotation(v3f(player->getPitch(), 0, cameratilt*player->hurt_tilt_strength));
	m_headnode->updateAbsolutePosition();

	// Compute relative camera position and target
	v3f rel_cam_pos = v3f(0,0,0);
	v3f rel_cam_target = v3f(0,0,1);
	v3f rel_cam_up = v3f(0,1,0);

	if (m_view_bobbing_anim != 0 && m_camera_mode < CAMERA_MODE_THIRD)
	{
		f32 bobfrac = my_modf(m_view_bobbing_anim * 2);
		f32 bobdir = (m_view_bobbing_anim < 0.5) ? 1.0 : -1.0;

		#if 1
		f32 bobknob = 1.2;
		f32 bobtmp = sin(pow(bobfrac, bobknob) * M_PI);
		//f32 bobtmp2 = cos(pow(bobfrac, bobknob) * M_PI);

		v3f bobvec = v3f(
			0.3 * bobdir * sin(bobfrac * M_PI),
			-0.28 * bobtmp * bobtmp,
			0.);

		//rel_cam_pos += 0.2 * bobvec;
		//rel_cam_target += 0.03 * bobvec;
		//rel_cam_up.rotateXYBy(0.02 * bobdir * bobtmp * M_PI);
		float f = 1.0;
		f *= m_cache_view_bobbing_amount;
		rel_cam_pos += bobvec * f;
		//rel_cam_target += 0.995 * bobvec * f;
		rel_cam_target += bobvec * f;
		rel_cam_target.Z -= 0.005 * bobvec.Z * f;
		//rel_cam_target.X -= 0.005 * bobvec.X * f;
		//rel_cam_target.Y -= 0.005 * bobvec.Y * f;
		rel_cam_up.rotateXYBy(-0.03 * bobdir * bobtmp * M_PI * f);
		#else
		f32 angle_deg = 1 * bobdir * sin(bobfrac * M_PI);
		f32 angle_rad = angle_deg * M_PI / 180;
		f32 r = 0.05;
		v3f off = v3f(
			r * sin(angle_rad),
			r * (cos(angle_rad) - 1),
			0);
		rel_cam_pos += off;
		//rel_cam_target += off;
		rel_cam_up.rotateXYBy(angle_deg);
		#endif

	}

	// Compute absolute camera position and target
	m_headnode->getAbsoluteTransformation().transformVect(m_camera_position, rel_cam_pos);
	m_headnode->getAbsoluteTransformation().rotateVect(m_camera_direction, rel_cam_target - rel_cam_pos);

	v3f abs_cam_up;
	m_headnode->getAbsoluteTransformation().rotateVect(abs_cam_up, rel_cam_up);

	// Seperate camera position for calculation
	v3f my_cp = m_camera_position;

	// Reposition the camera for third person view
	if (m_camera_mode > CAMERA_MODE_FIRST)
	{
		if (m_camera_mode == CAMERA_MODE_THIRD_FRONT)
			m_camera_direction *= -1;

		my_cp.Y += 2;

		// Calculate new position
		bool abort = false;
		for (int i = BS; i <= BS*2.75; i++)
		{
			my_cp.X = m_camera_position.X + m_camera_direction.X*-i;
			my_cp.Z = m_camera_position.Z + m_camera_direction.Z*-i;
			if (i > 12)
				my_cp.Y = m_camera_position.Y + (m_camera_direction.Y*-i);

			// Prevent camera positioned inside nodes
			INodeDefManager *nodemgr = m_gamedef->ndef();
			MapNode n = c_env.getClientMap().getNodeNoEx(floatToInt(my_cp, BS));
			const ContentFeatures& features = nodemgr->get(n);
			if(features.walkable)
			{
				my_cp.X += m_camera_direction.X*-1*-BS/2;
				my_cp.Z += m_camera_direction.Z*-1*-BS/2;
				my_cp.Y += m_camera_direction.Y*-1*-BS/2;
				abort = true;
				break;
			}
		}

		// If node blocks camera position don't move y to heigh
		if (abort && my_cp.Y > player_position.Y+BS*2)
			my_cp.Y = player_position.Y+BS*2;
	}

	// Update offset if too far away from the center of the map
	m_camera_offset.X += CAMERA_OFFSET_STEP*
			(((s16)(my_cp.X/BS) - m_camera_offset.X)/CAMERA_OFFSET_STEP);
	m_camera_offset.Y += CAMERA_OFFSET_STEP*
			(((s16)(my_cp.Y/BS) - m_camera_offset.Y)/CAMERA_OFFSET_STEP);
	m_camera_offset.Z += CAMERA_OFFSET_STEP*
			(((s16)(my_cp.Z/BS) - m_camera_offset.Z)/CAMERA_OFFSET_STEP);

	// Set camera node transformation
	m_cameranode->setPosition(my_cp-intToFloat(m_camera_offset, BS));
	m_cameranode->setUpVector(abs_cam_up);
	// *100.0 helps in large map coordinates
	m_cameranode->setTarget(my_cp-intToFloat(m_camera_offset, BS) + 100 * m_camera_direction);

	// update the camera position in front-view mode to render blocks behind player
	if (m_camera_mode == CAMERA_MODE_THIRD_FRONT)
		m_camera_position = my_cp;

	// Get FOV
	if (player->zoom) {
		m_wieldnode->setVisible(false);
	} else {
		m_wieldnode->setVisible(true);
	}
	f32 fov_degrees = m_draw_control.fov;

	// Greater FOV if running
	v3f speed = player->getSpeed();

	if (m_cache_movement_fov) {
		auto fov_was = m_draw_control.fov_add;
		m_draw_control.fov_add = speed.dotProduct(m_camera_direction)/(BS*4);
		if (m_draw_control.fov_add > fov_was + 1)
			m_draw_control.fov_add = fov_was + ( m_draw_control.fov_add - fov_was) / 3;
		else if (m_draw_control.fov_add < fov_was - 1)
			m_draw_control.fov_add = fov_was - (fov_was - m_draw_control.fov_add) / 3;
		fov_degrees -= m_draw_control.fov_add;
	}

	fov_degrees = MYMAX(MYMIN(fov_degrees, 150.0), 10.0);

	// FOV and aspect ratio
	m_aspect = (f32) porting::getWindowSize().X / (f32) porting::getWindowSize().Y;
	m_fov_y = fov_degrees * M_PI / 180.0;
	// Increase vertical FOV on lower aspect ratios (<16:10)
	m_fov_y *= MYMAX(1.0, MYMIN(1.4, sqrt(16./10. / m_aspect)));
	m_fov_x = 2 * atan(m_aspect * tan(0.5 * m_fov_y));
	m_cameranode->setAspectRatio(m_aspect);
	m_cameranode->setFOV(m_fov_y);

	// Position the wielded item
	//v3f wield_position = v3f(45, -35, 65);
	v3f wield_position = v3f(55, -35, 65);
	//v3f wield_rotation = v3f(-100, 120, -100);
	v3f wield_rotation = v3f(-100, 120, -100);
	wield_position.Y += fabs(m_wield_change_timer)*320 - 40;
	if(m_digging_anim < 0.05 || m_digging_anim > 0.5)
	{
		f32 frac = 1.0;
		if(m_digging_anim > 0.5)
			frac = 2.0 * (m_digging_anim - 0.5);
		// This value starts from 1 and settles to 0
		f32 ratiothing = pow((1.0f - tool_reload_ratio), 0.5f);
		//f32 ratiothing2 = pow(ratiothing, 0.5f);
		f32 ratiothing2 = (easeCurve(ratiothing*0.5))*2.0;
		wield_position.Y -= frac * 25.0 * pow(ratiothing2, 1.7f);
		//wield_position.Z += frac * 5.0 * ratiothing2;
		wield_position.X -= frac * 35.0 * pow(ratiothing2, 1.1f);
		wield_rotation.Y += frac * 70.0 * pow(ratiothing2, 1.4f);
		//wield_rotation.X -= frac * 15.0 * pow(ratiothing2, 1.4f);
		//wield_rotation.Z += frac * 15.0 * pow(ratiothing2, 1.0f);
	}
	if (m_digging_button != -1)
	{
		f32 digfrac = m_digging_anim;
		wield_position.X -= 50 * sin(pow(digfrac, 0.8f) * M_PI);
		wield_position.Y += 24 * sin(digfrac * 1.8 * M_PI);
		wield_position.Z += 25 * 0.5;

		// Euler angles are PURE EVIL, so why not use quaternions?
		core::quaternion quat_begin(wield_rotation * core::DEGTORAD);
		core::quaternion quat_end(v3f(80, 30, 100) * core::DEGTORAD);
		core::quaternion quat_slerp;
		quat_slerp.slerp(quat_begin, quat_end, sin(digfrac * M_PI));
		quat_slerp.toEuler(wield_rotation);
		wield_rotation *= core::RADTODEG;
	} else {
		f32 bobfrac = my_modf(m_view_bobbing_anim);
		wield_position.X -= sin(bobfrac*M_PI*2.0) * 3.0;
		wield_position.Y += sin(my_modf(bobfrac*2.0)*M_PI) * 3.0;
	}
	m_wieldnode->setPosition(wield_position);
	m_wieldnode->setRotation(wield_rotation);

	m_wieldnode->setColor(player->light_color);

	// Set render distance
	updateViewingRange();

	// If the player is walking, swimming, or climbing,
	// view bobbing is enabled and free_move is off,
	// start (or continue) the view bobbing animation.
	const bool movement_XZ = hypot(speed.X, speed.Z) > BS;
	const bool movement_Y = fabs(speed.Y) > BS;

	const bool walking = movement_XZ && player->touching_ground;
	const bool swimming = (movement_XZ || player->swimming_vertical) && player->in_liquid;
	const bool climbing = movement_Y && player->is_climbing;
	if ((walking || swimming || climbing) &&
			m_cache_view_bobbing &&
			(!g_settings->getBool("free_move") || !m_gamedef->checkLocalPrivilege("fly")))
	{
		// Start animation
		m_view_bobbing_state = 1;
		m_view_bobbing_speed = MYMIN(speed.getLength(), 40);
	}
	else if (m_view_bobbing_state == 1)
	{
		// Stop animation
		m_view_bobbing_state = 2;
		m_view_bobbing_speed = 60;
	}
}

void Camera::updateViewingRange()
{
	if (m_draw_control.range_all) {
		m_cameranode->setFarValue(100000.0);
		return;
	}

//fm dynamic range:
/*
	m_added_busytime += busytime_in;
	m_added_frames += 1;

	m_frametime_counter -= frametime_in;
	if (m_frametime_counter > 0)
		return;
	m_frametime_counter = 0.2; // Same as ClientMap::updateDrawList interval
*/
	/*dstream<<FUNCTION_NAME
			<<": Collected "<<m_added_frames<<" frames, total of "
			<<m_added_busytime<<"s."<<std::endl;

	dstream<<"m_draw_control.blocks_drawn="
			<<m_draw_control.blocks_drawn
			<<", m_draw_control.blocks_would_have_drawn="
			<<m_draw_control.blocks_would_have_drawn
			<<std::endl;*/

	// Get current viewing range and FPS settings
	f32 viewing_range_min = g_settings->getFloat("viewing_range");
	viewing_range_min = MYMAX(15.0, viewing_range_min);

	f32 viewing_range_max = g_settings->getFloat("viewing_range_max");
	viewing_range_max = MYMAX(viewing_range_min, viewing_range_max);
	// vrange+position must be smaller than 32767
	viewing_range_max = MYMIN(viewing_range_max, 32760 - MYMAX(MYMAX(std::abs(m_camera_position.X/BS), std::abs(m_camera_position.Y/BS)), std::abs(m_camera_position.Z/BS)));

	f32 wanted_fps = m_cache_wanted_fps;
	wanted_fps = MYMAX(wanted_fps, 1.0);

	// todo: remake
	if (m_draw_control.fps > wanted_fps && m_draw_control.fps_avg > wanted_fps * 1.2) {
		m_draw_control.wanted_range += 1;
	} else if (m_draw_control.fps_avg < wanted_fps) {
		if (m_draw_control.fps < wanted_fps * 0.7) {
			m_draw_control.wanted_range *= 0.9;
		} else if (m_draw_control.fps < wanted_fps) {
			m_draw_control.wanted_range -= 1;
		}
	} else if (m_draw_control.fps < wanted_fps * 0.7) {
		m_draw_control.wanted_range *= 0.9;
	}

	// Immediately apply hard limits
	if(m_draw_control.wanted_range < viewing_range_min)
		m_draw_control.wanted_range = viewing_range_min;
	if(m_draw_control.wanted_range > viewing_range_max)
		m_draw_control.wanted_range = viewing_range_max;

	int farmesh = g_settings->getS32("farmesh");
	int farmesh_step = g_settings->getS32("farmesh_step");
	int farmesh_wanted = g_settings->getS32("farmesh_wanted");

#if OOOOOOOOOOOOOOOld
	f32 wanted_frametime = 1.0 / wanted_fps;

	//m_draw_control.wanted_min_range = viewing_range_min;
	//m_draw_control.wanted_max_blocks = (2.0*m_draw_control.blocks_would_have_drawn)+1;
	//if (m_draw_control.wanted_max_blocks < 10)
		//m_draw_control.wanted_max_blocks = 10;

/*
	f32 block_draw_ratio = 1.0;
	if (m_draw_control.blocks_would_have_drawn != 0)
	{
		block_draw_ratio = (f32)m_draw_control.blocks_drawn
			/ (f32)m_draw_control.blocks_would_have_drawn;
	}
*/

	// Calculate the average frametime in the case that all wanted
	// blocks had been drawn
	f32 frametime = m_added_busytime / m_added_frames /* / block_draw_ratio */ ;

	m_added_busytime = 0.0;
	m_added_frames = 0;

	f32 wanted_frametime_change = wanted_frametime - frametime;
	//dstream<<"wanted_frametime_change="<<wanted_frametime_change<<std::endl;
	//g_profiler->avg("wanted_frametime_change", wanted_frametime_change);
#endif

	static int framecnt = 0;
	m_draw_control.fps_wanted = wanted_fps;
	if (farmesh) {
			//infostream<<" m_draw_control.fps="<<m_draw_control.fps<< " wanted_fps="<< wanted_fps << " m_draw_control.fps_avg="<< m_draw_control.fps_avg <<" wanted_fps*1.4="<< wanted_fps*1.4 /*<<" block_draw_ratio="<<block_draw_ratio */<< " wanted_frametime="<< wanted_frametime <<" .blocks_would_have_drawn=" <<m_draw_control.blocks_would_have_drawn <<" .blocks_drawn=" <<m_draw_control.blocks_drawn <<std::endl;
			if (m_draw_control.fps > wanted_fps && m_draw_control.fps_avg >= wanted_fps*1.3) {
				if (++framecnt > m_draw_control.fps_avg/2) {
					if (m_draw_control.wanted_range >= farmesh_wanted) {
						m_draw_control.farmesh = (int)m_draw_control.farmesh + 1;
						framecnt = 0;
					}
					if (m_draw_control.farmesh >= farmesh*1.3 && m_draw_control.farmesh_step < farmesh_step) {
						++m_draw_control.farmesh_step;
						framecnt = 0;
					}
				}
			} else if (m_draw_control.fps <= wanted_fps*0.8){
				float farmesh_was = m_draw_control.farmesh;
				if (m_draw_control.fps <= wanted_fps*0.6)
					m_draw_control.farmesh = farmesh;
				else if (m_draw_control.fps <= wanted_fps*0.7)
					m_draw_control.farmesh *= 0.5;
				else if (m_draw_control.farmesh>10)
					m_draw_control.farmesh *= 0.8;
				else
					m_draw_control.farmesh -= 1;
				if (m_draw_control.farmesh < farmesh)
					m_draw_control.farmesh = farmesh;
				if (m_draw_control.farmesh <= farmesh && m_draw_control.farmesh_step > 1 && m_draw_control.fps <= wanted_fps*0.3)
					--m_draw_control.farmesh_step;
				if (farmesh_was != m_draw_control.farmesh)
					return;
			}
	}

#if OOOOOOOOOOOOOOOld
	// If needed frametime change is small, just return
	// This value was 0.4 for many months until 2011-10-18 by c55;
	//if (fabs(wanted_frametime_change) < wanted_frametime*0.33)
	if (wanted_frametime_change > -wanted_frametime*0.33 && wanted_frametime_change < wanted_frametime*0.15)
	{
		//dstream<<"ignoring small wanted_frametime_change"<<std::endl;
		return;
	}

	f32 range = m_draw_control.wanted_range;
	f32 new_range = range;

	f32 d_range = range - m_range_old;
	f32 d_busytime = busytime_in - m_busytime_old;
	if (d_range != 0)
	{
		m_time_per_range = d_busytime / d_range;
	}
	//dstream<<"time_per_range="<<m_time_per_range<<std::endl;
	//g_profiler->avg("time_per_range", m_time_per_range);

	// The minimum allowed calculated frametime-range derivative:
	// Practically this sets the maximum speed of changing the range.
	// The lower this value, the higher the maximum changing speed.
	// A low value here results in wobbly range (0.001)
	// A low value can cause oscillation in very nonlinear time/range curves.
	// A high value here results in slow changing range (0.0025)
	// SUGG: This could be dynamically adjusted so that when
	//       the camera is turning, this is lower
	//f32 min_time_per_range = 0.0010; // Up to 0.4.7
	f32 min_time_per_range = 0.0005;
	if(m_time_per_range < min_time_per_range)
	{
		m_time_per_range = min_time_per_range;
		//dstream<<"m_time_per_range="<<m_time_per_range<<" (min)"<<std::endl;
	}
	else
	{
		//dstream<<"m_time_per_range="<<m_time_per_range<<std::endl;
	}

	f32 wanted_range_change = wanted_frametime_change / m_time_per_range;
	// Dampen the change a bit to kill oscillations
	//wanted_range_change *= 0.9;
	//wanted_range_change *= 0.75;
	wanted_range_change *= 0.5;
	if (wanted_range_change > 1)
		wanted_range_change *= 0.4;
	//dstream<<"wanted_range_change="<<wanted_range_change<<std::endl;

	//infostream<< " wanted_range_change=" << wanted_range_change <<" m_time_per_range="<<m_time_per_range << " wanted_frametime_change="<<wanted_frametime_change<< std::endl;

	// If needed range change is very small, just return
	if(fabs(wanted_range_change) < 0.001)
	{
		//dstream<<"ignoring small wanted_range_change"<<std::endl;
		return;
	}

	new_range += wanted_range_change;

	//f32 new_range_unclamped = new_range;
	new_range = MYMAX(new_range, viewing_range_min);
	new_range = MYMIN(new_range, viewing_range_max);
	/ * dstream<<"new_range="<<new_range_unclamped
			<<", clamped to "<<new_range<<std::endl;* /

	m_range_old = m_draw_control.wanted_range;
	m_busytime_old = busytime_in;

	m_draw_control.wanted_range = new_range;
#endif

	g_profiler->add("CM: wanted_range", m_draw_control.wanted_range);

	const auto viewing_range = m_draw_control.wanted_range;
/* mt static range:
	f32 viewing_range = g_settings->getFloat("viewing_range");
	m_draw_control.wanted_range = viewing_range;
*/
	m_cameranode->setFarValue((viewing_range < 2000) ? 2000 * BS : viewing_range * BS);
}

void Camera::setDigging(s32 button)
{
	if (m_digging_button == -1)
		m_digging_button = button;
}

void Camera::wield(const ItemStack &item)
{
	if (item.name != m_wield_item_next.name) {
		m_wield_item_next = item;
		if (m_wield_change_timer > 0)
			m_wield_change_timer = -m_wield_change_timer;
		else if (m_wield_change_timer == 0)
			m_wield_change_timer = -0.001;
	}
	IItemDefManager *idef = m_gamedef->idef();
	std::string itemname = item.getDefinition(idef).name;
	m_wieldlight_add = ((ItemGroupList)idef->get(itemname).groups)["wield_light"]*200/14;
}

void Camera::drawWieldedTool(irr::core::matrix4* translation)
{
	// Set vertex colors of wield mesh according to light level
	u8 li = m_wieldlight;
	if (g_settings->getBool("enable_shaders"))
	{
		if (li+m_wieldlight_add < 200)
			li += m_wieldlight_add;
		else
			li = 200;
	}
	video::SColor color(255,li,li,li);

	// Clear Z buffer so that the wielded tool stay in front of world geometry
	m_wieldmgr->getVideoDriver()->clearZBuffer();

	// Draw the wielded node (in a separate scene manager)
	scene::ICameraSceneNode* cam = m_wieldmgr->getActiveCamera();
	cam->setAspectRatio(m_cameranode->getAspectRatio());
	cam->setFOV(72.0*M_PI/180.0);
	cam->setNearValue(10);
	cam->setFarValue(1000);
	if (translation != NULL)
	{
		irr::core::matrix4 startMatrix = cam->getAbsoluteTransformation();
		irr::core::vector3df focusPoint = (cam->getTarget()
				- cam->getAbsolutePosition()).setLength(1)
				+ cam->getAbsolutePosition();

		irr::core::vector3df camera_pos =
				(startMatrix * *translation).getTranslation();
		cam->setPosition(camera_pos);
		cam->setTarget(focusPoint);
	}
	m_wieldmgr->drawAll();
}

void Camera::drawNametags()
{
	core::matrix4 trans = m_cameranode->getProjectionMatrix();
	trans *= m_cameranode->getViewMatrix();

	for (std::list<Nametag *>::const_iterator
			i = m_nametags.begin();
			i != m_nametags.end(); ++i) {
		Nametag *nametag = *i;
		if (nametag->nametag_color.getAlpha() == 0) {
			// Enforce hiding nametag,
			// because if freetype is enabled, a grey
			// shadow can remain.
			continue;
		}
		v3f pos = nametag->parent_node->getPosition() + v3f(0.0, 1.1 * BS, 0.0);
		f32 transformed_pos[4] = { pos.X, pos.Y, pos.Z, 1.0f };
		trans.multiplyWith1x4Matrix(transformed_pos);
		if (transformed_pos[3] > 0) {
			core::dimension2d<u32> textsize =
				g_fontengine->getFont()->getDimension(
				utf8_to_wide(nametag->nametag_text).c_str());
			f32 zDiv = transformed_pos[3] == 0.0f ? 1.0f :
				core::reciprocal(transformed_pos[3]);
			v2u32 screensize = m_driver->getScreenSize();
			v2s32 screen_pos;
			screen_pos.X = screensize.X *
				(0.5 * transformed_pos[0] * zDiv + 0.5) - textsize.Width / 2;
			screen_pos.Y = screensize.Y *
				(0.5 - transformed_pos[1] * zDiv * 0.5) - textsize.Height / 2;
			core::rect<s32> size(0, 0, textsize.Width, textsize.Height);
			g_fontengine->getFont()->draw(utf8_to_wide(nametag->nametag_text).c_str(),
					size + screen_pos, nametag->nametag_color);
		}
	}
}

Nametag *Camera::addNametag(scene::ISceneNode *parent_node,
		std::string nametag_text, video::SColor nametag_color)
{

	auto nametag_text_wide = utf8_to_wide(nametag_text);
	if (nametag_text_wide.size() > 15) {
		nametag_text_wide.resize(15);
		nametag_text_wide += L".";
		nametag_text = wide_to_utf8(nametag_text_wide);
	}

	Nametag *nametag = new Nametag(parent_node, nametag_text, nametag_color);
	m_nametags.push_back(nametag);
	return nametag;
}

void Camera::removeNametag(Nametag *nametag)
{
	m_nametags.remove(nametag);
	delete nametag;
}
