/*
 * Libero Vocab
 *     An app for Android systems which allows to do practice with kvtml
 *     vocabulary files.
 *     This program is a fork of another program called "Vocab Drill" by:
 *       - Károly Kiripolszky <karcsi@ekezet.com>
 *       - Matthias Völlinger <matthias.voellinger@gmx.de>
 *
 *     Copyright (C) 2019, 2020  Lo Iacono Massimo (massimol@inventati.org)
 *
 *     This program is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package org.inventati.massimol.liberovocab.activities.test_types;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Set;

import android.content.Intent;
import android.os.Bundle;
import android.view.KeyEvent;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.View.OnClickListener;
import android.widget.Button;
import android.widget.LinearLayout;
import android.widget.TextView;

import org.inventati.massimol.liberovocab.Config;
import org.inventati.massimol.liberovocab.Question;
import org.inventati.massimol.liberovocab.R;
import org.inventati.massimol.liberovocab.helpers.Gradient;
import org.inventati.massimol.liberovocab.kvtml.Kvtml;

public class FlashcardActivity extends TestActivity
{
	/**
	 * this mode shows both question and answer at the same time; used only for
	 * consistancy
	 */
	final public static String LEARN_MODE = "learnmode";

	/**
	 * this mode only shows the question
	 */
	final public static String VIEW_MODE = "viewmode";

	/**
	 * this mode shows the answer
	 */
	final public static String ANSWER_MODE = "answermode";

	/**
	 * The LineraLayout of the answer:
	 */
	private LinearLayout mLinearLayoutAnswer;

	/**
	 * shows the answer
	 */
	private TextView mAnswerText;

	/**
	 * shows a comment
	 */
	private TextView mCommentText;

	/**
	 * left button: "answer question", "knew the answer" and "previous question"
	 */
	private Button mButtonLeft;

	/**
	 * right button: "answer this question later", "didn't know the answer" and
	 * "next question"
	 */
	private Button mButtonRight;

	/**
	 * for learnmode: position in hashmap.keySet().toArray() to be able to
	 * navigate
	 */
	private int mPosition = 0;
	
	private ArrayList<String> mEntryIds = null;

	@Override
	public void onCreate(Bundle savedInstanceState)
	{
		super.onCreate(savedInstanceState);

        // Get the intent:
		Intent i = getIntent();
		if (i.hasExtra(LEARN_MODE))
			mLearningMode = (Boolean) i.getSerializableExtra(LEARN_MODE);

		mLinearLayoutAnswer = (LinearLayout) findViewById(R.id.LinearLayout_answer);
		mAnswerText = (TextView) findViewById(R.id.text_fc_answer);
		mCommentText = (TextView) findViewById(R.id.text_fc_comment);
		mButtonLeft = (Button) findViewById(R.id.button_fc_left);
		mButtonRight = (Button) findViewById(R.id.button_fc_right);

		if (mLearningMode)
		{
			mButtonLeft.setText(R.string.button_fc_back);
			mButtonLeft.setTag(LEARN_MODE);
			mButtonLeft.setEnabled(false);
			mButtonRight.setText(R.string.button_fc_next);
			mButtonRight.setTag(LEARN_MODE);
			mLinearLayoutAnswer.setVisibility(View.VISIBLE);
			mAnswerText.setVisibility(View.VISIBLE);
			mCommentText.setVisibility(View.VISIBLE);
		}

		// initialize data, has to be done before setting the onClickListener
		restart();

		// left button: "answer question", "knew the answer" and "previous question"
		mButtonLeft.setOnClickListener(new OnClickListener()
		{
			@Override
			public void onClick(View v)
			{
				if (((String) mButtonLeft.getTag()).equals(VIEW_MODE))
				{
					// user tries to answer the question

					// Update the UI:
					setUIForSecondPhase();

					// Update the TextView of the text of question, in the case of it wasn't shown
					// before, and the button for show this text:
					mQuestionText.setText(transQuestion.text);
					mQuestionText.setTextSize(getResources().getDimension(R.dimen.large_text));

					mButtonToShowQuestionText.setEnabled(false);
					mButtonToShowQuestionText.setVisibility(View.GONE);
				}
				else if (((String) mButtonLeft.getTag()).equals(ANSWER_MODE))
				{
					// user knew the answer

					// The previous MediaPlayer object is released:
					if (mediaPlayer != null)
						mediaPlayer.release();

					// Update the UI:
					setUIForFirstPhase();

					if (correct(mQuestion.getSolution()))
						updateQuestion();
				}
				else if (mLearningMode)
				{
					// user wants to see the previous question

					// The previous MediaPlayer object is released:
					if (mediaPlayer != null)
						mediaPlayer.release();

					if (mPosition == 1)
						mButtonLeft.setEnabled(false);
					else if (mPosition == mMaxQuestions - 1)
						mButtonRight.setEnabled(true);

					mPosition--;
					updateQuestion();
				}
			}
		});

		// right button: "answer this question later", "didn't know the answer" and
		// "next question"
		mButtonRight.setOnClickListener(new OnClickListener()
		{
			@Override
			public void onClick(View v)
			{
				if (((String) mButtonRight.getTag()).equals(VIEW_MODE))
				{
					// user wants to answer this question later

					// The previous MediaPlayer object is released:
					if (mediaPlayer != null)
						mediaPlayer.release();

					mQuestionText.setTextSize(getResources().getDimension(R.dimen.large_text));
					mQuestion = mQuestionFactory.create(mQuestionData, mMaxItems);
					updateQuestion();
				}
				else if (((String) mButtonRight.getTag()).equals(ANSWER_MODE))
				{
					// user didn't know the answer

					// The previous MediaPlayer object is released:
					if (mediaPlayer != null)
						mediaPlayer.release();

					// Update the UI:
					setUIForFirstPhase();

					mistake(mQuestion.getSolution());
					updateQuestion();
				}
				else if (mLearningMode)
				{
					// user wants to see the next question

					// The previous MediaPlayer object is released:
					if (mediaPlayer != null)
						mediaPlayer.release();

					if (mPosition == 0)
						mButtonLeft.setEnabled(true);
					else if (mPosition == mMaxQuestions - 2)
						mButtonRight.setEnabled(false);

					mPosition++;
					updateQuestion();
				}
			}
		});

		recolorize();
	}

	@Override
	protected Question getNewQuestion(HashMap<String, Kvtml.Entry> entries, int maxItems)
	{
		if (mLearningMode)
		{
			if (mEntryIds == null)
			{
				// create a sorted array of entry ids for navigation
				Set<String> entryKeySet = entries.keySet();
				mEntryIds = new ArrayList<String>();
				mEntryIds.addAll(entryKeySet);
				Collections.sort(mEntryIds);
			}
			// use an array so we can navigate forwards and backwards
			Kvtml.Entry solution = entries.get(mEntryIds.get(mPosition));
			return new Question(entries, solution, null);
		} else
			return super.getNewQuestion(entries, maxItems);
	}

	@Override
	protected void loadContentView()
	{
		setContentView(R.layout.activity_flashcard);
	}

	@Override
	protected void updateQuestion(boolean configChanged)
	{
		super.updateQuestion(configChanged);
        if (mQuestion == null)
        {
            finish();
            return;
        }

		mAnswerText.setText("");
		mCommentText.setText("");

		Kvtml.Entry solution = mQuestion.getSolution();
		if (solution == null)
		{
			mAnswerText.setText("- - -");
			return;
		}

		Kvtml.Translation answer = solution.translations.get(Config.getAnswerLangId());
		if (answer == null)
		{
			mAnswerText.setText("- - -");
			return;
		}

		// additional views
		mAnswerText.setText(String.valueOf(answer.text));
		mCommentText.setText(String.valueOf(answer.comment));
	}

	/**
	 * Returns the number of pixel corresponding of a number of dp:
	 */
	public int pixelsFromDP(int pixels)
	{
		return (int)(pixels * getResources().getDisplayMetrics().density + 0.5f);
	}

	@Override
	protected String getTitleString()
	{
		Kvtml.Entry solution = mQuestion.getSolution();

		// Translation which regards at the language to learn.
		Kvtml.Translation translationForGrade = solution.translations.get(Config.getLanguageForGradeId());

		if (mLearningMode)
			return String.format("%d/%d - " + getResources().getString(R.string.grade) + ": %d", 1 + mPosition, mMaxQuestions, translationForGrade.grade.currentGrade);
		else
			return String.format("%d/%d - " + getResources().getString(R.string.grade) + ": %d", 1 + (mMaxQuestions - mEntriesLeft), mMaxQuestions, translationForGrade.grade.currentGrade);
	}

	@Override
	public boolean onKeyDown(int keyCode, KeyEvent event)
	{
		return super.onKeyDown(keyCode, event);
	}

	/**
	 * Executed when the test was finished
	 * that is when all questions were tested.
	 */
	@Override
	protected void showFinishDialog()
	{
		super.showFinishDialog();
	}

    /**
     * Resets the test and updates the UI.
     */
    @Override
    protected void restart()
    {
    	// The user interface in the Flashcard type of test, with the exception of the learning
		// mode, knows two phases: the first when the user doesn't see the answer, the second
		// when the user sees the answer and declares whether he or she knew the correct answer.
		// If the restart appends when the user is in the second phase, the restart function must
		// reset the first phase.
        if (!mLearningMode)
    	{
			// Update the UI:
			setUIForFirstPhase();
		}

		super.restart();
    }

	/**
	 * Set the UI for the first phase:
	 */
	private void setUIForFirstPhase()
	{
		mButtonLeft.setTag(VIEW_MODE);
		mButtonLeft.setText(R.string.button_fc_answer);
		mButtonRight.setTag(VIEW_MODE);
		mButtonRight.setText(R.string.button_fc_next);
		mLinearLayoutAnswer.setVisibility(View.GONE);
		mAnswerText.setVisibility(View.GONE);
		mCommentText.setVisibility(View.GONE);
		recolorize();
	}

	/**
	 * Set the UI for the second phase:
	 */
	private void setUIForSecondPhase()
	{
		mButtonLeft.setTag(ANSWER_MODE);
		mButtonLeft.setText(R.string.button_fc_right);
		mButtonRight.setTag(ANSWER_MODE);
		mButtonRight.setText(R.string.button_fc_wrong);
		mLinearLayoutAnswer.setVisibility(View.VISIBLE);
		mAnswerText.setVisibility(View.VISIBLE);
		mCommentText.setVisibility(View.VISIBLE);
		recolorize();
	}

	private void recolorize()
	{
		final ArrayList<View> vs = new ArrayList<View>();
		vs.add(mLangText);
		vs.add(mQuestionText);
		if (mAnswerText.getVisibility() == View.VISIBLE)
			vs.add(mAnswerText);
		if (mCommentText.getVisibility() == View.VISIBLE)
			vs.add(mCommentText);
		vs.add(mButtonLeft);
		vs.add(mButtonRight);
		Gradient.colorize(vs);
	}
}