/*
 * Libero Vocab
 *     An app for Android systems which allows to do practice with kvtml
 *     vocabulary files.
 *     This program is a fork of another program called "Vocab Drill" by:
 *       - Károly Kiripolszky <karcsi@ekezet.com>
 *       - Matthias Völlinger <matthias.voellinger@gmx.de>
 *
 *     Copyright (C) 2019, 2020  Lo Iacono Massimo (massimol@inventati.org)
 *
 *     This program is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package org.inventati.massimol.liberovocab.dialogs;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map.Entry;
import java.util.SortedSet;
import java.util.TreeSet;

import android.support.v7.app.AppCompatActivity;
import android.support.v7.app.ActionBar;

import android.app.AlertDialog;
import android.content.DialogInterface;
import android.content.Intent;
import android.os.Bundle;
import android.text.Editable;
import android.text.Html;
import android.text.TextWatcher;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.WindowManager.LayoutParams;
import android.widget.EditText;
import android.widget.Filter.FilterListener;
import android.widget.ListView;
import android.widget.AdapterView;
import android.widget.TextView;
import android.widget.Toast;

import org.inventati.massimol.liberovocab.Config;
import org.inventati.massimol.liberovocab.R;
import org.inventati.massimol.liberovocab.adapters.EntryListAdapter;
import org.inventati.massimol.liberovocab.kvtml.Kvtml;

public class EntryBrowserDialog extends AppCompatActivity
{
	public static final int SWAP_LANGUAGE_DIALOG = 1000;
	public static final int ANSWER_LANGUAGE_DIALOG = 1001;
	public static final int QUESTION_LANGUAGE_DIALOG = 1002;

	/**
	 * Caller Activity sets this.
	 */
	public static Kvtml data = null;

	/**
	 * Caller Activity sets this.
	 */
	private String mWindowTitle = "";

	private String mDisplayLanguageId = "";
	private EditText mEditTextFilter = null;
	private TextView mResultsNumText = null;
	private static EntryListAdapter sAdapter = null;
	private String mFilterText = null;
	private FilterListener mFilterListener;

	/**
	 * Sort entries alphabetically if True.
	 */
	private boolean mSortAlphabetically = true;

	/**
	 * The ListView object:
	 */
	private ListView list;

	/**
	 * The customized TextView for the ActionBar:
	 */
	private TextView mTextViewForActionBar;

	@Override
	protected void onCreate(Bundle savedInstanceState)
	{
		super.onCreate(savedInstanceState);
		getWindow().setFlags(LayoutParams.FLAG_FULLSCREEN, LayoutParams.FLAG_FULLSCREEN);
		setContentView(R.layout.dialog_entry_browser);

		// This activity has an ActionBar:
		ActionBar actionBar = getSupportActionBar();
		actionBar.setHomeButtonEnabled(true);

		// In order to hide the icon in the ActionBar:
		//actionBar.setIcon(null);
		//actionBar.setIcon(android.R.color.transparent);

		// In order to customize the title in the ActionBar to be multiline and small:
		actionBar.setDisplayShowCustomEnabled(true);
		actionBar.setDisplayShowTitleEnabled(false);

		LayoutInflater inflator = LayoutInflater.from(this);
		View mCustomViewForActionBar = inflator.inflate(R.layout.view_action_bar_multiline_and_small, null);
		mTextViewForActionBar = ((TextView) mCustomViewForActionBar.findViewById(R.id.action_bar_title));
		actionBar.setCustomView(mCustomViewForActionBar);

		// Get the Intent:
		Intent i = getIntent();
		if (i != null)
		{
			mSortAlphabetically = i.getBooleanExtra("doSort", mSortAlphabetically);
			mWindowTitle = i.getStringExtra("windowTitle") != null ? i.getStringExtra("windowTitle") : mWindowTitle;
		}

		// Set both the title of the Activity and the customized text of the ActionBar,
		// although the first isn't shown:
		setTitle(mWindowTitle);
		mTextViewForActionBar.setText(mWindowTitle);

		mDisplayLanguageId = Config.getQuestionLangId();
		if (mDisplayLanguageId == null)
		{
			Toast.makeText(this, R.string.error_wrong_kvtml_format, Toast.LENGTH_LONG).show();
			finish();
			return;
		}

		list = (ListView) findViewById(android.R.id.list);
		list.setTextFilterEnabled(true);

		list.setOnItemClickListener(new AdapterView.OnItemClickListener()
			{
				@Override
				public void onItemClick(AdapterView<?> parent, View view, int position, long id)
				{
					//String item = (String) parent.getItemAtPosition(position);
				}
			}
		);

		// The list of items is prepared:
		loadEntries();

		// The list of items is added to the ListView object:
		list.setAdapter(sAdapter);

		mResultsNumText = (TextView) findViewById(R.id.text_results_num);
		mFilterListener = new FilterListener()
			{
				@Override
				public void onFilterComplete(int count)
				{
					if (mFilterText != null)
					{
						int ct = sAdapter.getCount();
						mResultsNumText.setText(Html.fromHtml(String.format("<b>Results: </b>%d", ct)));
						mResultsNumText.setVisibility(View.VISIBLE);
					}
					else
						mResultsNumText.setVisibility(View.GONE);
				}
			};

		mEditTextFilter = (EditText) findViewById(R.id.text_filter);
		mEditTextFilter.addTextChangedListener(new TextWatcher()
			{
				@Override
				public void onTextChanged(CharSequence s, int start, int before, int count)
				{
					if (s == null || s.toString().trim().length() == 0)
						mFilterText = null;
					else
						mFilterText = s.toString();
					sAdapter.getFilter().filter(mFilterText, mFilterListener);
				}

				@Override
				public void beforeTextChanged(CharSequence s, int start, int count, int after)
				{
				}

				@Override
				public void afterTextChanged(Editable s)
				{
				}
			}
		);
	}

	@Override
	public boolean onCreateOptionsMenu(Menu menu)
	{
		MenuInflater inflater = getMenuInflater();
		inflater.inflate(R.menu.browser_menu, menu);

		// If there is only one language defined in the kvtml object:
		if (Config.lastData.identifiers.size() == 1)
		{
			// Hide the menu for swapping languages:
			MenuItem item = menu.findItem(R.id.option_swap_languages);
			item.setVisible(false);

			// Hide the menus for chaging question language and answer language:
			item = menu.findItem(R.id.option_change_answer_lang);
			item.setVisible(false);
			item = menu.findItem(R.id.option_change_question_lang);
			item.setVisible(false);
		}

		// If there are two or more languages defined in the kvtml object:
		else
		{
			// Show the menu for swapping languages:
			MenuItem item = menu.findItem(R.id.option_swap_languages);
			item.setVisible(true);

			// TODO: if we consider useful this functionality, toggle switch to true the
			//       visibility of these items.
			// Show the menus for chaging question language and answer language:
			item = menu.findItem(R.id.option_change_answer_lang);
			item.setVisible(false);
			item = menu.findItem(R.id.option_change_question_lang);
			item.setVisible(false);
		}

		return super.onCreateOptionsMenu(menu);
	}

	@Override
	public boolean onOptionsItemSelected(MenuItem item)
	{
		switch (item.getItemId())
		{
			case R.id.option_swap_languages:
				prepareLanguageDialogs(SWAP_LANGUAGE_DIALOG);
				return true;

			case R.id.option_change_answer_lang:
				prepareLanguageDialogs(ANSWER_LANGUAGE_DIALOG);
				return true;

			case R.id.option_change_question_lang:
				prepareLanguageDialogs(QUESTION_LANGUAGE_DIALOG);
				return true;
		}
		return false;
	}

	private void prepareLanguageDialogs(int id)
	{
		final Object[] languagesIds = Config.lastData.identifiers.keySet().toArray();

		if (languagesIds.length < 2)
			return;

		switch (id)
		{
			case SWAP_LANGUAGE_DIALOG:
				showSwapLanguagesDialog(languagesIds);
				break;
			case QUESTION_LANGUAGE_DIALOG:
				showQuestionLanguageDialog(languagesIds);
				break;
			case ANSWER_LANGUAGE_DIALOG:
				showAnswerLanguageDialog(languagesIds);
				break;
		}
	}

	/**
	 * Only called whether there are exactly two languages in the kvtml collection.
	 */
	protected void showSwapLanguagesDialog(final Object[] languagesIds)
	{
		new AlertDialog.Builder(this)
				.setTitle(R.string.menu_swap_languages)
				.setMessage(R.string.message_swap_languages_only_for_this_activity)
				.setCancelable(true)
				.setPositiveButton(R.string.button_yes,
						new DialogInterface.OnClickListener()
						{
							@Override
							public void onClick(DialogInterface dialog, int id)
							{
								if(mDisplayLanguageId.equals(Config.getAnswerLangId()))
									mDisplayLanguageId = Config.getQuestionLangId();
								else
									mDisplayLanguageId = Config.getAnswerLangId();

								// The list of items is prepared:
								loadEntries();

								// The list of items is added to the ListView object:
								list.setAdapter(sAdapter);
							}
						}
				)
				.setNegativeButton(R.string.button_no,
						new DialogInterface.OnClickListener()
						{
							@Override
							public void onClick(DialogInterface dialog, int id)
							{
								dialog.cancel();
							}
						}
				)
				.create()
				.show();
	}

	/**
	 * Only called whether there are three or more languages in the kvtml collection.
	 *
	 * TODO: can be useful this functionality?
	 */
	protected void showQuestionLanguageDialog(final Object[] languagesIds)
	{
	}

	/**
	 * Only called whether there are three or more languages in the kvtml collection.
	 *
	 * TODO: can be useful this functionality?
	 */
	protected void showAnswerLanguageDialog(final Object[] languagesIds)
	{
	}

	/**
	 * TODO: it is possible to sort the list in other manners: for grade, for time, eccetera.
	 */
	private void loadEntries()
	{
		if (data == null)
		{
			finish();
			return;
		}

		ArrayList<Kvtml.Entry> entries = null;
		if (mSortAlphabetically)
			entries = getSortedEntries(data.entries);
		else
		{
			entries = new ArrayList<Kvtml.Entry>();
			entries.addAll(data.entries.values());
		}

		if (entries == null)
		{
			Toast.makeText(this, R.string.error_loading_entries, Toast.LENGTH_LONG).show();
			finish();
			return;
		}

		HashMap<String, Kvtml.Identifier> ids = data.identifiers;
		if (ids == null)
		{
			Toast.makeText(this, R.string.error_loading_identifiers, Toast.LENGTH_LONG).show();
			finish();
			return;
		}

		String idName = ids.get(mDisplayLanguageId).name;
		if (idName == null)
		{
			Toast.makeText(this, R.string.error_loading_identifier, Toast.LENGTH_LONG).show();
			finish();
			return;
		}

		Iterator<Kvtml.Entry> itera = entries.iterator();
		String sTitle = String.format("%s (%s [#%d])", mWindowTitle, idName, data.entries.size());

		// Set both the title of the Activity and the customized text of the ActionBar,
		// although the first isn't shown:
		setTitle(sTitle);
		mTextViewForActionBar.setText(sTitle);

		List<Kvtml.Entry> listItems = new ArrayList<Kvtml.Entry>();
		while (itera.hasNext())
			listItems.add(itera.next());

		sAdapter = new EntryListAdapter(this, R.id.rowtext, listItems, mDisplayLanguageId);
	}

	private ArrayList<Kvtml.Entry> getSortedEntries(HashMap<String, Kvtml.Entry> entries)
	{
		if (data == null)
			return null;

		SortedSet<Kvtml.Entry> sortedSet = null;
		Object[] sortedArray = null;

		// The ordination object:
		sortedSet = new TreeSet<Kvtml.Entry>(
			new Comparator<Kvtml.Entry>()
				{
					// The criteria in order to compare two Entries in alphabetic sense:
					@Override
					public int compare(Kvtml.Entry lhs, Kvtml.Entry rhs)
					{
						Kvtml.Translation t;
						String s1;
						String s2;

						// We handle the case of bad formatted kvtml collections:
						try
						{
							t = lhs.translations.get(mDisplayLanguageId);
							s1 = String.valueOf(t.text);
						}
						catch (Exception e)
						{
							s1 = "– – " + lhs.id + " – –";
						}

						try
						{
							t = rhs.translations.get(mDisplayLanguageId);
							s2 = String.valueOf(t.text);
						}
						catch (Exception e)
						{
							s2 = "– – " + rhs.id + " – –";
						}

						return s1.compareToIgnoreCase(s2);
					}
				}
		);

		try
		{
			ArrayList<Kvtml.Entry> values = new ArrayList<Kvtml.Entry>(entries.values());
			sortedSet.addAll(values);
			sortedArray = sortedSet.toArray();
		}
		catch (Exception e)
		{
			return null;
		}

		if (sortedArray == null)
			return null;

		ArrayList<Kvtml.Entry> sortedEntries = new ArrayList<Kvtml.Entry>();
		Kvtml.Entry entry = null;

		for (int n = 0, mn = sortedArray.length; n < mn; n++)
		{
			entry = (Kvtml.Entry) sortedArray[n];
			if (entry == null)
				// XXX: must be reviewed: why not continue here?
				return null;
			sortedEntries.add(entry);
		}

		return sortedEntries;
	}

	public int getPositionById(String id)
	{
		if (data == null)
			return -1;

		Iterator<String> it = data.identifiers.keySet().iterator();
		int n = 0;

		while (it.hasNext())
		{
			if (String.valueOf(it.next()).equals(id))
				return n;
			n++;
		}

		return -1;
	}

	public String getIdByPosition(int index)
	{
		if (data == null)
			return null;

		Iterator<Entry<String, Kvtml.Identifier>> it = data.identifiers.entrySet().iterator();
		Kvtml.Identifier identifier = null;
		int n = 0;

		while (it.hasNext())
		{
			identifier = it.next().getValue();
			if (n == index)
				return identifier.id;
			n++;
		}

		return null;
	}
}