/*
 * Libero Vocab
 *     An app for Android systems which allows to do practice with kvtml
 *     vocabulary files.
 *     This program is a fork of another program called "Vocab Drill" by:
 *       - Károly Kiripolszky <karcsi@ekezet.com>
 *       - Matthias Völlinger <matthias.voellinger@gmx.de>
 *
 *     Copyright (C) 2019, 2020  Lo Iacono Massimo (massimol@inventati.org)
 *
 *     This program is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package org.inventati.massimol.liberovocab.dialogs;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import android.support.v7.app.AppCompatActivity;

import android.app.Activity;
import android.content.Intent;
import android.content.SharedPreferences;
import android.os.Bundle;
import android.view.View;
import android.view.WindowManager;
import android.widget.ListView;
import android.widget.AdapterView;
import android.widget.TextView;
import android.widget.Toast;

import org.inventati.massimol.liberovocab.Config;
import org.inventati.massimol.liberovocab.R;
import org.inventati.massimol.liberovocab.adapters.FileListAdapter;

public abstract class OpenDialog extends AppCompatActivity
{
	// protected static String PREF_BROWSER_DEFAULT_DIR = "browser_default_dir";
	public static final String INTENT_PATH_PARAMETER = "path";
	public static final String INTENT_NAME_PARAMETER = "name";
	public static final String INTENT_PARENT_PARAMETER = "parent";

	protected String mLastPathPrefName = null;

	/**
	 * Represents an entry in a picker dialog.
	 */
	public abstract class DialogEntry implements Comparable<DialogEntry>
	{
		protected String mPath;

		public DialogEntry(String path)
		{
			mPath = path;
		}

		public String getPath()
		{
			return mPath;
		}

		public boolean canRead()
		{
			return true;
		}

		public abstract String getName();
		public abstract String getParent();
		public abstract DialogEntry[] getFileList();
		public abstract DialogEntry[] getFolderList();
		public abstract boolean isDirectory();
		abstract protected boolean isSelected();
		abstract protected boolean exists();

		@Override
		public int compareTo(DialogEntry another)
		{
			return mPath.compareTo(another.getPath());
		}
	}

	protected SharedPreferences mPrefs;

	protected DialogEntry mCurrentEntry = null;
	protected String mDefaultPath = null;
	protected String mStoragePublicExtPath_1 = null;
	protected String mStoragePublicExtPath_2 = null;
	protected String mStoragePrivateExtPath_1 = null;
	protected String mStoragePrivateExtPath_2 = null;
	protected String mSelectedFile = null;
	protected boolean mEnableDirScan = true;
	protected List<DialogEntry> mPaths = null;

	private FileListAdapter mAdapter;
	private TextView mTxtGoUp = null;
	TextView mTxtPath = null;

	/**
	 * The ListView object:
	 */
	private ListView list;

	/**
	 * Initialize File Open dialog.
	 */
	@Override
	public void onCreate(Bundle savedInstanceState)
	{
		super.onCreate(savedInstanceState);

		// enter fullscreen mode (hide status bar)
		getWindow().setFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN,
				             WindowManager.LayoutParams.FLAG_FULLSCREEN);

		// This activity has an ActionBar:
		getSupportActionBar().setHomeButtonEnabled(true);

		// create default objects
		mLastPathPrefName = getLastPathPrefName();
		mPrefs = Config.getPrefs();
		mTxtPath = (TextView) findViewById(R.id.currentPath);
		mTxtGoUp = (TextView) findViewById(R.id.go_to_parent_directory);
		mAdapter = new FileListAdapter(this, R.layout.item_file_list);
		mAdapter.setParentDirLabel(getResources().getString(R.string.go_to_parent_directory));
		mPaths = new ArrayList<DialogEntry>();

		list = (ListView) findViewById(android.R.id.list);
		list.setTextFilterEnabled(true);

		list.setOnItemClickListener(new AdapterView.OnItemClickListener()
			{
				@Override
				public void onItemClick(AdapterView<?> parent, View view, int position, long id)
				{
					listItemClick(position);
				}
			}
		);

		// The list of items is prepared:
		initialize(savedInstanceState);
		updateList(mCurrentEntry);

		// The list of items is added to the ListView object:
		list.setAdapter(mAdapter);
	}

	/**
	 * Updates the list with the contents of a folder (no recursion).
	 *
	 * @param path to the folder
	 */
	protected void updateList(DialogEntry entry)
	{
		if (entry == null)
			return;

		// Update path display:
		mCurrentEntry = entry;
		String path = entry.getPath();

		mTxtPath.setText(path);

		// Clear lists:
		mPaths.clear();
		mAdapter.clear();

		// Show the "up..." item if not in root:
		if (path.equals("/"))
			mTxtGoUp.setVisibility(View.GONE);
		else
			mTxtGoUp.setVisibility(View.VISIBLE);

		// Get directory list and sort it:
		DialogEntry[] folders = entry.getFolderList();
		if (folders != null && 0 < folders.length)
		{
			Arrays.sort(folders);
			if (Config.scanDirectories && mEnableDirScan)
			{
				// show number of files in folder if enabled,
				// XXX: This may be slow...
				DialogEntry folder = null;
				for (int i = 0, I = folders.length; i < I; i++)
				{
					folder = folders[i];
					DialogEntry[] list = folder.getFileList();
					if (list != null && 0 < list.length)
						mAdapter.add(String.format("%s (%d)%s", folder.getName(), list.length, File.separator));
					else
						mAdapter.add(String.format("%s%s", folder.getName(), File.separator));
					mPaths.add(folder);
				}
			} else
			{
				DialogEntry folder = null;
				for (int i = 0, I = folders.length; i < I; i++)
				{
					folder = folders[i];
					mAdapter.add(String.format("%s%s", folder.getName(), File.separator));
					mPaths.add(folder);
				}
			}
		}

		// Get kvtml files list and sort it:
		updateListForKvtmlFiles(entry);
	}

	// Get kvtml files list and sort it:
	void updateListForKvtmlFiles(DialogEntry entry)
	{
		DialogEntry[] files = entry.getFileList();
		if (files == null || files.length == 0)
			return;
		Arrays.sort(files);
		DialogEntry file = null;
		for (int i = 0, I = files.length; i < I; i++)
		{
			file = files[i];
			mAdapter.add(file.getName());
			mPaths.add(file);
		}
	}

	/**
	 * Good place for extending classes to set path variables.
	 *
	 * @param savedInstanceState
	 */
	protected void initialize(Bundle savedInstanceState)
	{
		// mStoragePublicExtPath_1 = "";
		// mDefaultPath = "";
		// mCurrentPath = "";
	}

	/**
	 * When a list item is selected.
	 */
	protected void listItemClick(int position)
	{
		final DialogEntry selectedItem = mPaths.get(position);

		if (selectedItem.isSelected())
		{
			// pass result back to caller
			final Intent data = new Intent();
			data.putExtra(INTENT_PATH_PARAMETER, selectedItem.getPath());
			data.putExtra(INTENT_NAME_PARAMETER, selectedItem.getName());
			data.putExtra(INTENT_PARENT_PARAMETER, selectedItem.getParent());
			String lastPath = selectedItem.getParent();
			final SharedPreferences.Editor editor = mPrefs.edit();
			editor.putString(mLastPathPrefName, lastPath)
				  .commit();
			setResult(Activity.RESULT_OK, data);
			finish();
			return;
		}

		// verify read permission
		if (selectedItem.canRead())
			updateList(selectedItem);
		else
		{
			String errorText = getResources().getString(R.string.error_cannot_read_folder);
			Toast.makeText(this, errorText, Toast.LENGTH_LONG).show();
		}
	}

	/**
	 * Save configuration on pause.
	 */
	@Override
	protected void onPause()
	{
		super.onPause();
		if (mCurrentEntry == null)
			return;
		final SharedPreferences.Editor editor = mPrefs.edit();
		editor
			.putString(mLastPathPrefName, mCurrentEntry.getPath())
			.commit();
	}

	/**
	 * Save instance state.
	 */
	@Override
	protected void onSaveInstanceState(Bundle outState)
	{
		super.onSaveInstanceState(outState);
		outState.putString(mLastPathPrefName, mCurrentEntry.getPath());
	}

	public void onGoUpClick(View view)
	{
		final DialogEntry parent = newEntry(mCurrentEntry.getParent());
		if (parent == null)
			return;
		updateList(parent);
	}

	protected abstract String getLastPathPrefName();

	abstract protected DialogEntry newEntry(String path);
}
