/*
 * Libero Vocab
 *     An app for Android systems which allows to do practice with kvtml
 *     vocabulary files.
 *     This program is a fork of another program called "Vocab Drill" by:
 *       - Károly Kiripolszky <karcsi@ekezet.com>
 *       - Matthias Völlinger <matthias.voellinger@gmx.de>
 *
 *     Copyright (C) 2019, 2020  Lo Iacono Massimo (massimol@inventati.org)
 *
 *     This program is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package org.inventati.massimol.liberovocab.kvtml;

import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;

final public class Kvtml
{
	final public static String DOCTYPE =
		" kvtml PUBLIC \"kvtml2.dtd\" \"http://edu.kde.org/kvtml/kvtml2.dtd\"";

	final public static String VERSION = "2.0";

	final public static String ROOT_TAG = "kvtml";

	final public static String INFORMATION_TAG = "information";
	final public static String GENERATOR_TAG = "generator";
	final public static String TITLE_TAG = "title";
	final public static String DATE_TAG = "date";

	final public static String IDENTIFIERS_TAG = "identifiers";
	final public static String IDENTIFIER_TAG = "identifier";
	final public static String NAME_TAG = "name";
	final public static String LOCALE_TAG = "locale";

	final public static String ENTRIES_TAG = "entries";
	final public static String ENTRY_TAG = "entry";
	final public static String COMMENT_TAG = "comment";
	final public static String TRANSLATION_TAG = "translation";
	final public static String TEXT_TAG = "text";
	final public static String SOUND_TAG = "sound";

	final public static String GRADE_TAG = "grade";
	final public static String CURRENT_GRADE_TAG = "currentgrade";
	final public static String COUNT_TAG = "count";
	final public static String ERROR_COUNT_TAG = "errorcount";
	final public static String GRADE_DATE_TAG = "date";
	final public static String GRADE_DATE_DATE_FORMAT = "yyyy-MM-dd'T'HH:mm:ss";

	final public static String LESSONS_TAG = "lessons";
	final public static String CONTAINER_TAG = "container";
	final public static String INPRACTICE_TAG = "inpractice";

	public class Node
	{
		public String id;
	}

	public class Identifier extends Node
	{
		public String name;
		public String locale;
	}

	public class Grade
	{
		public int currentGrade;
		public int count;
		public int errorCount;
		public String date;

		public Grade()
		{
			currentGrade = 0;
			count = 0;
			errorCount = 0;
			date = "";
		}
	}

	public class Translation extends Node
	{
		public String text;
		public String sound;
		public String comment;
		public Grade grade;

		public Translation()
		{
			text = "";
			sound = "";
			comment = "";
			grade = new Grade();
		}

		public void setGrade(Grade iGrade)
		{
			grade = iGrade;
		}
	}

	public class Entry extends Node
	{
		public HashMap<String, Translation> translations;

		public Entry()
		{
			translations = new HashMap<String, Translation>();
		}
	}

	public class Lesson extends Node
	{
		public String name;
		public Boolean inpractice;
		public Set<String> keys;
		public HashMap<String, Lesson> subLessons;

		public Lesson()
		{
			keys = new HashSet<String>();
			subLessons = new HashMap<String, Lesson>();
		}
	}

	public String generator;
	public String title;
	public Date date;
	public String version;

	public HashMap<String, Identifier> identifiers;
	public HashMap<String, Entry> entries;
	public HashMap<String, Lesson> lessons;

	public Kvtml()
	{
		entries = new HashMap<String, Entry>();
		identifiers = new HashMap<String, Identifier>();
		lessons = new HashMap<String, Lesson>();
		version = null;
	}

	public Kvtml.Identifier newIdentifier()
	{
		return new Kvtml.Identifier();
	}

	public Kvtml.Grade newGrade()
	{
		return new Kvtml.Grade();
	}

	public Kvtml.Entry newEntry()
	{
		return new Kvtml.Entry();
	}

	public Kvtml.Translation newTranslation()
	{
		return new Kvtml.Translation();
	}

	public Kvtml.Lesson newLesson()
	{
		return new Kvtml.Lesson();
	}

	/**
	 * Clones a KVTML object containing the specified entries. Useful for creating
	 * test documents at runtime.
	 *
	 * @param clonedEntries
	 * @return A new KVTML document object.
	 */
	public Kvtml cloneFromEntries(HashMap<String, Kvtml.Entry> clonedEntries)
	{
		Kvtml clone = new Kvtml();
		clone.version = VERSION;
		clone.generator = new String(generator);
		clone.title = new String(title);
		clone.date = (Date) date.clone();
		clone.identifiers.putAll(identifiers);
		clone.entries.putAll(clonedEntries);
		clone.lessons.putAll(lessons);
		return clone;
	}
}