/*
 * Libero Vocab
 *     An app for Android systems which allows to do practice with kvtml
 *     vocabulary files.
 *     This program is a fork of another program called "Vocab Drill" by:
 *       - Károly Kiripolszky <karcsi@ekezet.com>
 *       - Matthias Völlinger <matthias.voellinger@gmx.de>
 *
 *     Copyright (C) 2019, 2020  Lo Iacono Massimo (massimol@inventati.org)
 *
 *     This program is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package org.inventati.massimol.liberovocab.kvtml;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;

/**
 * A FileReader class to track the number of bytes read from a file.
 */
public class KvtmlFileReader extends FileReader
{
	public interface OnReadingFileListener
	{
		/**
		 * Triggered on reading a file.
		 * 
		 * @param current Number of already read bytes.
		 * @param total The total number of bytes.
		 */
		public void onReadingFile(long current, long total);
	}
	
	private OnReadingFileListener mOnReadingFileListener = null;
	private long mFileSize;
	private long mCurrentByte;
	
	public KvtmlFileReader(String filename) throws FileNotFoundException
	{
		super(filename);
		mFileSize = (new File(filename)).length();
		mCurrentByte = 0;
		// create an empty listener so we can spare checks for null in the read()
		// method
		clearOnReadingFileListener();
	}
	
	@Override
	public int read(char[] buffer, int offset, int length) throws IOException
	{
		// increases the number of bytes read
		mCurrentByte += length;
		mOnReadingFileListener.onReadingFile(mCurrentByte, mFileSize);
		return super.read(buffer, offset, length);
	}
	
	public void setOnReadingFileListener(OnReadingFileListener listener)
	{
		if (listener == null)
			return;
		mOnReadingFileListener = listener;
	}
	
	public void clearOnReadingFileListener()
	{
		mOnReadingFileListener = new OnReadingFileListener()
		{
			@Override
			public void onReadingFile(long current, long total)
			{
			}
		};
	}
	
	public long getFileSize()
	{
		return mFileSize;
	}
}