/*
 * Libero Vocab
 *     An app for Android systems which allows to do practice with kvtml
 *     vocabulary files.
 *     This program is a fork of another program called "Vocab Drill" by:
 *       - Károly Kiripolszky <karcsi@ekezet.com>
 *       - Matthias Völlinger <matthias.voellinger@gmx.de>
 *
 *     Copyright (C) 2019, 2020  Lo Iacono Massimo (massimol@inventati.org)
 *
 *     This program is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package org.inventati.massimol.liberovocab;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.SharedPreferences;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.pm.PackageManager.NameNotFoundException;

import org.inventati.massimol.liberovocab.helpers.Gradient;
import org.inventati.massimol.liberovocab.kvtml.Kvtml;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.Iterator;

@SuppressWarnings("LoopStatementThatDoesntLoop")
public final class Config
{
	/**
	 * Name of preferences file.
	 */
	public final static String PREFS_FILE = "Config";
	public final static String PREF_RECENT_FILE = "recent_%d";
	public final static String STARTTIME_FILE = "STARTTIME";

	public final static String KVTML_ASSET_DIRECTORY = "kvtml-files";

	//public final static String PROGRESS_FILE = "liberovocab.kvtml";

	/**
	 * Interval date format for the Leitner-System.
	 *
	 * Month:Day:Hour
	 *
	 * Supported by Parley by the KDE Educational Project.
	 */
	public final static String INTERVAL_DATE_FORMAT = "MM:dd:HH";

	/**
	 * String delimiter used by concatenation routines.
	 */
	public final static String DELIM = ",";

	/**
	 * .Number of recently opened files to keep in the list:
	 */
	public static int maxRecentFilesNum = 20;

	/**
	 * Handle to the currently used KVTMl file.
	 */
	public static File inputFile;

	/**
	 * Previously loaded KVTML data.
	 */
	public static Kvtml lastData;

	/**
	 * Maximum number of choices in multiple choice mode.
	 */
	public static int choiceNumber = 4;

	/**
	 * Fill the list  with already answered questions.
	 */
	public static boolean forceChoiceNumber = true;

	/**
	 * Maximum number of choices in multiple choice mode.
	 */
	public static int distanceLimitTouch = 5;

	/**
	 * Maximum number in statistics:
	 */
	public static int numMaxStatistic = 15;

	/**
	 * Using sound in place of text for questions.
	 */
	public static boolean forceQuestionWithSound = true;

    /**
     * Replace separator char.
     */
    public static boolean replaceSeparatorChar = false;

    /**
     * Show diagnostic message for audio.
     */
    public static boolean showDiagnosticMessageForAudio = false;

	/**
	 * Time to sleep before jumping to the next question.
	 */
	public static long questionDelay = 200;

	/**
	 * Whether use a full or a short message when the question text il hidden. The value.
	 */
	private static String fullOrShortMessageForHiddenQuestionTextValue;

	/**
	 * Base directory for audio files. The value.
	 */
	private static String baseDirectoryForAudioFilesValue;

	/**
	 * Alternative base directory for audio files.
	 */
	private static String baseDirectoryForAudioFilesOtherPath = "/";

	/**
	 * Starting string to remove in path preference.
	 */
	private static String startStringToBeRemovedInPath = "";

	/**
	 * If true the file browser will show the number of lesson files in
	 * directories.
	 */
	public static boolean scanDirectories = true;

	public static boolean confirmExit = true;
	public static boolean confirmProgress = true;

	/**
	 * The language of the question:
	 */
	private static String sQuestionLanguageId = null;

	/**
	 * The language of the answer:
	 */
	private static String sAnswerLanguageId = null;

	/**
	 * The id of the language whose grade will be modified for each practiced entry.
	 *
	 * In this application, the strategy used is that the graduation is
	 * applied always to the language of the question.
	 */
	private static String sLanguageForGradeId = null;

	/**
	 * Standard practice intervals for the Leitner System.
	 */
	public final static String[] defaultPracticeIntervals = {
		"00:00:12", "00:01:00", "00:03:00",
		"00:07:00", "00:14:00", "01:00:00",
		"02:00:00" };

	public static String[] practiceIntervals = {
		"00:00:12", "00:01:00", "00:03:00",
		"00:07:00", "00:14:00", "01:00:00",
		"02:00:00" };

	public static String[] recentFilesList;

	private static SharedPreferences sPrefs;
	private static File sCacheDir;
	private static String sVersionName;
	private static int sVersionCode = -1;

	private static boolean sMuteNewsDialog = false;
	private static int sPreviousVersionCode = 0;
	private static String sPreviousVersionName = "";

    private static int colorStartForGradient;
    private static int colorEndForGradient;

	/**
	 * Maximus height of the TextView for the answer and the comment (in dp ?):
	 */
	public final static int MAX_HEIGHT_VIEW_DP = 150;

	/**
	 * Loads the configuration from the preferences file.
	 */
	public static SharedPreferences load(Context context)
	{
		sVersionCode = -1;
		try
		{
			setStartupTime(context);
			final PackageManager pm = context.getPackageManager();
			final PackageInfo inf = pm.getPackageInfo(context.getPackageName(), 0);
			sVersionName = inf.versionName;
			sVersionCode = inf.versionCode;
		}
		catch (NameNotFoundException e)
		{
			sVersionName = "?.?";
			e.printStackTrace();
		}
		catch (IOException e)
		{
			e.printStackTrace();
		}

		sCacheDir = context.getCacheDir();
		sCacheDir.mkdir();

		sPrefs = context.getSharedPreferences(PREFS_FILE, Context.MODE_PRIVATE);

		sMuteNewsDialog = sPrefs.getBoolean("muteNewsDialog", sMuteNewsDialog);
		sPreviousVersionCode = sPrefs.getInt("previousVersionCode", sPreviousVersionCode);
		sPreviousVersionName = sPrefs.getString("previousVersionName", sPreviousVersionName);

		choiceNumber = sPrefs.getInt("choiceNumber", choiceNumber);
		forceChoiceNumber = sPrefs.getBoolean("forceChoiceNumber", forceChoiceNumber);
		distanceLimitTouch = sPrefs.getInt("distanceLimitTouch", distanceLimitTouch);
		numMaxStatistic = sPrefs.getInt("numMaxStatistic", numMaxStatistic);
		forceQuestionWithSound = sPrefs.getBoolean("forceQuestionWithSound", forceQuestionWithSound);
        replaceSeparatorChar = sPrefs.getBoolean("replaceSeparatorChar", replaceSeparatorChar);
		showDiagnosticMessageForAudio = sPrefs.getBoolean("showDiagnosticMessageForAudio", showDiagnosticMessageForAudio);
		// XXX: conversion necessarry for backwards compatibility
		questionDelay = Long.parseLong(sPrefs.getString("questionDelay", String.valueOf(questionDelay)));
		fullOrShortMessageForHiddenQuestionTextValue = sPrefs.getString("fullOrShortMessageForHiddenQuestionTextValue", fullOrShortMessageForHiddenQuestionTextValue);
		baseDirectoryForAudioFilesValue = sPrefs.getString("baseDirectoryForAudioFilesValue", baseDirectoryForAudioFilesValue);
		baseDirectoryForAudioFilesOtherPath = sPrefs.getString("baseDirectoryForAudioFilesOtherPath", baseDirectoryForAudioFilesOtherPath);
		startStringToBeRemovedInPath = sPrefs.getString("startStringToBeRemovedInPath", startStringToBeRemovedInPath);
		confirmExit = 1 == sPrefs.getInt("confirmExit", confirmExit ? 1 : 0);
		confirmProgress = 1 == sPrefs.getInt("confirmProgress", confirmProgress ? 1 : 0);
		scanDirectories = 1 == sPrefs.getInt("scanDirectories", scanDirectories ? 1 : 0);
		maxRecentFilesNum = sPrefs.getInt("maxRecentFilesNum", maxRecentFilesNum);
		String load = sPrefs.getString("practiceIntervals", null);
		if (load != null)
		{
			// ensure we have 7 values
			String[] tmp = new String[defaultPracticeIntervals.length];
			String[] tmp2 = load.split(DELIM);
			for (int n = 0, N = tmp.length, N2 = tmp2.length; n < N; n++)
				if (n < N2)
					tmp[n] = tmp2[n];
				else
					tmp[n] = defaultPracticeIntervals[n];
			practiceIntervals = tmp;
		}

		loadRecentFilesList();
		Gradient.buildCache(16);
		return sPrefs;
	}

	/**
	 * Saves the configuration to the preferences file.
	 */
	public static boolean save()
	{
		SharedPreferences.Editor edit = sPrefs.edit();

		edit.putInt("choiceNumber", choiceNumber);
		edit.putBoolean("forceChoiceNumber", forceChoiceNumber);
		edit.putInt("distanceLimitTouch", distanceLimitTouch);
		edit.putInt("numMaxStatistic", numMaxStatistic);
		edit.putBoolean("forceQuestionWithSound", forceQuestionWithSound);
		edit.putBoolean("replaceSeparatorChar", replaceSeparatorChar);
		edit.putBoolean("showDiagnosticMessageForAudio", showDiagnosticMessageForAudio);
		edit.putInt("confirmExit", confirmExit ? 1 : 0);
		edit.putInt("confirmProgress", confirmProgress ? 1 : 0);
		edit.putInt("scanDirectories", scanDirectories ? 1 : 0);
		edit.putInt("maxRecentFilesNum", maxRecentFilesNum);
		edit.putBoolean("muteNewsDialog", sMuteNewsDialog);
		// XXX: conversion necessarry for backwards compatibility
		edit.putString("questionDelay", String.valueOf(questionDelay));
		edit.putString("fullOrShortMessageForHiddenQuestionTextValue", fullOrShortMessageForHiddenQuestionTextValue);
		edit.putString("baseDirectoryForAudioFilesValue", baseDirectoryForAudioFilesValue);
		edit.putString("baseDirectoryForAudioFilesOtherPath", baseDirectoryForAudioFilesOtherPath);
		edit.putString("startStringToBeRemovedInPath", startStringToBeRemovedInPath);
		// generate array string for storage
		String store = "";
		for (int i = 0; i < 7; i++)
		{
			store += practiceIntervals[i];
			if (i < 6)
				store += DELIM;
		}
		store = store.substring(0, store.length() - 1);
		edit.putString("practiceIntervals", store);
		edit.putInt("previousVersionCode", sPreviousVersionCode);
		edit.putString("previousVersionName", sPreviousVersionName);
		return edit.commit();
	}

	/**
	 * Initialize the language id for the question, the answer and the
	 * graduated language.
	 *
	 * If those were already initialized, does nothing.
	 *
	 * If it is the first time, the first and second identifiers of the
	 * kvtml collection will be used for question and answer languages.
	 *
	 * If a second identifier doesn't exist, the answer language will
	 * be set to null.
	 */
	public static void inizializeLanguages()
	{
		if (lastData == null)
			return;

		// If it's already had been set, return the value:
		if (sQuestionLanguageId != null)
			return;

		// If it is the first time, get the first identifier in the kvtml collection.
		// Moreover, set the value of the answer language to the second identifiers
		// if that exists, on the contrary it.ll be null.
		Iterator<String> it = lastData.identifiers.keySet().iterator();
		String answer = null;
		try
		{
			while (it.hasNext())
			{
				sQuestionLanguageId = String.valueOf(it.next());
				sLanguageForGradeId = sQuestionLanguageId;

				answer = String.valueOf(it.next());
				break;
			}
		}
		catch(Exception e)
		{
			e.printStackTrace();
			sAnswerLanguageId = null;
		}

		sAnswerLanguageId = answer;
	}

	public static void setColorStartForGradient(int color)
	{
		colorStartForGradient = color;
	}

	public static int getColorStartForGradient()
	{
		return colorStartForGradient;
	}

	public static void setColorEndForGradient(int color)
	{
		colorEndForGradient = color;
	}

	public static int getColorEndForGradient()
	{
		return colorEndForGradient;
	}

	public static void setFullOrShortMessageForHiddenQuestionTextValue(String s)
	{
		fullOrShortMessageForHiddenQuestionTextValue = s;
	}

	public static String getFullOrShortMessageForHiddenQuestionTextValue()
	{
		return fullOrShortMessageForHiddenQuestionTextValue;
	}

	public static void setBaseDirectoryForAudioFilesValue(String s)
	{
		baseDirectoryForAudioFilesValue = s;
	}

	public static String getBaseDirectoryForAudioFilesValue()
	{
		return baseDirectoryForAudioFilesValue;
	}

	public static void setBaseDirectoryForAudioFilesOtherPath(String s)
	{
		baseDirectoryForAudioFilesOtherPath = s;
	}

	public static String getBaseDirectoryForAudioFilesOtherPath()
	{
		return baseDirectoryForAudioFilesOtherPath;
	}

    public static void setStartStringToBeRemovedInPath(String s)
    {
		startStringToBeRemovedInPath = s;
    }

    public static String getStartStringToBeRemovedInPath()
    {
        return startStringToBeRemovedInPath;
    }

	public static String getQuestionLangId()
	{
		if (lastData == null)
			return null;

		return sQuestionLanguageId;
	}

	public static String getAnswerLangId()
	{
		return sAnswerLanguageId;
	}

	public static String getLanguageForGradeId()
	{
		return sLanguageForGradeId;
	}

	/**
	 * Set the id of the language of the questions and updates the language of the
	 * answer accordingly.
	 *
	 * Accordingly updates also the language to whom the grade'll be applied.
	 */
	public static void setQuestionLangId(String id)
	{
		sQuestionLanguageId = id;
		sLanguageForGradeId = sQuestionLanguageId;
	}

	/**
	 * Set the id of the language of the answer and updates the language of the
	 * questions accordingly.
	 *
	 * Accordingly updates also the language to whom the grade'll be applied.
	 */
	public static void setAnswerLangId(String id)
	{
		sAnswerLanguageId = id;
	}

	/**
	 * Loads and returns the recent files' list and removes paths that don't
	 * exist.
	 *
	 * @return True on success
	 */
	@SuppressLint("DefaultLocale")
	public static boolean loadRecentFilesList()
	{
		SharedPreferences.Editor edit = sPrefs.edit();
		String path;
		String key;
		recentFilesList = new String[maxRecentFilesNum];
		int i = 0;
		for (int n = 0; n < maxRecentFilesNum; n++)
		{
			key = String.format(PREF_RECENT_FILE, n);
			path = sPrefs.getString(key, null);
			if (path == null)
				edit.putString(key, "");
			else if ((new File(path)).exists())
			{
				//Log.d(Config.class.getSimpleName(), String.format("recent[%d]: '%s'", i, path));
				recentFilesList[i] = path;
				i++;
			}
		}
		// TODO: test me: n = 0 || n = i
		for (int n = i; n < maxRecentFilesNum; n++)
			if (recentFilesList[n] == null)
				recentFilesList[n] = "";
		return edit.commit();
	}

	/**
	 * Inserts a path on top of the recent files' list.
	 */
	public static boolean insertRecentFile(String path)
	{
		if (recentFilesList == null || recentFilesList.length == 0 || path == null)
			return false;
		int mn = recentFilesList.length;
		String[] tmpList = new String[mn];
		int i = 0;
        for (String aRecentFilesList : recentFilesList)
        {
            if (!aRecentFilesList.equals(path))
            {
                tmpList[i] = aRecentFilesList;
                i++;
            }
        }
		recentFilesList = new String[mn];
		recentFilesList[0] = path;
		for (int n = 1; n < mn; n++)
			recentFilesList[n] = tmpList[n - 1] != null ? tmpList[n - 1] : "";
		return true;
	}

	public static void removeRecentFile(String path)
	{
		if (recentFilesList == null || recentFilesList.length == 0 || path == null)
			return;
		int mn = recentFilesList.length;
		String[] tmpList = new String[mn - 1];
		int i = 0;
        for (String aRecentFilesList : recentFilesList)
        {
            if (!aRecentFilesList.equals(path))
            {
                tmpList[i] = aRecentFilesList;
                i++;
            }
        }
		recentFilesList = tmpList;
	}

	@SuppressLint("DefaultLocale")
	public static boolean clearRecentFilesList()
	{
		SharedPreferences.Editor edit = sPrefs.edit();
		recentFilesList = new String[maxRecentFilesNum];
		String key;
		try
		{
			for (int n = 0; n < maxRecentFilesNum; n++)
			{
				key = String.format(PREF_RECENT_FILE, n);
				edit.putString(key, "");
				recentFilesList[n] = "";
			}
		}
		catch (ArrayIndexOutOfBoundsException e)
		{
			return edit.commit();
		}
		return edit.commit();
	}

	@SuppressLint("DefaultLocale")
	public static boolean saveRecentFilesList()
	{
		if (!hasRecentFiles())
			return false;
		SharedPreferences.Editor edit = sPrefs.edit();
		for (int n = 0, mn = recentFilesList.length; n < mn; n++)
			edit.putString(String.format(PREF_RECENT_FILE, n), recentFilesList[n]);
		return edit.commit();
	}

	public static boolean hasRecentFiles()
	{
		return recentFilesList != null && 0 < recentFilesList.length
			&& recentFilesList[0] != null && 0 < recentFilesList[0].length();
	}

	public static boolean deleteDir(File dir)
    {
        if (dir != null && dir.isDirectory())
        {
            String[] children = dir.list();
            for (String aChildren : children)
            {
                boolean success = deleteDir(new File(dir, aChildren));
                if (!success)
                {
                    return false;
                }
            }
        }
        return dir != null && dir.delete();
    }

	public static boolean clearCacheDir()
	{
		deleteDir(sCacheDir);
		loadRecentFilesList();
		return true;
	}

	public static String getVersionName()
	{
		return sVersionName;
	}

	public static int getVersionCode()
	{
		return sVersionCode;
	}

	public static boolean setStartupTime(Context context) throws IOException
	{
		long ts;
		boolean ret = true;
		// installation identification file
		File iidf = new File(context.getFilesDir(), STARTTIME_FILE);
		try
		{
			if (!iidf.exists())
			{
				ts = System.currentTimeMillis();
				FileOutputStream fos = new FileOutputStream(iidf);
				ObjectOutputStream out = new ObjectOutputStream(fos);
				out.writeLong(ts);
				out.close();
			}
			else
			{
				FileInputStream fin = new FileInputStream(iidf);
				ObjectInputStream in = new ObjectInputStream(fin);
				in.close();
			}
		}
		catch (IOException e)
		{
			e.printStackTrace();
			ret = false;
		}
		catch (NumberFormatException e)
		{
			//Log.w(Config.class.getSimpleName(), String.format("Invalid STARTTIME. Deleting file: %s", iidf.getAbsolutePath()));
			e.printStackTrace();
			iidf.delete();
			ret = false;
		}
		return ret;
	}

	public static SharedPreferences getPrefs()
	{
		return sPrefs;
	}

	public static int getPreviousVersionCode()
	{
		return sPreviousVersionCode;
	}

	public static String getPreviousVersionName()
	{
		return sPreviousVersionName;
	}

	public static void invalidatePreviouseVersionCode()
	{
		sPreviousVersionCode = getVersionCode();
	}

	public static void invalidatePreviouseVersionName()
	{
		sPreviousVersionName = getVersionName();
	}
}