/*
 * Libero Vocab
 *     An app for Android systems which allows to do practice with kvtml
 *     vocabulary files.
 *     This program is a fork of another program called "Vocab Drill" by:
 *       - Károly Kiripolszky <karcsi@ekezet.com>
 *       - Matthias Völlinger <matthias.voellinger@gmx.de>
 *
 *     Copyright (C) 2019, 2020  Lo Iacono Massimo (massimol@inventati.org)
 *
 *     This program is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package org.inventati.massimol.liberovocab.activities;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import android.app.Activity;
import android.support.v7.app.AppCompatActivity;
import android.support.v7.app.ActionBar;

import android.app.AlertDialog;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.SharedPreferences;
import android.preference.PreferenceManager;
import android.os.Bundle;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.WindowManager.LayoutParams;
import android.view.ViewGroup;
import android.widget.AdapterView;
import android.widget.ArrayAdapter;
import android.widget.Button;
import android.widget.CompoundButton;
import android.widget.ExpandableListView;
import android.widget.LinearLayout;
import android.widget.RadioButton;
import android.widget.Spinner;
import android.widget.TextView;
import android.widget.Toast;

import org.inventati.massimol.liberovocab.Config;
import org.inventati.massimol.liberovocab.R;
import org.inventati.massimol.liberovocab.activities.test_types.FlashcardActivity;
import org.inventati.massimol.liberovocab.activities.test_types.MultipleChoiceActivity;
import org.inventati.massimol.liberovocab.activities.test_types.WritingActivity;
import org.inventati.massimol.liberovocab.adapters.LessonExpListAdapter;
import org.inventati.massimol.liberovocab.dialogs.EntryBrowserDialog;
import org.inventati.massimol.liberovocab.dialogs.CollectionBrowserDialog;
import org.inventati.massimol.liberovocab.helpers.EntriesFilter;
import org.inventati.massimol.liberovocab.helpers.Gradient;
import org.inventati.massimol.liberovocab.kvtml.Kvtml;
import org.inventati.massimol.liberovocab.kvtml.KvtmlWriter;

public class FilterActivity extends AppCompatActivity
{
	public final static String TARGET_ACTIVITY = "target_activity";
	public final static String EXTRA_FILTER = "extra_filter";

	public static final int SWAP_LANGUAGE_DIALOG = 1000;
	public static final int ANSWER_LANGUAGE_DIALOG = 1001;
	public static final int QUESTION_LANGUAGE_DIALOG = 1002;

	public static final int REQUEST_CODE_FROM_DIALOG_FOR_COLLECTIONS_LOAD_LESSONS = 2000;
	public static final int REQUEST_CODE_FROM_DIALOG_FOR_COLLECTIONS_PRACTICE_WORDS = 2001;

	private Spinner spinnerStart, spinnerEnd;
	private ArrayAdapter<CharSequence> adapterStart, adapterEnd;
	private RadioButton mRadioLeitner, mRadioLearn;
	private ExpandableListView expListView;
	LessonExpListAdapter mExpListAdptr;

	/**
	 * The customized TextView for the ActionBar:
	 */
	private TextView mTextViewForActionBar;

	/**
	 * The string to recognize the Activity the process come here from:
	 */
	private String targetActivity;

	// The button done:
	private Button doneButton;

	// If we have to filter for collection of entries, this will container the name
	// of the collection else null:
	private String nameOfCollectionOfEntriesForFilter;

	@Override
	public void onCreate(Bundle savedInstanceState)
	{
		super.onCreate(savedInstanceState);
		getWindow().setFlags(LayoutParams.FLAG_FULLSCREEN, LayoutParams.FLAG_FULLSCREEN);
		setContentView(R.layout.activity_filter);

		// getting target activity from the intent
		Intent i = getIntent();
		targetActivity = (String) i.getSerializableExtra(TARGET_ACTIVITY);

		// This activity has an ActionBar:
		ActionBar actionBar = getSupportActionBar();
		actionBar.setHomeButtonEnabled(true);

		// In order to customize the title in the ActionBar to be scrollable:
		actionBar.setDisplayShowCustomEnabled(true);
		actionBar.setDisplayShowTitleEnabled(false);

		LayoutInflater inflator = LayoutInflater.from(FilterActivity.this);
		View mCustomViewForActionBar = inflator.inflate(R.layout.view_action_bar_scrollable, null);
		mTextViewForActionBar = ((TextView) mCustomViewForActionBar.findViewById(R.id.action_bar_title));
		actionBar.setCustomView(mCustomViewForActionBar);

		// Normally we don't have to filter for collection of entries:
		nameOfCollectionOfEntriesForFilter = null;

		// In order to set the title in the ActionBar:
		setTitleInActionBar();

		// Objects used for the persistance of the preferences regard this activity:
		SharedPreferences prefForThisActivity = PreferenceManager.getDefaultSharedPreferences(this);
		final SharedPreferences.Editor prefEditor = prefForThisActivity.edit();

		// expandable list view
		expListView = (ExpandableListView) findViewById(R.id.explistview_lesson_filter);

		// set the header of the expandable list view
		LayoutInflater inflater = getLayoutInflater();
		ViewGroup expListHeader = (ViewGroup) inflater.inflate(R.layout.header_exp_list,
															   expListView, false);
		if (targetActivity.equals(EntryBrowserDialog.class.getName()))
		{
			LinearLayout practicePrefs = (LinearLayout) expListHeader.findViewById(R.id.layout_practice_prefs);
			practicePrefs.setVisibility(View.GONE);
		}
		mExpListAdptr = new LessonExpListAdapter(this, Config.lastData.lessons);
		expListView.addHeaderView(expListHeader);
		expListView.setAdapter(mExpListAdptr);

		mRadioLeitner = (RadioButton) findViewById(R.id.radio_leitner);
		mRadioLearn = (RadioButton) findViewById(R.id.radio_learn);
		if (!targetActivity.equals(FlashcardActivity.class.getName()))
			mRadioLearn.setVisibility(View.GONE);

		// Set the preferences for the mRadioLeitner:
		mRadioLeitner.setOnCheckedChangeListener(new CompoundButton.OnCheckedChangeListener()
			{
				@Override
				public void onCheckedChanged(CompoundButton buttonView, boolean isChecked)
				{
					if (targetActivity.equals(MultipleChoiceActivity.class.getName()))
						prefEditor.putBoolean("RadioLeitnerStateForMultipleChoice", isChecked);
					else if (targetActivity.equals(WritingActivity.class.getName()))
						prefEditor.putBoolean("RadioLeitnerStateForWriting", isChecked);
					else if (targetActivity.equals(FlashcardActivity.class.getName()))
						prefEditor.putBoolean("RadioLeitnerStateForFlashcard", isChecked);
					prefEditor.commit();
				}
			}
		);

		// Set the preferences for the mRadioLearn:
		mRadioLearn.setOnCheckedChangeListener(new CompoundButton.OnCheckedChangeListener()
			{
				@Override
				public void onCheckedChanged(CompoundButton buttonView, boolean isChecked)
				{
					prefEditor.putBoolean("RadioLearnStateForFlashcard", isChecked);
					prefEditor.commit();
				}
			}
		);

		// Get the preferences for the mRadioLeitner and mRadioLearn:
		if (targetActivity.equals(MultipleChoiceActivity.class.getName()))
			mRadioLeitner.setChecked(prefForThisActivity.getBoolean("RadioLeitnerStateForMultipleChoice", false));
		else if (targetActivity.equals(WritingActivity.class.getName()))
			mRadioLeitner.setChecked(prefForThisActivity.getBoolean("RadioLeitnerStateForWriting", false));
		else if (targetActivity.equals(FlashcardActivity.class.getName()))
		{
			mRadioLeitner.setChecked(prefForThisActivity.getBoolean("RadioLeitnerStateForFlashcard", false));
			mRadioLearn.setChecked(prefForThisActivity.getBoolean("RadioLearnStateForFlashcard", false));
		}

		// The button done:
		doneButton = (Button) findViewById(R.id.button_done);
		doneButton.setOnClickListener(new OnClickListener()
			{
				@Override
				public void onClick(View v)
				{
					// If there aren't entries in filter and no collection of entries was
					// selected:
					if (!entriesInFilter() && nameOfCollectionOfEntriesForFilter == null)
					{
						Toast.makeText(getApplicationContext(), getResources().getString(R.string.selected_dataset_empty), Toast.LENGTH_LONG).show();
						return;
					}

					// get level range
					int start = Integer.valueOf((String) spinnerStart.getSelectedItem());
					int end = Integer.valueOf((String) spinnerEnd.getSelectedItem());
					if (start > end)
					{
						int tmp = end;
						end = start;
						start = tmp;
					}

					// To specify the type of filters have to be applied to the entries in order
					// to do the practice.
					final EntriesFilter filter = new EntriesFilter();

					// If we have to filter for collection of entries:
					if (nameOfCollectionOfEntriesForFilter != null)
						filter.add(EntriesFilter.FILTER_COLLECTION_OF_ENTRIES, nameOfCollectionOfEntriesForFilter);

					// Normally we don't have to filter for collection of entries:
					nameOfCollectionOfEntriesForFilter = null;

					filter.add(EntriesFilter.FILTER_GRADE_RANGE, String.format("%d:%d", start, end));

					if (mRadioLeitner.isChecked())
						filter.add(EntriesFilter.FILTER_LEITNER, String.valueOf(true));

					// Now the test activity is ready to start:
					if (targetActivity.equals(WritingActivity.class.getName()))
					{
						Intent intent = new Intent(FilterActivity.this,
																			WritingActivity.class);
						intent.putExtra(EXTRA_FILTER, filter);
						intent.putExtra(WritingActivity.TEST_TYPE, WritingActivity.TEST_WRITING);
						intent.putExtra(WritingActivity.GRADUATED_MODE, mRadioLeitner.isChecked());
						startActivity(intent);
					}
					else if (targetActivity.equals(MultipleChoiceActivity.class.getName()))
					{
						Intent intent = new Intent(FilterActivity.this,
																	MultipleChoiceActivity.class);
						intent.putExtra(EXTRA_FILTER, filter);
						intent.putExtra(MultipleChoiceActivity.TEST_TYPE, MultipleChoiceActivity.TEST_CHOICE);
						intent.putExtra(MultipleChoiceActivity.GRADUATED_MODE, mRadioLeitner.isChecked());
						startActivity(intent);
					}
					else if (targetActivity.equals(FlashcardActivity.class.getName()))
					{
						Intent intent = new Intent(FilterActivity.this,
																		FlashcardActivity.class);
						intent.putExtra(EXTRA_FILTER, filter);
						intent.putExtra(FlashcardActivity.TEST_TYPE, FlashcardActivity.TEST_FLASHCARD);
						intent.putExtra(FlashcardActivity.GRADUATED_MODE, mRadioLeitner.isChecked());
						intent.putExtra(FlashcardActivity.LEARN_MODE, mRadioLearn.isChecked());
						startActivity(intent);
					}
				}
			}
		);
		List<View> views = new ArrayList<View>();
		views.add(doneButton);
		Gradient.colorize(views, 1);

		// spinner start:
		spinnerStart = (Spinner) findViewById(R.id.spinner_start);
		adapterStart = ArrayAdapter.createFromResource(this, R.array.leitner_levels,
				                                       android.R.layout.simple_spinner_item);
		adapterStart.setDropDownViewResource(android.R.layout.simple_spinner_dropdown_item);
		spinnerStart.setAdapter(adapterStart);

		// Get the preferences for the spinnerStart:
		if (targetActivity.equals(MultipleChoiceActivity.class.getName()))
			spinnerStart.setSelection(prefForThisActivity.getInt("spinnerStartStateForMultipleChoice", 0));
		else if (targetActivity.equals(WritingActivity.class.getName()))
			spinnerStart.setSelection(prefForThisActivity.getInt("spinnerStartStateForWriting", 0));
		else if (targetActivity.equals(FlashcardActivity.class.getName()))
			spinnerStart.setSelection(prefForThisActivity.getInt("spinnerStartStateForFlashcard", 0));

		// Set the preferences for the spinnerStart:
		spinnerStart.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener()
			{
				public void onItemSelected(AdapterView<?> parent, View view, int pos, long id)
				{
					if (targetActivity.equals(MultipleChoiceActivity.class.getName()))
						prefEditor.putInt("spinnerStartStateForMultipleChoice", pos);
					else if (targetActivity.equals(WritingActivity.class.getName()))
						prefEditor.putInt("spinnerStartStateForWriting", pos);
					else if (targetActivity.equals(FlashcardActivity.class.getName()))
						prefEditor.putInt("spinnerStartStateForFlashcard", pos);
					prefEditor.commit();
				}

				public void onNothingSelected(AdapterView<?> parent) {
					// nothing
				}
			}
		);

		// spinner end:
		spinnerEnd = (Spinner) findViewById(R.id.spinner_end);
		adapterEnd = ArrayAdapter.createFromResource(this, R.array.leitner_levels,
			                                         android.R.layout.simple_spinner_item);
		adapterEnd.setDropDownViewResource(android.R.layout.simple_spinner_dropdown_item);
		spinnerEnd.setAdapter(adapterEnd);

		// Get the preferences for the spinnerEnd:
		if (targetActivity.equals(MultipleChoiceActivity.class.getName()))
			spinnerEnd.setSelection(prefForThisActivity.getInt("spinnerEndStateForMultipleChoice", 7));
		else if (targetActivity.equals(WritingActivity.class.getName()))
			spinnerEnd.setSelection(prefForThisActivity.getInt("spinnerEndStateForWriting", 7));
		else if (targetActivity.equals(FlashcardActivity.class.getName()))
			spinnerEnd.setSelection(prefForThisActivity.getInt("spinnerEndStateForFlashcard", 7));

		// Set the preferences for the spinnerEnd:
		spinnerEnd.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener()
			{
				public void onItemSelected(AdapterView<?> parent, View view, int pos, long id)
				{
					if (targetActivity.equals(MultipleChoiceActivity.class.getName()))
						prefEditor.putInt("spinnerEndStateForMultipleChoice", pos);
					else if (targetActivity.equals(WritingActivity.class.getName()))
						prefEditor.putInt("spinnerEndStateForWriting", pos);
					else if (targetActivity.equals(FlashcardActivity.class.getName()))
						prefEditor.putInt("spinnerEndStateForFlashcard", pos);
					prefEditor.commit();
				}

				public void onNothingSelected(AdapterView<?> parent)
				{
					// nothing
				}
			}
		);
	}

	/**
	 * Get the part of the title which is regard to the question and answer languages.
	 */
	private String getLanguagePartOfTitle()
	{
		String questionLangId = Config.getQuestionLangId();
		String answerLangId = Config.getAnswerLangId();

		String questionLangName = Config.lastData.identifiers.get(questionLangId).name;
		String answerLangName = Config.lastData.identifiers.get(answerLangId).name;

		questionLangName = questionLangId + "-" + questionLangName.substring(0, 2);
		answerLangName = answerLangId + "-" + answerLangName.substring(0, 2);

		String langString = String.format("(%s? » %s)", questionLangName, answerLangName);
		return langString;
	}

	/**
	 * In order to set the title in the ActionBar.
	 */
	private void setTitleInActionBar()
	{
		String languagePart = getLanguagePartOfTitle();

		if (targetActivity.equals(MultipleChoiceActivity.class.getName()))
			mTextViewForActionBar.setText(getResources().getString(R.string.menu_multiple_choice) + " " + languagePart);
		else if (targetActivity.equals(WritingActivity.class.getName()))
			mTextViewForActionBar.setText(getResources().getString(R.string.menu_writing) + " " + languagePart);
		else if (targetActivity.equals(FlashcardActivity.class.getName()))
			mTextViewForActionBar.setText(getResources().getString(R.string.menu_flashcards) + " " + languagePart);
	}

	@Override
	public boolean onCreateOptionsMenu(Menu menu)
	{
		MenuInflater inflater = getMenuInflater();
		inflater.inflate(R.menu.filter_menu, menu);

		// If there is only one language defined in the kvtml object:
		if (Config.lastData.identifiers.size() == 1)
		{
			// Hide the menu for swapping languages:
			MenuItem item = menu.findItem(R.id.option_swap_languages);
			item.setVisible(false);

			// Hide the menus for chaging question language and answer language:
			item = menu.findItem(R.id.option_change_answer_lang);
			item.setVisible(false);
			item = menu.findItem(R.id.option_change_question_lang);
			item.setVisible(false);
		}

		// If there are two or more languages defined in the kvtml object:
		else
		{
			// Show the menu for swapping languages:
			MenuItem item = menu.findItem(R.id.option_swap_languages);
			item.setVisible(true);

			// Show the menus for chaging question language and answer language:
			item = menu.findItem(R.id.option_change_answer_lang);
			item.setVisible(true);
			item = menu.findItem(R.id.option_change_question_lang);
			item.setVisible(true);
		}

		// In case there isn't any lesson in the kvtml object:
		if (Config.lastData.lessons.isEmpty())
		{
			MenuItem item = (MenuItem) menu.findItem(R.id.option_toggle_lessons);
			item.setVisible(false);
		}

		return super.onCreateOptionsMenu(menu);
	}

	@Override
	public boolean onPrepareOptionsMenu(Menu menu)
	{
		// In case there is at least one selected lesson:
		MenuItem item = (MenuItem) menu.findItem(R.id.option_save_collection_of_lessons);
		item.setVisible(Config.lastData.areInpracticeLessons());

		// In case there is at least one collection of lesson in the kvtml object:
		item = (MenuItem) menu.findItem(R.id.option_load_collection_of_lessons);
		item.setVisible(! Config.lastData.collectionsOfLessons.values().isEmpty());

		item = (MenuItem) menu.findItem(R.id.option_modify_collection_of_lessons);
		item.setVisible(! Config.lastData.collectionsOfLessons.values().isEmpty());

		// In case there is at least one collection of entries in the kvtml object:
		item = (MenuItem) menu.findItem(R.id.option_practice_collection_of_entries);
		item.setVisible(! Config.lastData.collectionsOfEntries.values().isEmpty());

		item = (MenuItem) menu.findItem(R.id.option_modify_collection_of_entries);
		item.setVisible(! Config.lastData.collectionsOfEntries.values().isEmpty());

		return super.onPrepareOptionsMenu(menu);
	}

	@Override
	public boolean onOptionsItemSelected(MenuItem item)
	{
		switch (item.getItemId())
		{
			case R.id.option_swap_languages:
				prepareLanguageDialogs(SWAP_LANGUAGE_DIALOG);
				return true;

			case R.id.option_change_answer_lang:
				prepareLanguageDialogs(ANSWER_LANGUAGE_DIALOG);
				return true;

			case R.id.option_change_question_lang:
				prepareLanguageDialogs(QUESTION_LANGUAGE_DIALOG);
				return true;

			case R.id.option_save:
				KvtmlWriter saver = new KvtmlWriter(Config.lastData, Config.inputFile.getPath());
				saver.save();
				return true;

			case R.id.option_toggle_lessons:
				HashMap<String, Kvtml.Lesson> lessons = Config.lastData.lessons;
				Boolean state = !lessons.get("0").inpractice;
				for (Kvtml.Lesson lesson : lessons.values())
				{
					lesson.inpractice = state;
					for (Kvtml.Lesson sublesson : lesson.subLessons.values())
					{
						sublesson.inpractice = state;
					}
				}
				mExpListAdptr.notifyDataSetChanged();
				return true;

			case R.id.option_save_collection_of_lessons:
				saveCollectionOfLessons();
				return true;

			case R.id.option_load_collection_of_lessons:
				loadCollectionOfLessons();
				return true;

			case R.id.option_modify_collection_of_lessons:
				modifyCollectionOfLessons();
				return true;

			case R.id.option_practice_collection_of_entries:
				practiceCollectionOfEntries();
				return true;

			case R.id.option_modify_collection_of_entries:
				modifyCollectionOfEntries();
				return true;
		}
		return false;
	}

	private void saveCollectionOfLessons()
	{
		CollectionBrowserDialog.data = Config.lastData;
		Intent i = new Intent(FilterActivity.this, CollectionBrowserDialog.class);
		i.putExtra("doSort", true);
		i.putExtra("typeOfCollection", CollectionBrowserDialog.SAVE_COLLECTION_OF_LESSONS);
		startActivity(i);
	}

	private void loadCollectionOfLessons()
	{
		CollectionBrowserDialog.data = Config.lastData;
		Intent i = new Intent(FilterActivity.this, CollectionBrowserDialog.class);
		i.putExtra("doSort", true);
		i.putExtra("typeOfCollection", CollectionBrowserDialog.LOAD_COLLECTION_OF_LESSONS);
		startActivityForResult(i, REQUEST_CODE_FROM_DIALOG_FOR_COLLECTIONS_LOAD_LESSONS);
	}

	private void modifyCollectionOfLessons()
	{
		CollectionBrowserDialog.data = Config.lastData;
		Intent i = new Intent(FilterActivity.this, CollectionBrowserDialog.class);
		i.putExtra("doSort", true);
		i.putExtra("typeOfCollection", CollectionBrowserDialog.MODIFY_COLLECTION_OF_LESSONS);
		startActivity(i);
	}

	private void practiceCollectionOfEntries()
	{
		if (mRadioLeitner.isChecked())
			new AlertDialog.Builder(this)
					.setTitle(R.string.practice_collection_of_entries_graduade_mode)
					.setMessage(R.string.practice_collection_of_entries_graduade_mode_question)
					.setCancelable(true)
					.setPositiveButton(R.string.button_yes,
							new DialogInterface.OnClickListener()
							{
								@Override
								public void onClick(DialogInterface dialog, int id)
								{
									practiceCollectionOfEntriesAfterConfirmation();
								}
							}
					)
					.setNegativeButton(R.string.button_no,
							new DialogInterface.OnClickListener()
							{
								@Override
								public void onClick(DialogInterface dialog, int id)
								{
									dialog.cancel();
								}
							}
					)
					.create()
					.show();
		else
			practiceCollectionOfEntriesAfterConfirmation();
	}

	private void practiceCollectionOfEntriesAfterConfirmation()
	{
		CollectionBrowserDialog.data = Config.lastData;
		Intent i = new Intent(FilterActivity.this, CollectionBrowserDialog.class);
		i.putExtra("doSort", true);
		i.putExtra("typeOfCollection", CollectionBrowserDialog.PRACTICE_COLLECTION_OF_ENTRIES);
		startActivityForResult(i, REQUEST_CODE_FROM_DIALOG_FOR_COLLECTIONS_PRACTICE_WORDS);
	}

	private void modifyCollectionOfEntries()
	{
		CollectionBrowserDialog.data = Config.lastData;
		Intent i = new Intent(FilterActivity.this, CollectionBrowserDialog.class);
		i.putExtra("doSort", true);
		i.putExtra("typeOfCollection", CollectionBrowserDialog.MODIFY_COLLECTION_OF_ENTRIES);
		startActivity(i);
	}

	/**
	 * Handles the results returned by dialogs for collections of lessons and
	 * collections of entries:.
	 */
	@Override
	protected void onActivityResult(int requestCode, int resultCode, Intent data)
	{
		if (data == null)
			return;
		if (resultCode != Activity.RESULT_OK)
			return;

		switch (requestCode)
		{
			case REQUEST_CODE_FROM_DIALOG_FOR_COLLECTIONS_LOAD_LESSONS:
				FilterActivity.this.recreate();
				break;
			case REQUEST_CODE_FROM_DIALOG_FOR_COLLECTIONS_PRACTICE_WORDS:
				nameOfCollectionOfEntriesForFilter = data.getStringExtra("name_of_collection_of_entries");
				doneButton.performClick();
				break;
		}
	}

	private void prepareLanguageDialogs(int id)
	{
		final Object[] languagesIds = Config.lastData.identifiers.keySet().toArray();

		if (languagesIds.length < 2)
			return;

		switch (id)
		{
			case SWAP_LANGUAGE_DIALOG:
				showSwapLanguagesDialog(languagesIds);
				break;
			case QUESTION_LANGUAGE_DIALOG:
				showQuestionLanguageDialog(languagesIds);
				break;
			case ANSWER_LANGUAGE_DIALOG:
				showAnswerLanguageDialog(languagesIds);
				break;
		}
	}

	/**
	 * Only called whether there are exactly two languages in the kvtml collection.
	 */
	protected void showSwapLanguagesDialog(final Object[] languagesIds)
	{
		new AlertDialog.Builder(this)
				.setTitle(R.string.menu_swap_languages)
				.setMessage(R.string.message_swap_languages)
				.setCancelable(true)
				.setPositiveButton(R.string.button_yes,
						new DialogInterface.OnClickListener()
						{
							@Override
							public void onClick(DialogInterface dialog, int id)
							{
								String oldAnswer = Config.getAnswerLangId();

								Config.setAnswerLangId(Config.getQuestionLangId());
								Config.setQuestionLangId(oldAnswer);

								// TODO: this is a workaround needed in order to resolve
								//       the issue that the scrollable text of the Action Bar
								//       doesn't update really when create() is executed again
								//       through recreate(). The issue appends only when the text
								//       is scrollable.
								// We set now the updated title in the ActionBar before recreate():
								setTitleInActionBar();

								// Recreate this Activity:
								FilterActivity.this.recreate();
							}
						}
				)
				.setNegativeButton(R.string.button_no,
						new DialogInterface.OnClickListener()
						{
							@Override
							public void onClick(DialogInterface dialog, int id)
							{
								dialog.cancel();
							}
						}
				)
				.create()
				.show();
	}

	/**
	 * Only called whether there are three or more languages in the kvtml collection.
	 */
	protected void showQuestionLanguageDialog(final Object[] languagesIds)
	{
		// The array formed by the name of all the languages present in the kvtml object:
		final CharSequence[] qitems = new CharSequence[languagesIds.length];
		for (int i = 0, mi = languagesIds.length; i < mi; i++)
			qitems[i] = Config.lastData.identifiers.get(languagesIds[i]).name;

		// We chech the current question language:
		int checked = 0;
		String qid = Config.getQuestionLangId();
		for (int i = 0; i < languagesIds.length; i++)
			if (String.valueOf(languagesIds[i]).equals(qid))
			{
				checked = i;
				break;
			}

		// The dialog:
		new AlertDialog.Builder(this)
                .setTitle(R.string.message_change_question_language)
				.setSingleChoiceItems(qitems, checked,
						new DialogInterface.OnClickListener()
						{
							@Override
							public void onClick(DialogInterface dialog, int which)
							{
							}
						}
				)
				.setPositiveButton(R.string.button_ok,
						new DialogInterface.OnClickListener()
						{
							@Override
							public void onClick(DialogInterface dialog, int id)
							{
								int selectedPosition = ((AlertDialog)dialog).getListView().getCheckedItemPosition();
								Config.setQuestionLangId(String.valueOf(languagesIds[selectedPosition]));

								// TODO: this is a workaround needed in order to resolve
								//       the issue that the scrollable text of the Action Bar
								//       doesn't update really when create() is executed again
								//       through recreate(). The issue appends only when the text
								//       is scrollable.
								// We set now the updated title in the ActionBar before recreate():
								setTitleInActionBar();

								// Recreate this Activity:
								FilterActivity.this.recreate();
							}
						}
				)
				.setNegativeButton(R.string.button_cancel,
						new DialogInterface.OnClickListener()
						{
							@Override
							public void onClick(DialogInterface dialog, int id)
							{
								dialog.cancel();
							}
						}
				)
				.create()
				.show();
	}

	/**
	 * Only called whether there are three or more languages in the kvtml collection.
	 */
	protected void showAnswerLanguageDialog(final Object[] languagesIds)
	{
		// The array formed by the name of all the languages present in the kvtml object:
		final CharSequence[] aitems = new CharSequence[languagesIds.length];
		for (int i = 0, mi = languagesIds.length; i < mi; i++)
			aitems[i] = Config.lastData.identifiers.get(languagesIds[i]).name;

		// We chech the question language in the present:
		int checked = 0;
		String aid = Config.getAnswerLangId();
		for (int i = 0; i < languagesIds.length; i++)
			if (String.valueOf(languagesIds[i]).equals(aid))
			{
				checked = i;
				break;
			}

		// The dialog:
		new AlertDialog.Builder(this)
				.setTitle(R.string.message_change_answer_language)
				.setSingleChoiceItems(aitems, checked,
						new DialogInterface.OnClickListener()
						{
							@Override
							public void onClick(DialogInterface dialog, int which)
							{
							}
						}
				)
				.setPositiveButton(R.string.button_ok,
						new DialogInterface.OnClickListener()
						{
							@Override
							public void onClick(DialogInterface dialog, int id)
							{
								int selectedPosition = ((AlertDialog)dialog).getListView().getCheckedItemPosition();
								Config.setAnswerLangId(String.valueOf(languagesIds[selectedPosition]));

								// TODO: this is a workaround needed in order to resolve
								//       the issue that the scrollable text of the Action Bar
								//       doesn't update really when create() is executed again
								//       through recreate(). The issue appends only when the text
								//       is scrollable.
								// We set now the updated title in the ActionBar before recreate():
								setTitleInActionBar();

								// Recreate this Activity:
								FilterActivity.this.recreate();
							}
						}
				)
				.setNegativeButton(R.string.button_cancel,
						new DialogInterface.OnClickListener()
						{
							@Override
							public void onClick(DialogInterface dialog, int id)
							{
								dialog.cancel();
							}
						}
				)
				.create()
				.show();
	}

	private Boolean entriesInFilter()
	{
		for (Kvtml.Lesson lesson : Config.lastData.lessons.values())
		{
			for (Kvtml.Lesson sublesson : lesson.subLessons.values())
			{
				if (sublesson.idOfEntries.size() != 0 && sublesson.inpractice)
					return true;
			}
			if (lesson.idOfEntries.size() != 0 && lesson.inpractice)
				return true;
		}
		return false;
	}
}
