/*
 * Libero Vocab
 *     An app for Android systems which allows to do practice with kvtml
 *     vocabulary files.
 *     This program is a fork of another program called "Vocab Drill" by:
 *       - Károly Kiripolszky <karcsi@ekezet.com>
 *       - Matthias Völlinger <matthias.voellinger@gmx.de>
 *
 *     Copyright (C) 2019, 2020  Lo Iacono Massimo (massimol@inventati.org)
 *
 *     This program is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package org.inventati.massimol.liberovocab.activities;

import android.support.v7.app.ActionBar;
import android.support.v7.app.AppCompatActivity;

import android.app.AlertDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.content.SharedPreferences;
import android.os.Bundle;
import android.preference.PreferenceFragment;
import android.preference.CheckBoxPreference;
import android.preference.EditTextPreference;
import android.preference.ListPreference;
import android.preference.Preference;
import android.preference.Preference.OnPreferenceChangeListener;
import android.preference.Preference.OnPreferenceClickListener;
import android.view.WindowManager.LayoutParams;

import org.inventati.massimol.liberovocab.Config;
import org.inventati.massimol.liberovocab.R;
import org.inventati.massimol.liberovocab.helpers.StatsOpenHelper;
import org.inventati.massimol.liberovocab.widgets.IntervalPreference;
import org.inventati.massimol.liberovocab.widgets.SeekBarPreference;
import org.inventati.massimol.liberovocab.widgets.AudioBaseFolderListPreference;

public class PreferencesActivity extends AppCompatActivity
{
	// This will be used in static situations as the PreferenceFragment class:
	private static Context activityContext;

    private static Preference mClearStatsPreference = null;

	@SuppressWarnings("deprecation")
	@Override
	protected void onCreate(Bundle savedInstanceState)
	{
		super.onCreate(savedInstanceState);
		getWindow().setFlags(LayoutParams.FLAG_FULLSCREEN, LayoutParams.FLAG_FULLSCREEN);

		activityContext = PreferencesActivity.this;

		// This activity has an ActionBar:
		ActionBar actionBar = getSupportActionBar();
		actionBar.setHomeButtonEnabled(true);

		// In order to set the title in the ActionBar:
		actionBar.setTitle(getResources().getString(R.string.preferences_window_label));

		// The Preferences are associated to this Activity:
		MyPreferenceFragment mypf = new MyPreferenceFragment();
		getFragmentManager().beginTransaction().replace(android.R.id.content, mypf).commit();

		updateScreen();
	}

	protected static void updateScreen()
	{
		if (mClearStatsPreference != null)
		{
			StatsOpenHelper stats = StatsOpenHelper.getInstance(activityContext, 0);
			mClearStatsPreference.setEnabled(stats.hasStats());
			stats.close();
		}
	}

	@Override
	protected void onDestroy()
	{
		Config.save();
		super.onDestroy();
	}

	@Override
	protected void onResume()
	{
		super.onResume();
		updateScreen();
	}


/**
 * The PreferenceFragment sub-class.
 */
	public static class MyPreferenceFragment extends PreferenceFragment
	{
		@Override
		public void onCreate(final Bundle savedInstanceState) {
			super.onCreate(savedInstanceState);
			addPreferencesFromResource(R.xml.preferences);

			// Clear stats preference:
			mClearStatsPreference = findPreference("clearStats");
			mClearStatsPreference.setOnPreferenceClickListener(new OnPreferenceClickListener()
															   {
																   @Override
																   public boolean onPreferenceClick(Preference preference)
																   {
																	   new AlertDialog.Builder(activityContext)
																			   .setTitle(R.string.pref_reset_stats)
																			   .setMessage(R.string.message_really_clear_global_stats)
																			   .setPositiveButton(R.string.button_yes, new DialogInterface.OnClickListener()
																					   {
																						   @Override
																						   public void onClick(DialogInterface dialog, int which)
																						   {
																							   StatsOpenHelper stats = StatsOpenHelper.getInstance(activityContext, 0);
																							   stats.clearAllStats();
																							   stats.close();
																							   updateScreen();
																						   }
																					   }
																			   )
																			   .setNegativeButton(R.string.button_no, null)
																			   .show();
																	   return true;
																   }
															   }
			);

            SeekBarPreference skbpref = null;

			// Choice number preference:
            skbpref = (SeekBarPreference) findPreference("choiceNumber");
            skbpref.setDefaultValue(Config.choiceNumber);
            skbpref.setSummaryPrefixText(R.string.summary_current_number);
            skbpref.setOnPreferenceChangeListener(new OnPreferenceChangeListener()
															 {
																 @Override
																 public boolean onPreferenceChange(Preference preference, Object newValue)
																 {
																	 Config.choiceNumber = (Integer) newValue;
																	 return true;
																 }
															 }
			);

			// Distance limit on touch preference:
            skbpref = (SeekBarPreference) findPreference("distanceLimitTouch");
            skbpref.setDefaultValue(Config.distanceLimitTouch);
            skbpref.setSummaryPrefixText(R.string.summary_current_number);
            skbpref.setOnPreferenceChangeListener(new OnPreferenceChangeListener()
															 {
																 @Override
																 public boolean onPreferenceChange(Preference preference, Object newValue)
																 {
																	 Config.distanceLimitTouch = (Integer) newValue;
																	 return true;
																 }
															 }
			);

            // Max number of recent files preference:
            skbpref = (SeekBarPreference) findPreference("maxRecentFilesNum");
            skbpref.setSummaryPrefixText(R.string.summary_current_number);
            skbpref.setDefaultValue(Config.maxRecentFilesNum);
            skbpref.setOnPreferenceChangeListener(new OnPreferenceChangeListener()
                                                                       {
                                                                           @Override
                                                                           public boolean onPreferenceChange(Preference preference, Object newValue)
                                                                           {
                                                                               Config.maxRecentFilesNum = (Integer) newValue;
                                                                               return true;
                                                                           }
                                                                       }
            );

			// Max number of items in statistics:
			skbpref = (SeekBarPreference) findPreference("numMaxStatistic");
			skbpref.setDefaultValue(Config.numMaxStatistic);
			skbpref.setSummaryPrefixText(R.string.summary_current_number);
			skbpref.setOnPreferenceChangeListener(new OnPreferenceChangeListener()
												  {
													  @Override
													  public boolean onPreferenceChange(Preference preference, Object newValue)
													  {
														  Config.numMaxStatistic = (Integer) newValue;
														  return true;
													  }
												  }
			);

			ListPreference lstpref = null;

			// Question delay preference:
			lstpref = (ListPreference) findPreference("questionDelay");
			lstpref.setValue(String.valueOf(Config.questionDelay));
			lstpref.setSummary(String.valueOf(Config.questionDelay) + " ms");
			lstpref.setOnPreferenceChangeListener(new OnPreferenceChangeListener()
												  {
													  @Override
													  public boolean onPreferenceChange(Preference preference, Object newValue)
													  {
														  preference.setSummary(String.valueOf(newValue) + " ms");
														  Config.questionDelay = Long.valueOf(String.valueOf(newValue));
														  return true;
													  }
												  }
			);

			// Message in place of the hidden text of the question preference:
			lstpref = (ListPreference) findPreference("fullOrShortMessageForHiddenQuestionText");
			lstpref.setValue(Config.getFullOrShortMessageForHiddenQuestionTextValue());

			if (Config.getFullOrShortMessageForHiddenQuestionTextValue().equals(getResources().getStringArray(R.array.pref_full_or_short_message_for_hidden_question_text_values)[0]))
				lstpref.setSummary(getResources().getStringArray(R.array.pref_full_or_short_message_for_hidden_question_text_labels)[0]);
			else if (Config.getFullOrShortMessageForHiddenQuestionTextValue().equals(getResources().getStringArray(R.array.pref_full_or_short_message_for_hidden_question_text_values)[1]))
				lstpref.setSummary(getResources().getStringArray(R.array.pref_full_or_short_message_for_hidden_question_text_labels)[1]);
			else
				lstpref.setSummary(getResources().getStringArray(R.array.pref_full_or_short_message_for_hidden_question_text_labels)[2]);

			lstpref.setOnPreferenceChangeListener(new OnPreferenceChangeListener()
												  {
													  @Override
													  public boolean onPreferenceChange(Preference preference, Object newValue)
													  {
														  Config.setFullOrShortMessageForHiddenQuestionTextValue((String) newValue);

														  if (((String) newValue).equals(getResources().getStringArray(R.array.pref_full_or_short_message_for_hidden_question_text_values)[0]))
															  preference.setSummary(getResources().getStringArray(R.array.pref_full_or_short_message_for_hidden_question_text_labels)[0]);
														  else if (((String) newValue).equals(getResources().getStringArray(R.array.pref_full_or_short_message_for_hidden_question_text_values)[1]))
															  preference.setSummary(getResources().getStringArray(R.array.pref_full_or_short_message_for_hidden_question_text_labels)[1]);
														  else
															  preference.setSummary(getResources().getStringArray(R.array.pref_full_or_short_message_for_hidden_question_text_labels)[2]);

														  return true;
													  }
												  }
			);

			AudioBaseFolderListPreference abfpref = null;

			// Base directory for audio files:
			abfpref = (AudioBaseFolderListPreference) findPreference("baseDirectoryForAudioFiles");
			abfpref.setValue(Config.getBaseDirectoryForAudioFilesValue());

			if (Config.getBaseDirectoryForAudioFilesValue().equals(getResources().getStringArray(R.array.pref_base_directory_for_audio_files_values)[0]))
				abfpref.setSummary(getResources().getStringArray(R.array.pref_base_directory_for_audio_files_labels)[0]);
			else if (Config.getBaseDirectoryForAudioFilesValue().equals(getResources().getStringArray(R.array.pref_base_directory_for_audio_files_values)[1]))
				abfpref.setSummary(getResources().getStringArray(R.array.pref_base_directory_for_audio_files_labels)[1]);
			else
				abfpref.setSummary(getResources().getStringArray(R.array.pref_base_directory_for_audio_files_labels)[2] + "\n\n" + Config.getBaseDirectoryForAudioFilesOtherPath());

			abfpref.setOnPreferenceChangeListener(new OnPreferenceChangeListener()
												  {
													  @Override
													  public boolean onPreferenceChange(Preference preference, Object newValue)
													  {
														  Config.setBaseDirectoryForAudioFilesValue((String) newValue);

														  if (((String) newValue).equals(getResources().getStringArray(R.array.pref_base_directory_for_audio_files_values)[0]))
															  preference.setSummary(getResources().getStringArray(R.array.pref_base_directory_for_audio_files_labels)[0]);
														  else if (((String) newValue).equals(getResources().getStringArray(R.array.pref_base_directory_for_audio_files_values)[1]))
															  preference.setSummary(getResources().getStringArray(R.array.pref_base_directory_for_audio_files_labels)[1]);
														  else
															  preference.setSummary(getResources().getStringArray(R.array.pref_base_directory_for_audio_files_labels)[2] + "\n\n" + Config.getBaseDirectoryForAudioFilesOtherPath());

														  return true;
													  }
												  }
			);

            IntervalPreference ivpref = null;

            // Setup Leitner-System preferences:
			for (int i = 0, I = Config.practiceIntervals.length; i < I; i++)
			{
				ivpref = (IntervalPreference) findPreference(String.format("leitner%s", i + 1));
				ivpref.setDefaultValue(Config.practiceIntervals[i]);
				ivpref.setOnPreferenceChangeListener(new OnPreferenceChangeListener()
													 {
														 @Override
														 public boolean onPreferenceChange(Preference preference, Object newValue)
														 {
															 int idx = Integer.valueOf(String.valueOf(preference.getKey()).split("leitner")[1]);
															 Config.practiceIntervals[idx - 1] = String.valueOf(newValue);
															 return true;
														 }
													 }
				);
			}

			CheckBoxPreference cbpref = null;

			// Choice number preference:
			cbpref = (CheckBoxPreference) findPreference("forceChoiceNumber");
			cbpref.setChecked(Config.forceChoiceNumber);
			cbpref.setOnPreferenceChangeListener(new OnPreferenceChangeListener()
												 {
													 @Override
													 public boolean onPreferenceChange(Preference preference, Object newValue)
													 {
														 Config.forceChoiceNumber = Boolean.valueOf(newValue.toString());
														 return true;
													 }
												 }
			);

			// Question with sound preference:
			cbpref = (CheckBoxPreference) findPreference("forceQuestionWithSound");
			cbpref.setChecked(Config.forceQuestionWithSound);
			cbpref.setOnPreferenceChangeListener(new OnPreferenceChangeListener()
												 {
													 @Override
													 public boolean onPreferenceChange(Preference preference, Object newValue)
													 {
														 Config.forceQuestionWithSound = Boolean.valueOf(newValue.toString());
														 return true;
													 }
												 }
			);

			// Replace separator char:
			cbpref = (CheckBoxPreference) findPreference("replaceSeparatorChar");
			cbpref.setChecked(Config.replaceSeparatorChar);
			cbpref.setOnPreferenceChangeListener(new OnPreferenceChangeListener()
												 {
													 @Override
													 public boolean onPreferenceChange(Preference preference, Object newValue)
													 {
														 Config.replaceSeparatorChar = Boolean.valueOf(newValue.toString());
														 return true;
													 }
												 }
			);

			// Show diagnostic message for audio:
			cbpref = (CheckBoxPreference) findPreference("showDiagnosticMessageForAudio");
			cbpref.setChecked(Config.showDiagnosticMessageForAudio);
			cbpref.setOnPreferenceChangeListener(new OnPreferenceChangeListener()
												 {
													 @Override
													 public boolean onPreferenceChange(Preference preference, Object newValue)
													 {
														 Config.showDiagnosticMessageForAudio = Boolean.valueOf(newValue.toString());
														 return true;
													 }
												 }
			);

			// Scan directories preference:
			cbpref = (CheckBoxPreference) findPreference("scanDirectories");
			cbpref.setChecked(Config.scanDirectories);
			cbpref.setOnPreferenceChangeListener(new OnPreferenceChangeListener()
												 {
													 @Override
													 public boolean onPreferenceChange(Preference preference, Object newValue)
													 {
														 //Config.scanDirectories = String.valueOf(newValue).equals("1");
														 Config.scanDirectories = Boolean.valueOf(newValue.toString());
														 return true;
													 }
												 }
			);

			// Confirmations exit preference:
			cbpref = (CheckBoxPreference) findPreference("confirmExit");
			cbpref.setChecked(Config.confirmExit);
			cbpref.setOnPreferenceChangeListener(new OnPreferenceChangeListener()
												 {
													 @Override
													 public boolean onPreferenceChange(Preference preference, Object newValue)
													 {
														 //Config.confirmExit = String.valueOf(newValue).equals("1");
														 Config.confirmExit = Boolean.valueOf(newValue.toString());
														 return true;
													 }
												 }
			);

			// Confirmations progress preference:
			cbpref = (CheckBoxPreference) findPreference("confirmProgress");
			cbpref.setChecked(Config.confirmProgress);
			cbpref.setOnPreferenceChangeListener(new OnPreferenceChangeListener()
												 {
													 @Override
													 public boolean onPreferenceChange(Preference preference, Object newValue)
													 {
														 //Config.confirmProgress = String.valueOf(newValue).equals("1");
														 Config.confirmProgress = Boolean.valueOf(newValue.toString());
														 return true;
													 }
												 }
			);

			EditTextPreference etxpref = null;

			// Starting string to remove in path preference:
			etxpref = (EditTextPreference) findPreference("startStringToBeRemovedInPath");
			etxpref.setText(Config.getStartStringToBeRemovedInPath());
			etxpref.setSummary(getResources().getString(R.string.pref_start_string_to_be_removed_in_path_summary) + Config.getStartStringToBeRemovedInPath() + "\"");
			etxpref.setOnPreferenceChangeListener(new OnPreferenceChangeListener()
												 {
													 @Override
													 public boolean onPreferenceChange(Preference preference, Object newValue)
													 {
														 Config.setStartStringToBeRemovedInPath(newValue.toString());
														 preference.setSummary(getResources().getString(R.string.pref_start_string_to_be_removed_in_path_summary) + Config.getStartStringToBeRemovedInPath() + "\"");
														 return true;
													 }
												 }
			);

            Preference pref = null;

			// Clear cache preference:
            pref = findPreference("clearCache");
            pref.setOnPreferenceClickListener(new OnPreferenceClickListener()
															   {
																   @Override
																   public boolean onPreferenceClick(Preference preference)
																   {
																	   new AlertDialog.Builder(activityContext)
																			   .setTitle(R.string.pref_clear_cache)
																			   .setMessage(R.string.message_really_clear_cache)
																			   .setPositiveButton(R.string.button_yes, new DialogInterface.OnClickListener()
																					   {
																						   @Override
																						   public void onClick(DialogInterface dialog, int which)
																						   {
																							   Config.clearCacheDir();
																							   updateScreen();
																						   }
																					   }
																			   )
																			   .setNegativeButton(R.string.button_no, null).show();
																	   return true;
																   }
															   }
			);

			// Reset settings preference:
			pref = findPreference("clearSettings");
			pref.setOnPreferenceClickListener(new OnPreferenceClickListener()
											  {
												  @Override
												  public boolean onPreferenceClick(Preference preference)
												  {
													  new AlertDialog.Builder(activityContext)
															  .setTitle(R.string.pref_clear_settings)
															  .setMessage(R.string.message_really_clear_settings)
															  .setPositiveButton(R.string.button_yes, new DialogInterface.OnClickListener()
																	  {
																		  @Override
																		  public void onClick(DialogInterface dialog, int which)
																		  {
																			  final SharedPreferences prefs = activityContext.getSharedPreferences(Config.PREFS_FILE, MODE_PRIVATE);
																			  SharedPreferences.Editor edit = prefs.edit();
																			  edit.clear().commit();
																			  updateScreen();
																		  }
																	  }
															  )
															  .setNegativeButton(R.string.button_no, null).show();
													  return true;
												  }
											  }
			);
		}

		@Override
		public void onResume()
		{
			super.onResume();

			AudioBaseFolderListPreference abfpref = null;

			// Base directory for audio files:
			abfpref = (AudioBaseFolderListPreference) findPreference("baseDirectoryForAudioFiles");
			abfpref.setValue(Config.getBaseDirectoryForAudioFilesValue());

            if (Config.getBaseDirectoryForAudioFilesValue().equals(getResources().getStringArray(R.array.pref_base_directory_for_audio_files_values)[0]))
                abfpref.setSummary(getResources().getStringArray(R.array.pref_base_directory_for_audio_files_labels)[0]);
            else if (Config.getBaseDirectoryForAudioFilesValue().equals(getResources().getStringArray(R.array.pref_base_directory_for_audio_files_values)[1]))
                abfpref.setSummary(getResources().getStringArray(R.array.pref_base_directory_for_audio_files_labels)[1]);
            else
                abfpref.setSummary(getResources().getStringArray(R.array.pref_base_directory_for_audio_files_labels)[2] + "\n\n" + Config.getBaseDirectoryForAudioFilesOtherPath());
		}
	}
}