/*
 * Libero Vocab
 *     An app for Android systems which allows to do practice with kvtml
 *     vocabulary files.
 *     This program is a fork of another program called "Vocab Drill" by:
 *       - Károly Kiripolszky <karcsi@ekezet.com>
 *       - Matthias Völlinger <matthias.voellinger@gmx.de>
 *
 *     Copyright (C) 2019, 2020  Lo Iacono Massimo (massimol@inventati.org)
 *
 *     This program is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package org.inventati.massimol.liberovocab.dialogs;

import android.support.v4.content.ContextCompat;

import java.io.File;
import java.io.FileFilter;
import java.util.Locale;

import android.os.Bundle;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;

import org.inventati.massimol.liberovocab.R;

public class FileOpenDialog extends OpenDialog
{
	// Filter object for the directories:
	private FileFilter mDirFilter = new FileFilter()
									{
										@Override
										public boolean accept(File pathname)
										{
											if (pathname.isDirectory()
												&& !pathname.isFile()
												&& !pathname.isHidden())
												return true;
											return false;
										}
									};

	// Filter object for the files with extension kvtml:
	private FileFilter mFileFilter = new FileFilter()
									{
										@Override
										public boolean accept(File pathname)
										{
											if (pathname.isDirectory() && !pathname.isFile())
												return false;
											if (pathname.getName().toLowerCase(Locale.getDefault()).endsWith(".kvtml"))
												return true;
											return false;
										}
									};

	public class FileDialogEntry extends DialogEntry
	{
		private File mFile = null;
		private String mName = null;

		public FileDialogEntry(String path) throws NullPointerException
		{
			super(path);

			mFile = new File(path);
			if (mFile == null || !mFile.exists())
				throw new NullPointerException(String.format("Path not found: '%s'", path));

			mPath = mFile.getAbsolutePath();
			mName = mFile.getName();
		}

		@Override
		public String getName()
		{
			return mName;
		}

		@Override
		public String getParent()
		{
			return mFile.getParent();
		}

		@Override
		public DialogEntry[] getFileList()
		{
			File[] list = mFile.listFiles(mFileFilter);
			if (list == null || list.length == 0)
				return null;
			DialogEntry[] result = new DialogEntry[list.length];
			for (int i = 0, I = list.length; i < I; i++)
				result[i] = new FileDialogEntry(list[i].getPath());
			return result;
		}

		@Override
		public DialogEntry[] getFolderList()
		{
			File[] list = mFile.listFiles(mDirFilter);
			if (list == null || list.length == 0)
				return null;
			DialogEntry[] result = new DialogEntry[list.length];
			for (int i = 0, I = list.length; i < I; i++)
				result[i] = new FileDialogEntry(list[i].getPath());
			return result;
		}

		@Override
		public boolean canRead()
		{
			return mFile.canRead();
		}

		@Override
		public boolean isDirectory()
		{
			return mFile.isDirectory();
		}

		@Override
		protected boolean isSelected()
		{
			return !this.isDirectory();
		}

		@Override
		protected boolean exists()
		{
			return mFile.exists();
		}
	}

	/**
	 * Initialize File Open dialog.
	 */
	@Override
	public void onCreate(Bundle savedInstanceState) {
		onCreateThis();
		super.onCreate(savedInstanceState);
	}

	public void onCreateThis() {
		setContentView(R.layout.dialog_file_open);
	}

	protected void initialize(Bundle savedInstanceState)
	{
		setExternalStorages();
		mDefaultPath = mStoragePublicExtPath_1;

		final String path = getIntent().getStringExtra(INTENT_PATH_PARAMETER);

		try
		{
			if (path != null)
				mCurrentEntry = newEntry(path);
			else
				mCurrentEntry = newEntry(mPrefs.getString(mLastPathPrefName, mDefaultPath));
		}
		catch (NullPointerException e)
		{
			e.printStackTrace();
			System.err.println("### ERROR: Exception in FileOpenDialog ###");
			mCurrentEntry = newEntry(mDefaultPath);
		}

		// Check configuration change:
		try
		{
			if (savedInstanceState != null)
				mCurrentEntry = newEntry(savedInstanceState.getString(mLastPathPrefName));
		}
		catch (NullPointerException e)
		{
			e.printStackTrace();
			System.err.println("### ERROR: Exception in FileOpenDialog ###");
			mCurrentEntry = newEntry(mDefaultPath);
		}
	}

	@Override
	protected DialogEntry newEntry(String path) throws NullPointerException
	{
		return new FileDialogEntry(path);
	}

	private void setExternalStorages()
	{
		// In order to set the first private external storage:
		File[] file_array = ContextCompat.getExternalFilesDirs(FileOpenDialog.this, null);
		try
		{
			mStoragePrivateExtPath_1 = file_array[0].getAbsolutePath();
		}
		catch (NullPointerException e)
		{
			mStoragePrivateExtPath_1 = "/";
		}

		// In order to set the second private external storage:
		try
		{
			mStoragePrivateExtPath_2 = file_array[1].getAbsolutePath();
		}
		catch (IndexOutOfBoundsException e)
		{
			mStoragePrivateExtPath_2 = null;
		}

		// In order to set the first public external storage:
		//
		// TODO : this is a workaround in order to get the root of external public directory.
		//        We'll gladly change this in case we find a better solution.
		//
		// Note: the precedent but deprecated method was:
		//       mStoragePublicExtPath_1 = Environment.getExternalStorageDirectory().getAbsolutePath();
		try
		{
			mStoragePublicExtPath_1 = file_array[0].getParentFile().getParentFile().getParentFile().getParentFile().getAbsolutePath();
		}
		catch (NullPointerException e)
		{
			mStoragePublicExtPath_1 = "/";
		}

		// In order to set the second public external storage:
		//
		// TODO : this is a workaround in order to get the root of external public directory.
		//        We'll gladly change this in case we find a better solution.
		try
		{
			mStoragePublicExtPath_2 = file_array[1].getParentFile().getParentFile().getParentFile().getParentFile().getAbsolutePath();
		}
		catch (IndexOutOfBoundsException e)
		{
			mStoragePublicExtPath_2 = null;
		}
		catch (NullPointerException e)
		{
			mStoragePublicExtPath_2 = "/";
		}
	}

	@Override
	public boolean onCreateOptionsMenu(Menu menu)
	{
		super.onCreateOptionsMenu(menu);
		final MenuInflater inflater = getMenuInflater();
		inflater.inflate(R.menu.open_dialog_menu, menu);

        // Hide the menu for the second public external storage:
        if (mStoragePublicExtPath_2 == null)
		{
            MenuItem item = menu.findItem(R.id.option_goto_public_ext_2);
            item.setVisible(false);
        }

        // Hide the menu for the second private external storage:
        if (mStoragePrivateExtPath_2 == null)
		{
            MenuItem item = menu.findItem(R.id.option_goto_private_ext_2);
            item.setVisible(false);
        }

		return true;
	}

	@Override
	public boolean onOptionsItemSelected(MenuItem item)
	{
		switch (item.getItemId())
		{
			case R.id.option_goto_public_ext_1:
				updateList(newEntry(mStoragePublicExtPath_1));
				return true;
			case R.id.option_goto_public_ext_2:
				updateList(newEntry(mStoragePublicExtPath_2));
				return true;
			case R.id.option_goto_private_ext_1:
				updateList(newEntry(mStoragePrivateExtPath_1));
				return true;
			case R.id.option_goto_private_ext_2:
				updateList(newEntry(mStoragePrivateExtPath_2));
				return true;
			default:
				return super.onOptionsItemSelected(item);
		}
	}

	@Override
	protected String getLastPathPrefName()
	{
		return "file_browser_default_dir";
	}
}