/*
 * Libero Vocab
 *     An app for Android systems which allows to do practice with kvtml
 *     vocabulary files.
 *     This program is a fork of another program called "Vocab Drill" by:
 *       - Károly Kiripolszky <karcsi@ekezet.com>
 *       - Matthias Völlinger <matthias.voellinger@gmx.de>
 *
 *     Copyright (C) 2019, 2020  Lo Iacono Massimo (massimol@inventati.org)
 *
 *     This program is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package org.inventati.massimol.liberovocab.dialogs;

import java.io.File;
import java.util.HashMap;

import android.support.v7.app.AppCompatActivity;

import android.app.AlertDialog;
import android.content.DialogInterface;
import android.app.Activity;
import android.content.Intent;
import android.net.Uri;
import android.os.Bundle;
import android.os.Environment;
import android.util.Log;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.WindowManager;
import android.widget.ListView;
import android.widget.AdapterView;

import org.inventati.massimol.liberovocab.Config;
import org.inventati.massimol.liberovocab.R;
import org.inventati.massimol.liberovocab.adapters.RecentFilesAdapter;

public class RecentFilesDialog extends AppCompatActivity
{
	//protected String mExtDir = null;

	public abstract class RecentItem
	{
		protected String mOrigPath = null;
		protected String mPath = null;
		protected String mProto = null;

		public RecentItem(String path, String proto)
		{
			mOrigPath = path;
			mProto = proto;
			/**
			 * XXX: FUN FACT: This didn't work on 2.1 device:
			 *   mPath = path.replace(proto != null ? proto : "", "")
			 * Tested on device: XPeria X10 mini pro
			 */
			if (proto != null)
				path = path.replace(proto, "");
			mPath = path;
		}

		public String getPath()
		{
			return mPath;
		}

		public String getLocalPath()
		{
			return mPath;
		}

		public String getOrigPath()
		{
			return mOrigPath;
		}

		public String getProto()
		{
			return mProto;
		}

		public String getOrigParent()
		{
			return (null == mProto ? "" : mProto) + getParent();
		}

		public abstract String getName();

		public abstract String getParent();

		public abstract String getDisplayPath();
	}

	public class RecentFile extends RecentItem
	{
		protected File mFile;

		public RecentFile(String path, String proto)
		{
			super(path, proto);
			mFile = new File(mPath);
		}

		public RecentFile(String path)
		{
			super(path, null);
			mFile = new File(mPath);
		}

		@Override
		public String getName()
		{
			return mFile.getName();
		}

		@Override
		public String getParent()
		{
			return mFile.getParent();
		}

		@Override
		public String getDisplayPath()
		{
			String s = getLocalPath();
			//Log.d(getClass().getSimpleName(), "display (ext): " + s);
			return s;
		}
	}

	private static HashMap<String, RecentItem> sList = null;

	private RecentFilesAdapter mAdapter;

	/**
	 * The ListView object:
	 */
	private ListView list;

	@Override
	public void onCreate(Bundle savedInstanceState)
	{
		super.onCreate(savedInstanceState);
		getWindow().setFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN,
				             WindowManager.LayoutParams.FLAG_FULLSCREEN);
        setContentView(R.layout.dialog_recent_files);

		// This activity has an ActionBar:
		getSupportActionBar().setHomeButtonEnabled(true);

		//mExtDir = Environment.getExternalStorageDirectory().getAbsolutePath();

		list = (ListView) findViewById(android.R.id.list);
		list.setTextFilterEnabled(true);

		list.setOnItemClickListener(new AdapterView.OnItemClickListener()
			{
				@Override
				public void onItemClick(AdapterView<?> parent, View view, int position, long id)
				{
					listItemClick(position);
				}
			}
		);

		// The list of items is prepared:
		sList = new HashMap<String, RecentItem>();
		mAdapter = new RecentFilesAdapter(this, R.layout.item_recent_file);
		updateList();

		// The list of items is added to the ListView object:
		list.setAdapter(mAdapter);
	}

	private void updateList()
	{
		mAdapter.clear();
		RecentItem entry = null;
		String path = null;
		for (int n = 0, N = Config.recentFilesList.length; n < N; n++)
		{
			path = Config.recentFilesList[n];
			if (path.length() == 0)
				continue;
			entry = createRecentFile(path);
			mAdapter.add(entry);
		}
	}

	private RecentItem createRecentFile(String path)
	{
		if (sList.containsKey(path))
			return sList.get(path);

		RecentItem ret = null;
		ret = new RecentFile(path);
		sList.put(path, ret);
		return ret;
	}

	@Override
	public boolean onCreateOptionsMenu(Menu menu)
	{
		super.onCreateOptionsMenu(menu);
		MenuInflater inflater = getMenuInflater();
		inflater.inflate(R.menu.recent_files_menu, menu);
		return true;
	}

	@Override
	public boolean onOptionsItemSelected(MenuItem item)
	{
		switch (item.getItemId())
		{
			case R.id.menu_clear:
				new AlertDialog.Builder(this)
					.setTitle(R.string.title_clear_list_of_recent_opened_files)
					.setMessage(R.string.message_really_clear_list_of_recent_opened_files)
					.setPositiveButton(R.string.button_yes,
						new DialogInterface.OnClickListener()
						{
							@Override
							public void onClick(DialogInterface dialog, int which)
							{
								Config.clearRecentFilesList();
								finish();
							}
						}).setNegativeButton(R.string.button_no, null).show();
				return true;

			default:
				return super.onOptionsItemSelected(item);
		}
	}

	/**
	 * When a list item is selected.
	 */
	protected void listItemClick(int position)
	{
		final Intent data = new Intent();
		final RecentItem item = createRecentFile(Config.recentFilesList[position]);
		data.setData(Uri.parse(item.getLocalPath()));
		setResult(Activity.RESULT_OK, data);
		finish();
	}
}
