/*
 * Libero Vocab
 *     An app for Android systems which allows to do practice with kvtml
 *     vocabulary files.
 *     This program is a fork of another program called "Vocab Drill" by:
 *       - Károly Kiripolszky <karcsi@ekezet.com>
 *       - Matthias Völlinger <matthias.voellinger@gmx.de>
 *
 *     Copyright (C) 2019, 2020  Lo Iacono Massimo (massimol@inventati.org)
 *
 *     This program is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package org.inventati.massimol.liberovocab.helpers;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import android.content.Context;
import android.content.res.AssetManager;
import android.widget.Toast;

import org.inventati.massimol.liberovocab.R;

public class AssetsUtils
{
	/**
	 * Return the text contained in a file into the assets.
	 * @param context
	 * @param filename the name of the asset file.
	 * @return The text of the asset file.
	 * @throws IOException
	 */
	public static String readText(Context context, String filename) throws IOException
	{
		String ret = null;

		AssetManager am = context.getAssets();
		InputStream is = am.open(filename);
		final int size = is.available();
		byte[] buffer = new byte[size];
		is.read(buffer);
		is.close();
		ret = new String(buffer);

		return ret;
	}

	/**
	 * Return the array whose elements are the names of all the files contained into
	 * a sub-directory of the assets.
	 * @param context
	 * @param path The path of the sub-directory of assets. It is relative, starting from assets.
	 *             Must indicate a directory.
	 * @return The array of String, one for each file contained into the sub-directory of assets.
	 */
	public static String[] listFiles(Context context, String path)
	{
		AssetManager am = context.getAssets();
		try
		{
			return am.list(path);
		}
		catch (IOException e)
		{
			e.printStackTrace();
			System.err.println("### ERROR: Exception in AssetsUtils ###");
			return null;
		}
	}

	/**
	 * Copies, eventually recursively, all the files into an original sub-folder of assets
	 * into a generic (that is at level of the system and not of the assets) destination folder.
	 *
	 * @param context
	 * @param assetsSubFolder The String corresponding to the sub-folder of assets, expressed
	 *                        relatively to assets. For example "aaa/bbb". Must correspond to
	 *                        a folder.
	 * @param destinationFolder A File object. It will attempt to create it as a folder.
	 * @return false if something went bad, otherwise true.
	 */
	public static Boolean copyFromAssetsToFolderRecursively(Context context,
															String assetsSubFolder,
															File destinationFolder)
	{
		// Preliminary controls:

		// The list of files and sub-folders in the origin assets sub-folder.
		// En passant we control whether the origin assets sub-folder is really a
		// folder and it isn't empty:
		String[] listFilesInAssetsFolder = listFiles(context, assetsSubFolder);
		if (listFilesInAssetsFolder.length == 0)
		{
			System.err.println("### ERROR: in AssetsUtils: the assetsSubFolder is not a directory or it is empty. ###");
			return false;
		}

		// Controls on the destination folder:
		try
		{
			if (!destinationFolder.exists())
				destinationFolder.mkdir();
			else if (!destinationFolder.isDirectory())
			{
				String errorMessage = context.getResources().getString(R.string.error_copy_folder_exist_but_is_files) + "\n\n" + destinationFolder;
				System.err.println("### ERROR: in AssetsUtils: " + errorMessage + " ###");
				Toast.makeText(context, errorMessage, Toast.LENGTH_LONG).show();
				return false;
			}
		}
		catch (SecurityException e)
		{
			e.printStackTrace();
			System.err.println("### ERROR: Exception in AssetsUtils ###");
			return false;
		}

		// Both the origin assets sub-directory and the destination are folders.
		// We can continue:

		// The AssetManager object:
		AssetManager am = context.getAssets();

		// Each file in the origin folder is copied into the destination folder:
		File newFile;
		InputStream is;
		OutputStream os;
		for (int i = 0; i < listFilesInAssetsFolder.length; i++) {
			String assetsFilePath = assetsSubFolder + "/" + listFilesInAssetsFolder[i];
			try
			{
				newFile = new File(destinationFolder, listFilesInAssetsFolder[i]);

				// We handle here the case where the file is, in truth, a directory:
				try
				{
					// We try to open as a file:
					is = am.open(assetsFilePath);
				}
				catch (FileNotFoundException e)
				{
					// Ok! most likely this is not a file but, in truth, a directory:

					// We call recursively this method itself:
					Boolean wasCopyOK = copyFromAssetsToFolderRecursively(context, assetsFilePath, newFile);
					if (wasCopyOK)
						continue;
					else
						return false;
				}

				os = new FileOutputStream(newFile);

				final int size = is.available();
				byte[] buffer = new byte[size];

				is.read(buffer);
				os.write(buffer);
				is.close();
				os.close();
			}
			catch (IOException e)
			{
				e.printStackTrace();
				System.err.println("### ERROR: Exception in AssetsUtils ###");
				return false;
			}
			catch (NullPointerException e)
			{
				e.printStackTrace();
				System.err.println("### ERROR: Exception in AssetsUtils ###");
				return false;
			}
			catch (SecurityException e)
			{
				e.printStackTrace();
				System.err.println("### ERROR: Exception in AssetsUtils ###");
				return false;
			}
		}

		// If all went ok:
		return true;
	}

	/**
	 * Copies, eventually recursively, all the files into an original folder
	 * into a destination folder. Note that origin and destination folders are generics
	 * and don't regard the assets resource.
	 *
	 * @param originFolder Must be a folder.
	 * @param destinationFolder Must be a folder.
	 * @return false if something went bad, otherwise true.
	 */
	public static Boolean copyFromFolderToFolderRecursively(File originFolder, File destinationFolder)
	{
		// A preliminary control:
		if (!originFolder.isDirectory())
		{
			System.err.println("### ERROR: in AssetsUtils: the original folder is not a directory. ###");
			return false;
		}
		else if (!destinationFolder.isDirectory())
		{
			System.err.println("### ERROR: in AssetsUtils: the destination folder is not a directory. ###");
			return false;
		}

		// Both the File are folder. We can continue:
		try
		{
			// The list of the files and subfolders in the origin folder:
			File[] listFilesInOriginFolder = originFolder.listFiles();

			// Each file in the origin folder is copied into the destination folder:
			File newFile;
			InputStream is;
			OutputStream os;
			for (int i = 0; i < listFilesInOriginFolder.length; i++) {
				newFile = new File(destinationFolder, listFilesInOriginFolder[i].getName());

				is = new FileInputStream(listFilesInOriginFolder[i]);

				os = new FileOutputStream(newFile);

				final int size = is.available();
				byte[] buffer = new byte[size];

				is.read(buffer);
				os.write(buffer);
				is.close();
				os.close();
			}
		}
		catch (IOException e)
		{
			e.printStackTrace();
			System.err.println("### ERROR: Exception in AssetsUtils ###");
			return false;
		}
		catch (NullPointerException e)
		{
			e.printStackTrace();
			System.err.println("### ERROR: Exception in AssetsUtils ###");
			return false;
		}
		catch (SecurityException e)
		{
			e.printStackTrace();
			System.err.println("### ERROR: Exception in AssetsUtils ###");
			return false;
		}

		// If all went ok:
		return true;
	}
}