/*
 * Libero Vocab
 *     An app for Android systems which allows to do practice with kvtml
 *     vocabulary files.
 *     This program is a fork of another program called "Vocab Drill" by:
 *       - Károly Kiripolszky <karcsi@ekezet.com>
 *       - Matthias Völlinger <matthias.voellinger@gmx.de>
 *
 *     Copyright (C) 2019, 2020  Lo Iacono Massimo (massimol@inventati.org)
 *
 *     This program is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package org.inventati.massimol.liberovocab.helpers;

import java.io.Serializable;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;

import android.annotation.SuppressLint;
import android.util.Log;

import org.inventati.massimol.liberovocab.Config;
import org.inventati.massimol.liberovocab.kvtml.Kvtml;
import org.inventati.massimol.liberovocab.kvtml.interval.ParsedInterval;

/**
 * provide a simple way to filter the vocabulary data
 */
@SuppressLint("UseSparseArrays")
public class EntriesFilter implements Serializable
{
	/**
	 * Used for serialization.
	 */
	private static final long serialVersionUID = -2476363632100310243L;

	/*
	 * The currently implemented filters.
	 *
	 * Note that the filter based on the lessons is considered predefined and
	 * not an option and thus we don't consider a flag for this filter.
	 */

	/**
	 * Filter by the minimum number of errors a word has to have.
	 */
	public static final int FILTER_FAILED = 0;

	/**
	 * Filter only the newly entered words.
	 */
	public static final int FILTER_NEW_WORDS = 1;

	/**
	 * Filter all words that have been practiced before a given date.
	 */
	public static final int FILTER_LAST_PRACTICED = 2;

	/**
	 * filters according to the grades of the Leitner-system.
	 */
	public static final int FILTER_LEITNER = 3;

	/**
	 * Filter a range of grades.
	 */
	public static final int FILTER_GRADE_RANGE = 4;

	/**
	 * Filter according a collection of entries.
	 */
	public static final int FILTER_COLLECTION_OF_ENTRIES = 5;

	/**
	 * Field where store the filters witch must be applied.
	 */
	private HashMap<Integer, String> mFilterList;

	/**
	 * Constructor.
	 */
	public EntriesFilter()
	{
		mFilterList = new HashMap<Integer, String>();
	}

	/**
	 * Add a filter to the filter list.
	 * @param filterType The Integer index of the filter to add.
	 * @param filterValue The String value of the filter to add.
	 */
	public void add(int filterType, String filterValue)
	{
		mFilterList.put(filterType, filterValue);
	}

	/**
	 * Remove a filter to the filter list according to the passed key.
	 * @param filterType The Integer index of this filter to remove.
	 * @return The value of the element removed.
	 */
	public String remove(int filterType)
	{
		return mFilterList.remove(filterType);
	}

	/**
	 * Checks if the specified filter has to be applied.
	 * @param filterType The Integer index of the filter witch to know if has to be applied of.
	 * @return True if the specified filter has to be applied.
	 */
	public Boolean hasFilter(int filterType)
	{
		return mFilterList.containsKey(filterType);
	}

	/**
	 * Get the value of a specified filter.
	 * @param filterType The Integer index of the filter witch to get the value of.
	 * @return The value of the specified filter.
	 */
	public String getValue(int filterType)
	{
		return mFilterList.get(filterType);
	}

	/**
	 * Method to filter all entries that fall into the selected lessons.
	 *
	 * @param lessons
	 * @param source for the entries
	 * @return filtered entries
	 */
	private HashMap<String, Kvtml.Entry> filterLessons(HashMap<String, Kvtml.Lesson> lessons, HashMap<String, Kvtml.Entry> source)
	{
		HashMap<String, Kvtml.Entry> entries = new HashMap<String, Kvtml.Entry>();

		for (Kvtml.Lesson lesson : lessons.values())
		{
			if (lesson.inpractice)
			{
				for (String key : lesson.idOfEntries)
				{
					if (!entries.containsKey(key))
					{
						entries.put(key, source.get(key));
					}
				}
			}
			for (Kvtml.Lesson sublesson : lesson.subLessons.values())
			{
				if (sublesson.inpractice)
				{
					for (String key : sublesson.idOfEntries)
					{
						if (!entries.containsKey(key))
						{
							entries.put(key, source.get(key));
						}
					}
				}
			}
		}
		return entries;
	}

	/**
	 * Method to filter by the Leitner system. To be called after filterLessons
	 * every grade corresponds to an time interval that has to elapse, before the
	 * entry is practiced again.
	 */
	@SuppressLint("SimpleDateFormat")
	private void filterLeitner(HashMap<String, Kvtml.Entry> entries)
	{
		// ProgressDialog progressDialog = new ProgressDialog(context);
		// progressDialog.setProgress(0);
		// progressDialog.setMax(entries.size());
		// progressDialog.setProgressStyle(ProgressDialog.STYLE_HORIZONTAL);
		// progressDialog.setMessage("Filtering Leitner System ...");
		// progressDialog.show();

		// initialize calendars
		Calendar[] compareCals = new Calendar[7];
		SimpleDateFormat df = new SimpleDateFormat(Kvtml.GRADE_DATE_DATE_FORMAT);
		// create calendars to compare date of entry against
		ParsedInterval interval = null;
		try
		{
			for (int i = 0, I = compareCals.length; i < I; i++)
			{
				compareCals[i] = Calendar.getInstance();
				interval = new ParsedInterval(Config.practiceIntervals[i]);
				addCalendar(compareCals[i], interval.toCalendar(), false);
			}
		}
		catch (ParseException e)
		{
		}
		Calendar entryCal = Calendar.getInstance();
		Iterator<Entry<String, Kvtml.Entry>> it = entries.entrySet().iterator();
		while (it.hasNext())
		{
			// progressDialog.setProgress(count);
			Kvtml.Entry entry = it.next().getValue();
			if (entry == null)
				break;

			// Translation which regards at the language to learn.
			Kvtml.Translation translationForGrade = entry.translations.get(Config.getLanguageForGradeId());
			if (translationForGrade == null)
			{
				it.remove();
				continue;
			}

			// practice all entries of grade 0
			if (translationForGrade.grade.currentGrade == 0)
				continue;

			try
			{
				// calendar with the time, where the entry has been practiced last
				entryCal.setTime(df.parse(translationForGrade.grade.date));
			}
			catch (ParseException e)
			{
				//Log.e(getClass().getSimpleName(), "Error parsing entry date.", e);
			}

			// if the entry has been practiced before the practice interval elapsed, remove it
			if (entryCal.after(compareCals[translationForGrade.grade.currentGrade - 1]))
				it.remove();
		}
		// progressDialog.dismiss();
		return;
	}

	// /**
	// * Method to filter all words that have not been practiced before or with
	// currentGrade = 0 .
	// */
	// private void filterNewWords(HashMap<String, Kvtml.Entry> entries)
	// {
	// Iterator<Entry<String, org.inventati.massimol.liberovocab.kvtml.Kvtml.Entry>> it =
	// entries
	// .entrySet().iterator();
	// while (it.hasNext())
	// {
	// Map.Entry<String, Kvtml.Entry> nextEntry = it.next();
	// Kvtml.Entry entry = nextEntry.getValue();
	// // only practice new words
	// if (entry.currentGrade != 0)
	// it.remove();
	// }
	// }

	/**
	 * Method to filter all words that fall into a specified grade range.
	 *
	 * @param entries the data to be filtered
	 */
	private void filterGradeRange(HashMap<String, Kvtml.Entry> entries)
	{
		String[] rangeStr = mFilterList.get(EntriesFilter.FILTER_GRADE_RANGE)
			.split(":");
		int start = Integer.valueOf(rangeStr[0]);
		int end = Integer.valueOf(rangeStr[1]);

		Iterator<Entry<String, Kvtml.Entry>> it = entries.entrySet().iterator();
		while (it.hasNext())
		{
			Map.Entry<String, Kvtml.Entry> nextEntry = it.next();

			// entry can have null value here, probably coz of call to remove() below
			Kvtml.Entry entry = nextEntry.getValue();
			if (entry == null)
				continue;

			// Translation which regards at the language to learn.
			Kvtml.Translation translationForGrade = entry.translations.get(Config.getLanguageForGradeId());
			if (translationForGrade == null)
			{
				it.remove();
				continue;
			}

			// filter out words that are not in the specified range
			if (translationForGrade.grade.currentGrade < start || translationForGrade.grade.currentGrade > end)
				it.remove();
		}
	}

	/**
	 * General method to filter according to the specified filter rules using the
	 * specialized filter method.
	 *
	 * @param entries the data to be filtered
	 */
	public HashMap<String, Kvtml.Entry> filter(HashMap<String, Kvtml.Entry> entries)
	{
		HashMap<String, Kvtml.Entry> filteredEntries = new HashMap<String, Kvtml.Entry>();

		// If we have to practice with a collection of entries, we bypass all other kind
		// of filters. The name of the collection of entries must be practiced is stored
		// as value in the filter element.
		if (hasFilter(EntriesFilter.FILTER_COLLECTION_OF_ENTRIES))
		{
			String name = getValue(EntriesFilter.FILTER_COLLECTION_OF_ENTRIES);
			for (Kvtml.CollectionOfEntries colEntries : Config.lastData.collectionsOfEntries.values())
			{
				if (colEntries.name.equals(name))
					for (String entryId : colEntries.idOfEntries)
						filteredEntries.put(entryId, Config.lastData.entries.get(entryId));
			}

			return filteredEntries;
		}

		if (!Config.lastData.lessons.values().isEmpty())
			filteredEntries.putAll(filterLessons(Config.lastData.lessons, entries));
		else
			filteredEntries.putAll(entries);

		if (hasFilter(EntriesFilter.FILTER_LEITNER))
			filterLeitner(filteredEntries);

		// if (hasFilter(Filter.FILTER_NEW_WORDS))
		// 	filterNewWords(filteredEntries);

		if (hasFilter(EntriesFilter.FILTER_GRADE_RANGE))
			filterGradeRange(filteredEntries);

		return filteredEntries;
	}

	/**
	 * Adds two calendars: target = target +- sum
	 *
	 * @param target target calendar
	 * @param sum summand calendar
	 * @param addition plus or minus
	 */
	private static void addCalendar(Calendar target, Calendar sum,
		boolean addition)
	{
		int sign = addition ? 1 : -1;
		target.add(Calendar.MONTH, sign * sum.get(Calendar.MONTH));
		// DAY_OF_MONTH falls in [1,30/31], therefore we have to store an
		// increment of 0 days as 1 day
		target.add(Calendar.DAY_OF_MONTH, sign
			* (sum.get(Calendar.DAY_OF_MONTH) - 1));
		target.add(Calendar.HOUR_OF_DAY, sign * sum.get(Calendar.HOUR_OF_DAY));
	}
}
