/*
 * Libero Vocab
 *     An app for Android systems which allows to do practice with kvtml
 *     vocabulary files.
 *     This program is a fork of another program called "Vocab Drill" by:
 *       - Károly Kiripolszky <karcsi@ekezet.com>
 *       - Matthias Völlinger <matthias.voellinger@gmx.de>
 *
 *     Copyright (C) 2019, 2020  Lo Iacono Massimo (massimol@inventati.org)
 *
 *     This program is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package org.inventati.massimol.liberovocab.kvtml.interval;

import java.text.ParseException;
import java.util.Calendar;

public final class ParsedInterval
{
	private int[] mValues;

	public ParsedInterval(String interval)
	{
		mValues = new int[3];
		set(interval);
	}

	public ParsedInterval(int[] interval)
	{
		mValues = new int[3];
		set(interval);
	}

	public void set(String interval)
	{
		final String[] parts = interval.split(":");
		for (int i = 0, I = mValues.length, I2 = parts.length; i < I; i++)
		{
			if (i < I2)
				mValues[i] = Integer.parseInt(parts[i]);
			else
				mValues[i] = 0;
		}
		return;
	}

	public void set(int[] values)
	{
		if (values.length < 3)
			return;
		mValues[0] = values[0];
		mValues[1] = values[1];
		mValues[2] = values[2];
	}

	public void set(DateItem part, int value)
	{
		switch (part)
		{
			case MONTHS:
				mValues[0] = value;
				break;
			case DAYS:
				mValues[1] = value;
				break;
			case HOURS:
				mValues[2] = value;
				break;
		}
	}

	public int[] get()
	{
		return mValues;
	}

	public int getPart(DateItem part)
	{
		switch (part)
		{
			case MONTHS:
				return mValues[0];
			case DAYS:
				return mValues[1];
			case HOURS:
				return mValues[2];
		}
		return 0;
	}

	public String partToString(DateItem part)
	{
		int n = 0;
		switch (part)
		{
			case MONTHS:
				n = mValues[0];
				break;
			case DAYS:
				n = mValues[1];
				break;
			case HOURS:
				n = mValues[2];
				break;
		}
		return String.format("%02d", n);
	}

	/**
	 * Parses an interval string into a Calendar object.
	 *
	 * @param interval
	 * @throws ParseException
	 */
	public Calendar toCalendar() throws ParseException
	{
		// parse String according to Config.INTERVAL_DATE_FORMAT
		// XXX: Config.INTERVAL_DATE_FORMAT should somehow be referenced here
		Calendar cal = Calendar.getInstance();
		// increase day of month by one, as \delta day = 0 can not be stored
		cal.set(0, mValues[0],
			mValues[1] + 1,
			mValues[2], 0, 0);
		return cal;
	}

	public String toString()
	{
		return String.format("%02d:%02d:%02d", mValues[0], mValues[1], mValues[2]);
	}
}