/*
 * Libero Vocab
 *     An app for Android systems which allows to do practice with kvtml
 *     vocabulary files.
 *     This program is a fork of another program called "Vocab Drill" by:
 *       - Károly Kiripolszky <karcsi@ekezet.com>
 *       - Matthias Völlinger <matthias.voellinger@gmx.de>
 *
 *     Copyright (C) 2019-2021  Lo Iacono Massimo (massimol@inventati.org)
 *
 *     This program is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package org.inventati.massimol.liberovocab;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Random;

import org.inventati.massimol.liberovocab.kvtml.Kvtml;

public final class QuestionFactory
{
	private HashMap<String, Kvtml.Entry> mOrigEntries;

	@SuppressWarnings("unchecked")
	public QuestionFactory(HashMap<String, Kvtml.Entry> origEntries)
	{
		mOrigEntries = (HashMap<String, Kvtml.Entry>) origEntries.clone();
	}

	/**
	 * Returns a question with a specified number of potential answers.
	 *
	 * @param entries
	 * @param maxItems
	 * @return
	 */
	public Question create(HashMap<String, Kvtml.Entry> entries, int maxItems)
	{
		if (entries.size() == 0 || maxItems <= 0)
			return null;

		// Create a list of Entries from entries:
		List<Kvtml.Entry> localEntries = new ArrayList<Kvtml.Entry>();
		localEntries.addAll(entries.values());

		// Create the Entry for the solution. It is randomly picked (selected) among the Entries
		// of the list:
		Object[] entryKeys = entries.keySet().toArray();
		final Random rnd = new Random(System.nanoTime() * Double.doubleToLongBits(Math.random()));
		String solutionEntryId = String.valueOf(entryKeys[rnd.nextInt(entryKeys.length)]);
		Kvtml.Entry solution = entries.get(solutionEntryId);

		// Create a temporary list of Entries which contains both the Entry of the solution and
		// other entries randomly selected from localEntries. The number of items of the list
		// never exceeds maxItems.
		List<Kvtml.Entry> tmpListEntries = new ArrayList<Kvtml.Entry>();
		tmpListEntries.add(solution);
		Collections.shuffle(localEntries, rnd);
		int ct = 0;
		for (Kvtml.Entry entry : localEntries)
		{
			if (ct == (maxItems - 1))
				break;
			if (entry.id.equals(solution.id))
				continue;
			tmpListEntries.add(entry);
			ct++;
		}
		Collections.shuffle(tmpListEntries, rnd);

		// Create the almost definitive list of questions.
		// It can still have got a smaller number of items than maxItems:
		HashMap<String, Kvtml.Entry> questions = new HashMap<String, Kvtml.Entry>();
		int N = tmpListEntries.size() < maxItems ? tmpListEntries.size() : maxItems;
		Kvtml.Entry entry = null;
		for (int n = 0; n < N; n++)
		{
			entry = tmpListEntries.get(n);
			questions.put(entry.id, entry);
		}

		// In the event that number of items is smaller than maxItems and also forceChoiceNumber
		// is true, other Entries will be added from mOrigEntries trying to reach maxItems.
		// ATTENTION: nevertheless the list of questions can still have got a smaller number of
		// items than maxItems .
		if (Config.forceChoiceNumber && (questions.size() < maxItems))
		{
			List<Kvtml.Entry> origEntries = new ArrayList<Kvtml.Entry>();
			origEntries.addAll(mOrigEntries.values());
			Collections.shuffle(origEntries, rnd);

			ct = 0;
			int diff = maxItems - questions.size();
			for (int n = 0, max = origEntries.size(); n < max; n++)
			{
				if (ct == diff)
					break;
				entry = origEntries.get(n);
				if (questions.containsKey(entry.id))
					continue;
				questions.put(entry.id, entry);
				ct++;
			}
		}

		return new Question(questions, solution, rnd);
	}
}