/*
 * Libero Vocab
 *     An app for Android systems which allows to do practice with kvtml
 *     vocabulary files.
 *     This program is a fork of another program called "Vocab Drill" by:
 *       - Károly Kiripolszky <karcsi@ekezet.com>
 *       - Matthias Völlinger <matthias.voellinger@gmx.de>
 *
 *     Copyright (C) 2019-2021  Lo Iacono Massimo (massimol@inventati.org)
 *
 *     This program is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package org.inventati.massimol.liberovocab.activities;

import android.app.AlertDialog;
import android.app.Activity;
import android.content.DialogInterface;
import android.content.Intent;
import android.os.Bundle;
import android.support.v4.content.ContextCompat;
import android.text.method.ScrollingMovementMethod;
import android.text.util.Linkify;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.MotionEvent;
import android.view.View;
import android.view.View.OnClickListener;
import android.widget.Button;
import android.widget.Scroller;
import android.widget.TextView;

import org.inventati.massimol.liberovocab.Config;
import org.inventati.massimol.liberovocab.R;
import org.inventati.massimol.liberovocab.dialogs.AboutDialog;
import org.inventati.massimol.liberovocab.dialogs.FileCopyDialog;
import org.inventati.massimol.liberovocab.dialogs.FileOpenDialog;
import org.inventati.massimol.liberovocab.dialogs.LicenseDialog;
import org.inventati.massimol.liberovocab.dialogs.ManualDialog;
import org.inventati.massimol.liberovocab.dialogs.NewsDialog;
import org.inventati.massimol.liberovocab.dialogs.RecentFilesDialog;

import java.io.File;

public class MainMenuActivity extends BaseMenuActivity
{
	private final static int OPEN_FILE = 1000;
	private final static int OPEN_RECENT = 2000;

	private Button mOpenRecentFileButton;

	private static boolean sTouchable = false;

	@Override
	public void onCreate(Bundle savedInstanceState)
	{
		super.onCreate(savedInstanceState);

		final Bundle savedState = savedInstanceState;

		runOnUiThread(new Runnable()
			{
				@Override
				public void run()
				{
					try
					{
						Config.setColorStartForGradient(ContextCompat.getColor(MainMenuActivity.this, R.color.start_for_gradient));
						Config.setColorEndForGradient(ContextCompat.getColor(MainMenuActivity.this, R.color.end_for_gradient));
						Config.setFullOrShortMessageForHiddenQuestionTextValue(getResources().getStringArray(R.array.pref_full_or_short_message_for_hidden_question_text_values)[0]);
						Config.setBaseDirectoryForAudioFilesValue(getResources().getStringArray(R.array.pref_base_directory_for_audio_files_values)[0]);

						Config.load(getApplicationContext());

						// splash screen delay
						Thread.sleep(333);

						// call main app initialization
						initialize(savedState);
					}
					catch (InterruptedException e)
					{
					}
				}
			}
		);
	}

	private void initialize(Bundle savedInstanceState)
	{
		/*
		requestWindowFeature(Window.FEATURE_NO_TITLE);
		getWindow().setFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN,
			                 WindowManager.LayoutParams.FLAG_FULLSCREEN);
		*/
		setTheme(R.style.AppTheme);
		setContentView(R.layout.activity_main);

		//setTitle(String.format("v%s", Config.getVersionName()));

		// open new file button
		Button btn = (Button) findViewById(R.id.button_open_file);
		btn.setOnClickListener(new OnClickListener()
			{
				@Override
				public void onClick(View v)
				{
					startActivityForResult(new Intent(MainMenuActivity.this,
						                              FileOpenDialog.class),
							               OPEN_FILE);
				}
			}
		);

		// get handle to 'open last file' button
		mOpenRecentFileButton = (Button) findViewById(R.id.button_open_recent_file);
		mOpenRecentFileButton.setOnClickListener(new OnClickListener()
			{
				@Override
				public void onClick(View v)
				{
					startActivityForResult(new Intent(MainMenuActivity.this,
						                              RecentFilesDialog.class),
							               OPEN_RECENT);
				}
			}
		);

		// settings button
		btn = (Button) findViewById(R.id.button_settings);
		btn.setOnClickListener(new OnClickListener()
			{
				@Override
				public void onClick(View v)
				{
					startActivity(new Intent(MainMenuActivity.this,
							                 PreferencesActivity.class)
					             );
				}
			}
		);

		// quit button
		btn = (Button) findViewById(R.id.button_quit);
		btn.setOnClickListener(new OnClickListener()
			{
				@Override
				public void onClick(View v)
				{
					kill();
				}
			}
		);

		toggleMenuButtons();
		mMenuLayout.setVisibility(View.VISIBLE);
		sTouchable = true;

		// A Dialog is automatically showed in case of new version. This dialog is in order to
		// warn in the use of Kvtml files created and used in other programs too:
		if (Config.getPreviousVersionCode() < Config.getVersionCode() &&
				!Config.getPreviousVersionName().equals(Config.getVersionName()))
		{
			warnKvtmlFilesUsedInOtherProgramsDialog();
		}

		// A Dialog is automatically showed in case of new version. This dialog is in order to
		// ask the user whether he or she want the Kvtml files in assets are copied in the private
		// external storage of the device:
		if (Config.getPreviousVersionCode() < Config.getVersionCode() &&
			!Config.getPreviousVersionName().equals(Config.getVersionName()))
		{
			copyKvtmlFilesDialog();
		}

		// The dialog of the news automatically starts in case of new version:
		if (Config.getPreviousVersionCode() < Config.getVersionCode() &&
			!Config.getPreviousVersionName().equals(Config.getVersionName()))
		{
			Intent i = new Intent(this, NewsDialog.class);
			//i.putExtra("limit", 3);
			Config.invalidatePreviouseVersionCode();
			Config.invalidatePreviouseVersionName();
			Config.save();
			startActivity(i);
		}
	}

	private void warnKvtmlFilesUsedInOtherProgramsDialog()
	{
		AlertDialog.Builder dialog = new AlertDialog.Builder(this);
		dialog.setTitle(R.string.title_warn_kvtml_files_used_in_other_programs)
				.setMessage(R.string.message_warn_kvtml_files_used_in_other_programs)
				.setCancelable(true)
				.setPositiveButton(R.string.button_ok,
						new DialogInterface.OnClickListener()
						{
							@Override
							public void onClick(DialogInterface dialog, int id)
							{
								dialog.cancel();
							}
						}
				);

		AlertDialog alert = dialog.create();
		alert.show();

		// The TextView of the dialog must be scrollable:
		TextView textView = (TextView) alert.findViewById(android.R.id.message);
		textView.setAutoLinkMask(Linkify.WEB_URLS);
		textView.setScroller(new Scroller(this));
		textView.setVerticalScrollBarEnabled(true);
		textView.setMovementMethod(new ScrollingMovementMethod());
	}

	private void copyKvtmlFilesDialog()
	{
		AlertDialog.Builder dialog = new AlertDialog.Builder(this);
		dialog.setTitle(R.string.title_copy_kvtml_files)
				.setMessage(R.string.message_copy_kvtml_files)
				.setCancelable(true)
				.setPositiveButton(R.string.button_yes,
						new DialogInterface.OnClickListener()
						{
							@Override
							public void onClick(DialogInterface dialog, int id)
							{
								startActivity(new Intent(MainMenuActivity.this, FileCopyDialog.class));
							}
						}
				)
				.setNegativeButton(R.string.button_no,
						new DialogInterface.OnClickListener()
						{
							@Override
							public void onClick(DialogInterface dialog, int id)
							{
								dialog.cancel();
							}
						}
				);

		AlertDialog alert = dialog.create();
		alert.show();

		// The TextView of the dialog must be scrollable:
		TextView textView = (TextView) alert.findViewById(android.R.id.message);
		textView.setAutoLinkMask(Linkify.WEB_URLS);
		textView.setScroller(new Scroller(this));
		textView.setVerticalScrollBarEnabled(true);
		textView.setMovementMethod(new ScrollingMovementMethod());
	}

	private void kill()
	{
		Config.save();
		try
		{
			finalize();
		}
		catch (Throwable e)
		{
			e.printStackTrace();
		}
		System.exit(0);
	}

	@Override
	public boolean onCreateOptionsMenu(Menu menu)
	{
		MenuInflater inflater = getMenuInflater();
		inflater.inflate(R.menu.main_activity_menu, menu);

		MenuItem item = menu.findItem(R.id.option_copy_kvtml_files);
		item.setVisible(true);

		item = menu.findItem(R.id.option_manual);
		item.setVisible(true);

		item = menu.findItem(R.id.option_news);
		item.setVisible(true);

		item = menu.findItem(R.id.option_license);
		item.setVisible(true);

		item = menu.findItem(R.id.option_information);
		item.setVisible(true);

		return super.onCreateOptionsMenu(menu);
	}

	@Override
	public boolean onPrepareOptionsMenu(Menu menu)
	{
		return super.onPrepareOptionsMenu(menu);
	}

	@Override
	public boolean onOptionsItemSelected(MenuItem item)
	{
		switch (item.getItemId())
		{
			case R.id.option_copy_kvtml_files:
				copyKvtmlFilesDialog();
				return true;

			case R.id.option_manual:
				startActivity(new Intent(MainMenuActivity.this, ManualDialog.class));
				return true;

			case R.id.option_news:
				startActivity(new Intent(MainMenuActivity.this, NewsDialog.class));
				return true;

			case R.id.option_license:
				startActivity(new Intent(MainMenuActivity.this, LicenseDialog.class));
				return true;

			case R.id.option_information:
				startActivity(new Intent(MainMenuActivity.this, AboutDialog.class));
				return true;
		}
		return false;
	}

	/**
	 * Handles the results returned by dialogs.
	 */
	@Override
	protected void onActivityResult(int requestCode, int resultCode, Intent data)
	{
		if (data == null)
			return;
		if (resultCode != Activity.RESULT_OK)
			return;

		String selected = "";
		switch (requestCode)
		{
			case OPEN_RECENT:
				selected = data.getDataString();
			case OPEN_FILE:
				selected = String.format("%s", selected).equals("") ? data.getStringExtra(FileOpenDialog.INTENT_PATH_PARAMETER) : selected;
		}

		Config.inputFile = new File(selected);
		Config.insertRecentFile(Config.inputFile.getAbsolutePath());
		Config.saveRecentFilesList();
		startActivity(new Intent(this, VocabFileActivity.class));
	}

	@Override
	protected void toggleMenuButtons()
	{
		// show/hide recent files menu
		mOpenRecentFileButton.setVisibility(Config.hasRecentFiles() ? View.VISIBLE : View.GONE);
		updateMenuColours();
	}

	@Override
	public boolean onTouchEvent(MotionEvent event)
	{
		if (!sTouchable)
			return false;
		return super.onTouchEvent(event);
	}
}
