/*
 * Libero Vocab
 *     An app for Android systems which allows to do practice with kvtml
 *     vocabulary files.
 *     This program is a fork of another program called "Vocab Drill" by:
 *       - Károly Kiripolszky <karcsi@ekezet.com>
 *       - Matthias Völlinger <matthias.voellinger@gmx.de>
 *
 *     Copyright (C) 2019-2021  Lo Iacono Massimo (massimol@inventati.org)
 *
 *     This program is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package org.inventati.massimol.liberovocab.activities.test_types;

import java.io.File;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Locale;

import android.app.Activity;
import android.support.v7.app.AppCompatActivity;
import android.support.v7.app.ActionBar;

import android.os.Looper;
import android.support.annotation.Nullable;
import android.annotation.SuppressLint;
import android.app.AlertDialog;
import android.content.DialogInterface;
import android.content.Intent;
import android.media.MediaPlayer;
import android.media.AudioManager;
import android.net.Uri;
import android.os.Bundle;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.MotionEvent;
import android.view.View;
import android.view.WindowManager;
import android.widget.ImageButton;
import android.widget.TextView;
import android.widget.Toast;

import org.inventati.massimol.liberovocab.Config;
import org.inventati.massimol.liberovocab.Question;
import org.inventati.massimol.liberovocab.QuestionFactory;
import org.inventati.massimol.liberovocab.R;
import org.inventati.massimol.liberovocab.activities.FilterActivity;
import org.inventati.massimol.liberovocab.dialogs.EntryBrowserDialog;
import org.inventati.massimol.liberovocab.dialogs.CollectionBrowserDialog;
import org.inventati.massimol.liberovocab.helpers.EntriesFilter;
import org.inventati.massimol.liberovocab.helpers.Gradient;
import org.inventati.massimol.liberovocab.helpers.StatsOpenHelper;
import org.inventati.massimol.liberovocab.kvtml.Kvtml;
import org.inventati.massimol.liberovocab.kvtml.KvtmlWriter;

public abstract class TestActivity extends AppCompatActivity
{
    public class TestStateHolder
    {
        public HashMap<String, Kvtml.Entry> mistakes = null;
        public HashMap<String, Kvtml.Entry> questionData = null;

        public EntriesFilter mFilter = null;

        public Question question = null;

        public int mistakesNum = 0;
        public int maxQuestions;
        public int maxItems = 1;
        public int entriesLeft;
    }

    public static final int TEST_CHOICE = 1;
    public static final int TEST_WRITING = 2;
    public static final int TEST_FLASHCARD = 3;

    public static final String GRADUATED_MODE = "graduatedmode";
    public static final String TEST_TYPE = "testtype";

    public static final int REQUEST_CODE_FROM_DIALOG_FOR_SAVE_WORDS = 2000;

    /**
     * TextView for displaying the current language of questions/answers.
     */
    protected TextView mLangText = null;

    /**
     * The TextView for the current question.
     */
    protected TextView mQuestionText = null;

    /**
     * The ImageButton for showing the text of the current question.
     */
    protected ImageButton mButtonToShowQuestionText = null;

    /*
     * Properties related to test progress.
     */
    protected QuestionFactory mQuestionFactory = null;
    protected Question mQuestion = null;
    protected int mMaxQuestions;
    protected int mMaxItems = 1;
    protected int mEntriesLeft;

    /**
     * String for displaying the number of mistakes (comes from resources).
     */
    protected String mMistakesText = null;
    protected String mErrorNoTranslationText = null;

    protected int mMistakesNum = 0;
    protected HashMap<String, Kvtml.Entry> mMistakes = null;
    protected HashMap<String, Kvtml.Entry> mQuestionData = null;
    protected HashSet<String> idOfEntries = null;

    /**
     * The current type of test:
     */
    protected int mTestType = -1;

    /**
     * SQLite access helper.
     */
    private StatsOpenHelper mStats;

    /**
     * Identifier of the current KVTML file in the "files" SQL table.
     */
    private long mFileId = -1;

    /**
     * The translation of the current question.
     */
    protected Kvtml.Translation transQuestion = null;

    /**
     * The filter for this session
     */
    protected EntriesFilter mFilter;

    /**
     * If we are in graduated mode:
     */
    protected Boolean mGraduatedMode = false;

    /**
     * The MediaPlayer object for the sound:
     */
    MediaPlayer mediaPlayer;

    /**
     * If the sound is OK:
     */
    private Boolean isSoundOK;

    /**
     * The absolute path of the audio file:
     */
    private String pathSoundAbsolute = "";

    /**
     * The customized TextView for the ActionBar:
     */
    private TextView mTextViewForActionBar;

    /**
     * learn mode shows both question and answers and lets the user cycle forwards
     * and backwards
     */
    Boolean mLearningMode = false;

    @Override
    protected void onCreate(Bundle savedInstanceState)
    {
        super.onCreate(savedInstanceState);

        // Get filter:
        Intent i = getIntent();
        if (i.hasExtra(FilterActivity.EXTRA_FILTER))
            mFilter = (EntriesFilter) i.getSerializableExtra(FilterActivity.EXTRA_FILTER);

        // Get the type of test:
        if (i.hasExtra(TEST_TYPE))
            mTestType = (int) i.getSerializableExtra(TEST_TYPE);

        // Get if is in graduated mode:
        if (i.hasExtra(GRADUATED_MODE))
            mGraduatedMode = (Boolean) i.getSerializableExtra(GRADUATED_MODE);

        mQuestionFactory = new QuestionFactory(Config.lastData.entries);
        mStats = StatsOpenHelper.getInstance(this, mTestType);
        mFileId = mStats.getFileId(Config.inputFile.getName());

        // prepare activity window
        getWindow().setFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN,
                             WindowManager.LayoutParams.FLAG_FULLSCREEN);
        loadContentView();

        // This activity has an ActionBar:
        ActionBar actionBar = getSupportActionBar();
        actionBar.setHomeButtonEnabled(true);

        // In order to hide the icon in the ActionBar:
        //actionBar.setIcon(null);
        //actionBar.setIcon(android.R.color.transparent);

        // In order to customize the title in the ActionBar to be multiline and small:
        actionBar.setDisplayShowCustomEnabled(true);
        actionBar.setDisplayShowTitleEnabled(false);

        LayoutInflater inflator = LayoutInflater.from(this);
        View mCustomViewForActionBar = inflator.inflate(R.layout.view_action_bar_multiline_and_small, null);
        mTextViewForActionBar = ((TextView) mCustomViewForActionBar.findViewById(R.id.action_bar_title));
        actionBar.setCustomView(mCustomViewForActionBar);

        // preload some resources
        mMistakesText = getResources().getString(R.string.mistakes);
        mErrorNoTranslationText = getResources().getString(R.string.error_no_translation_for_entry);

        // get handles for widgets
        mLangText = (TextView) findViewById(R.id.text_language);
        mQuestionText = (TextView) findViewById(R.id.text_question);
        mButtonToShowQuestionText = (ImageButton) findViewById(R.id.button_to_show_text_question);

        /* Adding listener to mQuestionText (TextView) in order to handle the execution
         * of the sound.
         *
         *                          IMPORTANT
         *                          =========
         *
         * If the touch will be pointlike, the sound is alternately paused and continued.
         * else if the touch will be with movement, the sound is restarted.
         */
        mQuestionText.setOnTouchListener(new View.OnTouchListener()
                            {
                                private boolean wasMoved = false;

                                private float startingPosX;
                                private float startingPosY;

                                @Override
                                public boolean onTouch(View v, MotionEvent event)
                                {
                                    if (event.getActionMasked() == MotionEvent.ACTION_DOWN)
                                    {
                                        wasMoved = false;
                                        startingPosX = event.getX();
                                        startingPosY = event.getY();
                                        return true;
                                    }
                                    else if (event.getActionMasked() == MotionEvent.ACTION_MOVE)
                                    {
                                        wasMoved = true;
                                        return true;
                                    }
                                    else if (event.getActionMasked() == MotionEvent.ACTION_UP)
                                    {
                                        // We'll consider the finger moved only if it moves
                                        // more than a certain limit:
                                        if (wasMoved  &&  distanceInMillimeters(startingPosX, startingPosY, event) > Config.distanceLimitTouch)
                                            soundExecution();
                                        else
                                            try
                                            {
                                                if (mediaPlayer.isPlaying())
                                                    mediaPlayer.pause();
                                                else
                                                    mediaPlayer.start();
                                            }
                                            catch (IllegalStateException e)
                                            {
                                                e.printStackTrace();
                                                System.err.println("### ERROR: IllegalStateException in onTouch ###");

                                                soundExecution();
                                            }
                                            catch (NullPointerException e)
                                            {
                                                e.printStackTrace();
                                                System.err.println("### ERROR: NullPointerException in onTouch ###");
                                            }

                                        return true;
                                    }

                                    return true;
                                }
                            }
        );

        // Adding listener to mButtonToShowQuestionText (ImageButton) for the showing of the
        // question text:
        mButtonToShowQuestionText.setOnClickListener(new View.OnClickListener()
            {
                @Override
                public void onClick(View v)
                {
                    mQuestionText.setText(transQuestion.text);
                    mQuestionText.setTextSize(getResources().getDimension(R.dimen.large_text));

                    mButtonToShowQuestionText.setEnabled(false);
                    mButtonToShowQuestionText.setVisibility(View.GONE);
                }
            }
        );

        // prepare colours
        ArrayList<View> lst = new ArrayList<View>();
        lst.add(mLangText);
        lst.add(mQuestionText);
        Gradient.colorize(lst);

        // prepare questions
        mMistakes = new HashMap<String, Kvtml.Entry>();
    }

    /*
     * Return the distance covered by the finger from the starting position, recorded during the
     * ACTION_DOWN motion event, to the final position, appended during the ACTION_UP motion
     * event.
     */
    private double distanceInMillimeters(float startingPosX, float startingPosY, MotionEvent event)
    {
        // The distances in pixels:
        float distanceX = startingPosX - event.getX();
        float distanceY = startingPosY - event.getY();

        // The distances in inch. xdpi and ydpi are the densities
        // of the screen expressed in pixels per inch:
        distanceX /= getResources().getDisplayMetrics().xdpi;
        distanceY /= getResources().getDisplayMetrics().ydpi;

        // The distances in inch as double:
        double ddistanceX = Double.parseDouble(String.valueOf(distanceX));
        double ddistanceY = Double.parseDouble(String.valueOf(distanceY));

        // The distance in inch:
        double distance = Math.sqrt(ddistanceX*ddistanceX + ddistanceY*ddistanceY);

        // The distance in mm:
        double distanceMm = distance * 25.4;

        return distanceMm;
    }

    @Override
    public void onWindowFocusChanged(boolean hasFocus)
    {
        super.onWindowFocusChanged(hasFocus);
    }

    @Override
    public boolean onCreateOptionsMenu(Menu menu)
    {
        MenuInflater inflater = getMenuInflater();
        inflater.inflate(R.menu.test_menu, menu);

        // If we are practicing Flashcards:
        if (mTestType == TEST_FLASHCARD)
        {
            // Hide the menu item for show the solution:
            MenuItem item = menu.findItem(R.id.option_show_solution);
            item.setVisible(false);
        }

        // If we are in the learning mode:
        if (mLearningMode)
        {
            // Hide the menu items for showing the errors:
            MenuItem item = menu.findItem(R.id.option_show_mistakes);
            item.setVisible(false);

            // Hide the menu items for restarting the test:
            item = menu.findItem(R.id.option_restart);
            item.setVisible(false);
        }

        // If we are going to practice a collection of entries:
        if (mFilter != null && mFilter.hasFilter(EntriesFilter.FILTER_COLLECTION_OF_ENTRIES))
        {
            // Hide the menu items for showing the errors:
            MenuItem item = menu.findItem(R.id.option_save_collection_of_entries);
            item.setVisible(false);
        }

        return super.onCreateOptionsMenu(menu);
    }

    @Override
    public boolean onPrepareOptionsMenu(Menu menu)
    {
        return super.onPrepareOptionsMenu(menu);
    }

    @SuppressWarnings("deprecation")
    @Override
    public boolean onOptionsItemSelected(MenuItem item)
    {
        Toast mToast;
        switch (item.getItemId())
        {
            case R.id.option_save:
                KvtmlWriter saver = new KvtmlWriter(Config.lastData, Config.inputFile.getPath());
                saver.save();
                return true;

            case R.id.option_show_solution:
                Kvtml.Entry mEntry = Config.lastData.entries.get(mQuestion.getSolutionEntryId());
                Kvtml.Translation mTranslation = mEntry.translations.get(Config.getAnswerLangId());

                // TODO: this is a workaround. A better solution would be to create objects Kvtml, and
                //       obviously relative files, with all the fields always made explicit, also
                //       whether empty.
                String textOfSolution = mTranslation != null ? mTranslation.text : "- - -";

                mToast = Toast.makeText(this, textOfSolution, Toast.LENGTH_LONG);
                mToast.show();
                return true;

            case R.id.option_show_mistakes:
                if (mMistakesNum == 0)
                    return false;

                // The previous MediaPlayer object is released:
                if (mediaPlayer != null)
                    mediaPlayer.release();

                EntryBrowserDialog.data = Config.lastData.cloneFromEntries(mMistakes);
                Intent i = new Intent(this, EntryBrowserDialog.class);
                i.putExtra("doSort", false);
                i.putExtra("windowTitle", getResources().getString(R.string.title_show_mistakes));
                startActivity(i);
                return true;

            case R.id.option_restart:
                // The previous MediaPlayer object is released:
                if (mediaPlayer != null)
                    mediaPlayer.release();

                restart();
                return true;

            case R.id.option_diagnostic_message:
                // The previous MediaPlayer object is released:
                if (mediaPlayer != null)
                    mediaPlayer.release();

                if (Config.showDiagnosticMessageForAudio)
                {
                    Config.showDiagnosticMessageForAudio = false;
                    mToast = Toast.makeText(this, getResources().getString(R.string.menu_diagnostic_message_disabled), Toast.LENGTH_SHORT);
                }
                else
                {
                    Config.showDiagnosticMessageForAudio = true;
                    mToast = Toast.makeText(this, getResources().getString(R.string.menu_diagnostic_message_enabled), Toast.LENGTH_SHORT);
                }
                Config.save();
                mToast.show();
                return true;

            case R.id.option_save_collection_of_entries:
                saveCollectionOfEntries();
                return true;

            default:
                return super.onOptionsItemSelected(item);
        }
    }

    /**
     * @return a string to be set as title
     */
    protected String getTitleString()
    {
        double percent = (Double.valueOf(mMistakesNum) / Double.valueOf(mMaxQuestions)) * 100.0;
        if (percent != percent)
            percent = 0;

        //Log.i(getClass().getSimpleName(), String.format("question #%d", 1 + (mMaxQuestions - mEntriesLeft)));
        return String.format("%d/%d (%s: %d; %.2f%%)", 1 + (mMaxQuestions - mEntriesLeft), mMaxQuestions, mMistakesText, mMistakesNum, percent);
    }

    private void saveCollectionOfEntries()
    {
        CollectionBrowserDialog.data = Config.lastData;
        Intent i = new Intent(TestActivity.this, CollectionBrowserDialog.class);
        i.putExtra("doSort", true);
        i.putExtra("typeOfCollection", CollectionBrowserDialog.SAVE_COLLECTION_OF_ENTRIES);
        startActivityForResult(i, REQUEST_CODE_FROM_DIALOG_FOR_SAVE_WORDS);
    }

    /**
     * Handles the results returned by dialog for collections of entries:.
     */
    @Override
    protected void onActivityResult(int requestCode, int resultCode, Intent data)
    {
        if (data == null)
            return;
        if (resultCode != Activity.RESULT_OK)
            return;

        switch (requestCode)
        {
           case REQUEST_CODE_FROM_DIALOG_FOR_SAVE_WORDS:
                Config.lastData.saveCollectionOfEntries(data.getStringExtra("name_of_collection_of_entries"), data.getStringExtra("comment_of_collection_of_entries"), idOfEntries);
                break;
        }
    }

    @SuppressLint("DefaultLocale")
    protected void showFinishDialog()
    {
        if (mMistakesNum != 0)
        {
            EntryBrowserDialog.data = Config.lastData.cloneFromEntries(mMistakes);
            Intent i = new Intent(TestActivity.this, EntryBrowserDialog.class);
            i.putExtra("doSort", false);
            i.putExtra("windowTitle", getResources().getString(R.string.title_show_mistakes));
            startActivity(i);
        }

        // If we are in graduated mode, it asks if we want save to the file:
        if (mGraduatedMode)
        {
            if (Config.confirmProgress)
            {
                new AlertDialog.Builder(this)
                .setTitle(R.string.title_test_ended)
                .setMessage(R.string.message_save_progress)
                .setCancelable(true)
                        .setPositiveButton(R.string.button_yes,
                                new DialogInterface.OnClickListener()
                                {
                                    @Override
                                    public void onClick(DialogInterface dialog, int id)
                                    {
                                        KvtmlWriter saver = new KvtmlWriter(Config.lastData, Config.inputFile.getPath());
                                        saver.save();
                                        finish();
                                    }
                                }
                        )
                        .setNegativeButton(R.string.button_no,
                                new DialogInterface.OnClickListener()
                                {
                                    @Override
                                    public void onClick(DialogInterface dialog, int id)
                                    {
                                        finish();
                                    }
                                }
                        )
                        .create()
                        .show();
            }
            else
            {
                KvtmlWriter saver = new KvtmlWriter(Config.lastData, Config.inputFile.getPath());
                saver.save();
                finish();
            }
        }
        else
            finish();
    }

    /**
     * Increases the number of mistakes and saves the mistaken entry.
     *
     * All extending test activities should use this method for registering
     * mistakes.
     *
     * @param entry The mistaken entry
     */
    protected void mistake(Kvtml.Entry entry)
    {
        mMistakes.put(entry.id, entry);
        mMistakesNum++;
        mStats.addMistake(mFileId, entry.id, Config.getQuestionLangId(), mTestType);

        if (mFilter != null && mFilter.hasFilter(EntriesFilter.FILTER_LEITNER))
        {
            // Translation of this Entry which regards at the language to learn.
            Kvtml.Translation translationForGrade = entry.translations.get(Config.getLanguageForGradeId());

            translationForGrade.grade.errorCount++;
            translationForGrade.grade.currentGrade = 1;

            // Update the date:
            SimpleDateFormat df = new SimpleDateFormat(Kvtml.GRADE_DATE_DATE_FORMAT, Locale.getDefault());
            translationForGrade.grade.date = df.format(Calendar.getInstance().getTime());
        }
    }

    /**
     * All extending test activities should use this method for registering
     * correct answers.
     *
     * @param entry
     * @return Returns False if no more questions left.
     */
    protected boolean correct(Kvtml.Entry entry)
    {
        if (mFilter != null && mFilter.hasFilter(EntriesFilter.FILTER_LEITNER))
        {
            // Translation which regards at the language to learn.
            Kvtml.Translation translationForGrade = entry.translations.get(Config.getLanguageForGradeId());

            translationForGrade.grade.count++;
            switch (translationForGrade.grade.currentGrade)
            {
                case 0:
                    translationForGrade.grade.currentGrade = 2;
                    break;
                case 1:
                    if (!mMistakes.containsKey(entry.id))
                        translationForGrade.grade.currentGrade = 2;
                    break;
                default:
                    if (translationForGrade.grade.currentGrade < 7)
                        translationForGrade.grade.currentGrade++;
            }

            // Update the date:
            SimpleDateFormat df = new SimpleDateFormat(Kvtml.GRADE_DATE_DATE_FORMAT, Locale.getDefault());
            translationForGrade.grade.date = df.format(Calendar.getInstance().getTime());
        }

        mQuestionData.remove(entry.id);
        mEntriesLeft--;
        if (mEntriesLeft == 0)
        {
            showFinishDialog();
            return false;
        }
        return true;
    }

    /**
     * provides a question for updateQuestion
     *
     * @param entries Entries of the vocabulary data set
     * @param maxItems mMaxItems for MultipleChoice
     * @return
     */
    protected Question getNewQuestion(HashMap<String, Kvtml.Entry> entries, int maxItems)
    {
        return mQuestionFactory.create(entries, maxItems);
    }

    /**
     * Prepares a new question and updates the UI.
     *
     * @param configChanged
     */
    protected void updateQuestion(boolean configChanged)
    {
        mQuestion = !configChanged ? getNewQuestion(mQuestionData, mMaxItems) : mQuestion;
        if (mQuestion == null)
        {
            finish();
            return;
        }

        // set language ids
        String questionLangId = Config.getQuestionLangId();
        String answerLangId = Config.getAnswerLangId();

        // show language settings information
        final String langString = Config.lastData.identifiers.get(questionLangId).name +
                                  "? » " +
                                  Config.lastData.identifiers.get(answerLangId).name;
        mLangText.setText(langString);

        // display question
        Kvtml.Entry solution = mQuestion.getSolution();
        transQuestion = solution.translations.get(questionLangId);

        // update the UI:
        if (transQuestion == null)
        {
            mQuestionText.setText(mErrorNoTranslationText);

            mButtonToShowQuestionText.setEnabled(false);
            mButtonToShowQuestionText.setVisibility(View.GONE);
        }
        else
        {
            if (Config.forceQuestionWithSound && soundExecution()
                                              && isSoundOK)
            {
                if (!mLearningMode)
                {
                    mQuestionText.setTextSize(getResources().getDimension(R.dimen.noticeTextHidden));

                    if (Config.getFullOrShortMessageForHiddenQuestionTextValue().equals(getResources().getStringArray(R.array.pref_full_or_short_message_for_hidden_question_text_values)[0]))
                        mQuestionText.setText(R.string.noticeQuestionTextHiddenFull);
                    else if (Config.getFullOrShortMessageForHiddenQuestionTextValue().equals(getResources().getStringArray(R.array.pref_full_or_short_message_for_hidden_question_text_values)[1]))
                        mQuestionText.setText(R.string.noticeQuestionTextHiddenShort);
                    else if (Config.getFullOrShortMessageForHiddenQuestionTextValue().equals(getResources().getStringArray(R.array.pref_full_or_short_message_for_hidden_question_text_values)[2]))
                        mQuestionText.setText(R.string.noticeQuestionTextHiddenEmpty);
                    else
                        mQuestionText.setText(R.string.noticeQuestionTextHiddenFull);

                    mButtonToShowQuestionText.setEnabled(true);
                    mButtonToShowQuestionText.setVisibility(View.VISIBLE);
                }
                else
                {
                    mQuestionText.setText(transQuestion.text);

                    mButtonToShowQuestionText.setEnabled(false);
                    mButtonToShowQuestionText.setVisibility(View.GONE);
                }
            }
            else
            {
                mQuestionText.setText(transQuestion.text);
                mQuestionText.setTextSize(getResources().getDimension(R.dimen.large_text));

                mButtonToShowQuestionText.setEnabled(false);
                mButtonToShowQuestionText.setVisibility(View.GONE);
            }
        }

        // Set both the title of the Activity and the customized text of the ActionBar,
        // although the first isn't shown:
        String sTitle = getTitleString();
        setTitle(sTitle);
        mTextViewForActionBar.setText(sTitle);
    }

    protected void updateQuestion()
    {
        updateQuestion(false);
    }

    /**
     * Called by the listener of mQuestionText (TextView) for the execution of the sound.
     */
    private boolean soundExecution()
    {
        // A preliminary control on the sound field for the present translation is made
        // in order to establish that the sound path is a not an empty string:
        if (transQuestion.sound == null || transQuestion.sound.equals(""))
            return false;

        // Initialize the variable to control the result of the sound playing:
        isSoundOK = null;

        // The previous MediaPlayer object is released:
        if (mediaPlayer != null)
            mediaPlayer.release();

        // Initialize the field  pathSoundAbsolute:
        pathSoundAbsolute = "";

        // The Thread for executing the sound:
        Thread thread_sound;

        // The new MediaPlayer object is prepared:
        try
        {
            // We get the relative path of the sound file processing that comes from
            // the kvtml file:
            File fileSoundRelative = getFileSoundRelative(transQuestion.sound);
            if (fileSoundRelative == null)
                return false;

            /* ----------------------------------------------------------------------------------
               OK! We consider the path of the sound file could be existent and correct
               and than we proceed...
            ---------------------------------------------------------------------------------- */

            // We get the absolute path of the sound file placing a base part before the
            // relative part:
            String basePartForSound;
            if (Config.getBaseDirectoryForAudioFilesValue().equals(getResources().getStringArray(R.array.pref_base_directory_for_audio_files_values)[0]))
                basePartForSound = Config.inputFile.getParent();
            else if (Config.getBaseDirectoryForAudioFilesValue().equals(getResources().getStringArray(R.array.pref_base_directory_for_audio_files_values)[1]))
                basePartForSound = "/";
            else
                basePartForSound = Config.getBaseDirectoryForAudioFilesValue();

            File fileSoundAbsolute = new File(basePartForSound, fileSoundRelative.getPath());
            pathSoundAbsolute = fileSoundAbsolute.getPath();

            // We get the Uri object corresponding to the sound file:
            pathSoundAbsolute = Uri.decode(pathSoundAbsolute);
            final Uri uriSound = Uri.parse(pathSoundAbsolute);

            thread_sound = new Thread(new Runnable()
                                {
                                    @Override
                                    public void run()
                                    {
                                        try
                                        {
                                            mediaPlayer = new MediaPlayer();
                                            mediaPlayer.setAudioStreamType(AudioManager.STREAM_MUSIC);
                                            mediaPlayer.setDataSource(getApplicationContext(), uriSound);
                                            mediaPlayer.prepare();

                                            isSoundOK = true;

                                            mediaPlayer.start();
                                        }
                                        catch (IOException e)
                                        {
                                            isSoundOK = false;

                                            Looper.prepare();
                                            if (Config.showDiagnosticMessageForAudio)
                                                showDiagnosticMessageForAudio();
                                            Looper.loop();

                                            e.printStackTrace();
                                            System.err.println("### ERROR: IOException in soundExecution - thread ###");
                                        }
                                        catch (NullPointerException e)
                                        {
                                            isSoundOK = false;

                                            Looper.prepare();
                                            if (Config.showDiagnosticMessageForAudio)
                                                showDiagnosticMessageForAudio();
                                            Looper.loop();

                                            e.printStackTrace();
                                            System.err.println("### ERROR: NullPointerException in soundExecution - thread ###");
                                        }
                                    }
                                }
            );
            thread_sound.start();
        }
        catch (Exception e)
        {
            if (Config.showDiagnosticMessageForAudio)
                showDiagnosticMessageForAudio();

            e.printStackTrace();
            System.err.println("### ERROR: Exception in soundExecution ###");
            return false;
        }

        // We wait until the thread of the sound sets the variable  isSoundOK  true or false
        // controlling on regular intervals of time:
        int interval = 10;  // in ms
        while (isSoundOK == null  &&  thread_sound.isAlive())
        {
            try
            {
                thread_sound.join(interval);
            }
            catch (InterruptedException e)
            {
                e.printStackTrace();
            }
        }

        return isSoundOK;
    }

    /**
     * Return the File object corresponding to the sound file that comes from the kvtml file,
     * or null if any path seems possible.
     * Note: we always consider this path relative: if it is in reality absolute, we'll consider
     *       it relative to the root of the filesystem.
     *
     * @param soundPathFromVocab The path of the sound file as appears in the kvtml file.
     * @return The File object corresponding to the sound file after processing the path presents
     *         in the kvtml file, or null if it appears is not a correct path.
     * @throws NullPointerException
     */
    @Nullable
    private File getFileSoundRelative(String soundPathFromVocab) throws NullPointerException
    {
        // We eventually remove the starting part of the path:
        String startSubString = Config.getStartStringToBeRemovedInPath();
        if (! startSubString.equals("") && soundPathFromVocab.startsWith(startSubString))
            soundPathFromVocab = soundPathFromVocab.substring(startSubString.length());

        // If the path was reduced to the empty string, we will return null:
        if (soundPathFromVocab.equals(""))
            return null;

        // We eventually substitute '\' character with '/':
        if (Config.replaceSeparatorChar)
            soundPathFromVocab = soundPathFromVocab.replace('\\', '/');

        // Ok! Now we can return the file object:
        File fileSoundRelative = new File(soundPathFromVocab);
        return fileSoundRelative;
    }

    private void showDiagnosticMessageForAudio()
    {
        new AlertDialog.Builder(TestActivity.this)
                .setTitle(R.string.pref_show_diagnostic_message_for_audio_title)
                .setMessage(getResources().getText(R.string.diagnostic_message_for_audio) + pathSoundAbsolute)
                .setPositiveButton(R.string.button_ok,
                        new DialogInterface.OnClickListener() {
                            @Override
                            public void onClick(DialogInterface dialog, int which) {
                            }
                        }
                )
                .create()
                .show();
    }

    @Override
    public boolean onKeyDown(int keyCode, KeyEvent event)
    {
        /*
         * Executed when the back button was pressed.
         *
         * Ask the user if she really wants to quit the test and, if yes, ask if she want
         * save to the file too.
         *
         * This is handy since some ppl may think that pressing the back button
         * will take them back to the previous question.
         */
        if (keyCode == KeyEvent.KEYCODE_BACK && event.getRepeatCount() == 0)
        {
            // The previous MediaPlayer object is released:
            if (mediaPlayer != null)
                mediaPlayer.release();

            if (Config.confirmExit)
            {
                //AlertDialog.Builder builderQuit = new AlertDialog.Builder(this, R.style.Theme_AppCompat_Light_Dialog_Alert);
                AlertDialog.Builder builderQuit = new AlertDialog.Builder(this);
                builderQuit.setTitle(R.string.title_really_quit);
                builderQuit.setMessage(R.string.message_really_quit);
                final AlertDialog dialogQuit = builderQuit.create();

                /*
                 * OK! the user really wants to quit the test. Under these conditions, ask if she want
                 * save to the file too.
                 */
                DialogInterface.OnClickListener listenerQuitYes = new DialogInterface.OnClickListener()
                {
                    @Override
                    public void onClick(DialogInterface dialog, int which) {

                        if (mGraduatedMode)
                            if (Config.confirmProgress)
                                new AlertDialog.Builder(dialogQuit.getContext())
                                        .setTitle(R.string.title_really_quit)
                                        .setMessage(R.string.message_save)
                                        .setPositiveButton(R.string.button_yes,
                                                new DialogInterface.OnClickListener() {
                                                    @Override
                                                    public void onClick(DialogInterface dialog, int which) {
                                                        KvtmlWriter saver = new KvtmlWriter(Config.lastData,
                                                                Config.inputFile.getPath());
                                                        saver.save();
                                                        finish();
                                                    }
                                                }
                                        )
                                        .setNegativeButton(R.string.button_no,
                                                new DialogInterface.OnClickListener() {
                                                    @Override
                                                    public void onClick(DialogInterface dialog, int which) {
                                                        finish();
                                                    }
                                                }
                                        )
                                        .create()
                                        .show();
                            else
                            {
                                KvtmlWriter saver = new KvtmlWriter(Config.lastData, Config.inputFile.getPath());
                                saver.save();
                                finish();
                            }
                        else
                            finish();
                    }
                };

                builderQuit.setPositiveButton(R.string.button_yes, listenerQuitYes);
                builderQuit.setNegativeButton(R.string.button_no, null);
                builderQuit.show();
                return true;
            }
            else
            {
                if (mGraduatedMode)
                {
                    if (Config.confirmProgress)
                    {
                        new AlertDialog.Builder(this)
                                .setTitle(R.string.title_really_quit)
                                .setMessage(R.string.message_save)
                                .setPositiveButton(R.string.button_yes,
                                        new DialogInterface.OnClickListener()
                                        {
                                            @Override
                                            public void onClick(DialogInterface dialog, int which)
                                            {
                                                KvtmlWriter saver = new KvtmlWriter(Config.lastData,
                                                        Config.inputFile.getPath());
                                                saver.save();
                                                finish();
                                            }
                                        }
                                )
                                .setNegativeButton(R.string.button_no,
                                        new DialogInterface.OnClickListener()
                                        {
                                            @Override
                                            public void onClick(DialogInterface dialog, int which)
                                            {
                                                finish();
                                            }
                                        }
                                )
                                .create()
                                .show();
                    }
                    else
                    {
                        KvtmlWriter saver = new KvtmlWriter(Config.lastData, Config.inputFile.getPath());
                        saver.save();
                        finish();
                    }
                }
                else
                    finish();
            }
        }

        return super.onKeyDown(keyCode, event);
    }

    // It is automatically called by the system when a change of configuration of the Activity
    // was happened. When it happens, we save useful data, regards this instance, in an
    // TestStateHolder object before the instance dead. This date will be rescued calling the
    // system method "getLastCustomNonConfigurationInstance" and we can apply this date into the
    // new instance of the Activity.
    @Override
    public Object onRetainCustomNonConfigurationInstance()
    {
        TestStateHolder obj = new TestStateHolder();

        obj.entriesLeft = mEntriesLeft;
        obj.maxItems = mMaxItems;
        obj.maxQuestions = mMaxQuestions;
        obj.mistakes = mMistakes;
        obj.mistakesNum = mMistakesNum;
        obj.question = mQuestion;
        obj.questionData = mQuestionData;
        obj.mFilter = mFilter;

        return obj;
    }

    /**
     * Resets the test.
     *
     * @return True if the configuration changed.
     */
    @SuppressWarnings({"unchecked", "deprecation"})
    private final boolean resetTest()
    {
        // If a change of the state of the Activity was happened, we can rescue useful date of
        // the previous state. Otherwise obj will be null.
        TestStateHolder obj = (TestStateHolder) getLastNonConfigurationInstance();

        // Restore the state of the test upon configuration change:
        if (obj != null)
        {
            mEntriesLeft = obj.entriesLeft;
            mMaxItems = obj.maxItems;
            mMaxQuestions = obj.maxQuestions;
            mMistakes = obj.mistakes;
            mMistakesNum = obj.mistakesNum;
            mQuestion = obj.question;
            mQuestionData = obj.questionData;
            mFilter = obj.mFilter;

            return true;
        }

        // apply filters:
        HashMap<String, Kvtml.Entry> entries = (HashMap<String, Kvtml.Entry>) Config.lastData.entries.clone();

        mQuestionData = mFilter == null ? entries : mFilter.filter(entries);
        mMaxQuestions = mQuestionData.size();

        // For saving the collection of entries:
        idOfEntries = new HashSet<String>();
        for (Kvtml.Entry entry : mQuestionData.values())
            idOfEntries.add(entry.id);

        // If no more question, the Activity is finished and returns true:
        if (mMaxQuestions == 0)
        {
            Toast.makeText(this, R.string.message_no_entries_using_filters,
                                                                          Toast.LENGTH_LONG).show();
            this.finish();

            // make sure that getQuestion is not triggered
            return true;
        }

        // resetting other state variables
        mEntriesLeft = mMaxQuestions;
        mMistakesNum = 0;
        mMistakes.clear();

        return false;
    }

    /**
     * Resets the test and updates the UI.
     */
    protected void restart()
    {
        updateQuestion(resetTest());
    }

    /**
     * Sets the content view for the current test.
     */
    protected abstract void loadContentView();
}