/*
 * Libero Vocab
 *     An app for Android systems which allows to do practice with kvtml
 *     vocabulary files.
 *     This program is a fork of another program called "Vocab Drill" by:
 *       - Károly Kiripolszky <karcsi@ekezet.com>
 *       - Matthias Völlinger <matthias.voellinger@gmx.de>
 *
 *     Copyright (C) 2019-2021  Lo Iacono Massimo (massimol@inventati.org)
 *
 *     This program is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package org.inventati.massimol.liberovocab.dialogs;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.SortedSet;
import java.util.TreeSet;

import android.app.Activity;
import android.app.AlertDialog;
import android.content.DialogInterface;
import android.support.v7.app.AppCompatActivity;
import android.support.v7.app.ActionBar;

import android.content.Intent;
import android.os.Bundle;
import android.text.Editable;
import android.text.Html;
import android.text.TextWatcher;
import android.view.LayoutInflater;
import android.view.View;
import android.view.WindowManager.LayoutParams;
import android.widget.Button;
import android.widget.EditText;
import android.widget.Filter.FilterListener;
import android.widget.ListView;
import android.widget.AdapterView;
import android.widget.TextView;
import android.widget.Toast;

import org.inventati.massimol.liberovocab.Config;
import org.inventati.massimol.liberovocab.R;
import org.inventati.massimol.liberovocab.adapters.CollectionListAdapter;
import org.inventati.massimol.liberovocab.kvtml.Kvtml;

public class CollectionBrowserDialog extends AppCompatActivity
{
	public static final int SAVE_COLLECTION_OF_LESSONS = 1000;
	public static final int LOAD_COLLECTION_OF_LESSONS = 1001;
	public static final int MODIFY_COLLECTION_OF_LESSONS = 1002;
	public static final int SAVE_COLLECTION_OF_ENTRIES = 1003;
	public static final int PRACTICE_COLLECTION_OF_ENTRIES = 1004;
	public static final int MODIFY_COLLECTION_OF_ENTRIES = 1005;

	/**
	 * Caller Activity sets this.
	 */
	public static Kvtml data = null;

	/**
	 * Caller Activity sets this.
	 */
	private int mTypeOfCollection;

	/**
	 * Sort entries alphabetically if True.
	 */
	private boolean mSortAlphabetically = true;

	/**
	 * The name of the just selected collection:
	 */
	private String nameOfSelectedCollection;

	/**
	 * The EditText for the name of the collection:
	 */
	private EditText nameOfCollectionView;

	/**
	 * The EditText for the comment of the collection:
	 */
	private EditText commentOfCollectionView;

	/**
	 * True if the message in order to remember to save the file must be shown:
	 */
	private static boolean bShowMessageToSaveFile = false;

	private TextView mResultsNumText = null;
	private static CollectionListAdapter sAdapter = null;
	private String mFilterText = null;
	private FilterListener mFilterListener;

	@Override
	protected void onCreate(Bundle savedInstanceState)
	{
		super.onCreate(savedInstanceState);
		getWindow().setFlags(LayoutParams.FLAG_FULLSCREEN, LayoutParams.FLAG_FULLSCREEN);
		setContentView(R.layout.dialog_collection_browser);

		// This activity has an ActionBar:
		ActionBar actionBar = getSupportActionBar();
		actionBar.setHomeButtonEnabled(true);

		// In order to hide the icon in the ActionBar:
		//actionBar.setIcon(null);
		//actionBar.setIcon(android.R.color.transparent);

		// In order to customize the title in the ActionBar to be scrollable:
		actionBar.setDisplayShowCustomEnabled(true);
		actionBar.setDisplayShowTitleEnabled(false);

		// The customized TextView for the ActionBar:
		TextView mTextViewForActionBar;

		LayoutInflater inflator = LayoutInflater.from(this);
		View mCustomViewForActionBar = inflator.inflate(R.layout.view_action_bar_scrollable, null);
		mTextViewForActionBar = ((TextView) mCustomViewForActionBar.findViewById(R.id.action_bar_title));
		actionBar.setCustomView(mCustomViewForActionBar);

		// The label for description:
		TextView labelDescriptionView = (TextView) findViewById(R.id.decription);

		// The name of the collection:
		nameOfCollectionView = (EditText) findViewById(R.id.name_of_collection_edit);

		TextView labelNameOfCollectionView = (TextView) findViewById(R.id.name_of_collection_label);
		labelNameOfCollectionView.setText(R.string.name_of_collection);

		// The comment of the collection:
		commentOfCollectionView = (EditText) findViewById(R.id.comment_of_collection_edit);

		TextView labelCommentOfCollectionView = (TextView) findViewById(R.id.comment_of_collection_label);
		labelCommentOfCollectionView.setText(R.string.comment_of_collection);

		// The button Cancel:
		Button buttonCancel = (Button) findViewById(R.id.button_cancel);
		buttonCancel.setOnClickListener(new View.OnClickListener()
										{
											@Override
											public void onClick(View v)
											{
												if (bShowMessageToSaveFile)
													Toast.makeText(getApplicationContext(), getResources().getString(R.string.message_remember_save), Toast.LENGTH_LONG).show();
												bShowMessageToSaveFile = false;
												finish();
											}
										}
		);

		// The button Remove:
		Button buttonRemove = (Button) findViewById(R.id.button_remove);
		buttonRemove.setText(R.string.button_remove);
		buttonRemove.setOnClickListener(new View.OnClickListener()
										{
											@Override
											public void onClick(View v)
											{
												// The name of the collection must be not empty:
												if (nameOfCollectionView.getText().toString().equals(""))
												{
													Toast.makeText(getApplicationContext(), getResources().getString(R.string.message_name_of_collection_not_empty), Toast.LENGTH_SHORT).show();
													return;
												}

												performAction(true);
											}
										}
		);

		// The button which performs the action:
		Button buttonAction = (Button) findViewById(R.id.button_ok);
		buttonAction.setOnClickListener(new View.OnClickListener()
										{
											@Override
											public void onClick(View v)
											{
												// The name of the collection must be not empty:
												if (nameOfCollectionView.getText().toString().equals(""))
												{
													Toast.makeText(getApplicationContext(), getResources().getString(R.string.message_name_of_collection_not_empty), Toast.LENGTH_SHORT).show();
													return;
												}

												performAction(false);
											}
										}
		);

		// Get the Intent:
		Intent i = getIntent();
		if (i != null)
		{
			mSortAlphabetically = i.getBooleanExtra("doSort", mSortAlphabetically);
			mTypeOfCollection = i.getIntExtra("typeOfCollection", SAVE_COLLECTION_OF_LESSONS);
		}

		// Prepare the list of collect object that may be CollectionOfLessons or
		// CollectionOfEntries:
		ArrayList<Kvtml.Collect> collectList = new ArrayList<Kvtml.Collect>();

		// Set text of the title, actionBar, labels and buttons according to the different
		// type of collections managed by this dialog:
        if (mTypeOfCollection == SAVE_COLLECTION_OF_LESSONS)
		{
            collectList = new ArrayList<Kvtml.Collect>(data.collectionsOfLessons.values());

            // Set both the title of the Activity and the customized text of the ActionBar,
			// although the first isn't shown:
			setTitle(getResources().getString(R.string.menu_save_collection_of_lessons));
			mTextViewForActionBar.setText(getResources().getString(R.string.menu_save_collection_of_lessons));

			// The string for labels and buttons:
			labelDescriptionView.setText(R.string.description_save_collection_of_lessons);
			buttonAction.setText(R.string.button_save);
			buttonCancel.setText(R.string.button_cancel);
		}

		else if (mTypeOfCollection == LOAD_COLLECTION_OF_LESSONS)
		{
            collectList = new ArrayList<Kvtml.Collect>(data.collectionsOfLessons.values());

            // Set both the title of the Activity and the customized text of the ActionBar,
			// although the first isn't shown:
			setTitle(getResources().getString(R.string.menu_load_collection_of_lessons));
			mTextViewForActionBar.setText(getResources().getString(R.string.menu_load_collection_of_lessons));

			// The string for labels and buttons:
			labelDescriptionView.setText(R.string.description_load_collection_of_lessons);
			buttonAction.setText(R.string.button_load);
			buttonCancel.setText(R.string.button_cancel);
		}

		else if (mTypeOfCollection == MODIFY_COLLECTION_OF_LESSONS)
		{
            collectList = new ArrayList<Kvtml.Collect>(data.collectionsOfLessons.values());

            // Set both the title of the Activity and the customized text of the ActionBar,
			// although the first isn't shown:
			setTitle(getResources().getString(R.string.menu_modify_collection_of_lessons));
			mTextViewForActionBar.setText(getResources().getString(R.string.menu_modify_collection_of_lessons));

			// The string for labels and buttons:
			labelDescriptionView.setText(R.string.description_modify_collection_of_lessons);
			buttonAction.setText(R.string.button_modify);
			buttonCancel.setText(R.string.button_close);

			// The visibility of the remove button:
			buttonRemove.setVisibility(View.VISIBLE);
		}

		else if (mTypeOfCollection == SAVE_COLLECTION_OF_ENTRIES)
		{
            collectList = new ArrayList<Kvtml.Collect>(data.collectionsOfEntries.values());

            // Set both the title of the Activity and the customized text of the ActionBar,
			// although the first isn't shown:
			setTitle(getResources().getString(R.string.menu_save_collection_of_entries));
			mTextViewForActionBar.setText(getResources().getString(R.string.menu_save_collection_of_entries));

			// The string for labels and buttons:
			labelDescriptionView.setText(R.string.description_save_collection_of_entries);
			buttonAction.setText(R.string.button_save);
			buttonCancel.setText(R.string.button_cancel);
		}

		else if (mTypeOfCollection == PRACTICE_COLLECTION_OF_ENTRIES)
		{
            collectList = new ArrayList<Kvtml.Collect>(data.collectionsOfEntries.values());

            // Set both the title of the Activity and the customized text of the ActionBar,
			// although the first isn't shown:
			setTitle(getResources().getString(R.string.menu_practice_collection_of_entries));
			mTextViewForActionBar.setText(getResources().getString(R.string.menu_practice_collection_of_entries));

			// The string for labels and buttons:
			labelDescriptionView.setText(R.string.description_practice_collection_of_entries);
			buttonAction.setText(R.string.button_go);
			buttonCancel.setText(R.string.button_cancel);
		}

		else if (mTypeOfCollection == MODIFY_COLLECTION_OF_ENTRIES)
		{
            collectList = new ArrayList<Kvtml.Collect>(data.collectionsOfEntries.values());

            // Set both the title of the Activity and the customized text of the ActionBar,
			// although the first isn't shown:
			setTitle(getResources().getString(R.string.menu_modify_collection_of_entries));
			mTextViewForActionBar.setText(getResources().getString(R.string.menu_modify_collection_of_entries));

			// The string for labels and buttons:
			labelDescriptionView.setText(R.string.description_modify_collection_of_entries);
			buttonAction.setText(R.string.button_modify);
			buttonCancel.setText(R.string.button_close);

			// The visibility of the remove button:
			buttonRemove.setVisibility(View.VISIBLE);
		}

		// The list view:
		ListView list;

		list = (ListView) findViewById(android.R.id.list);
		list.setTextFilterEnabled(false);

		list.setOnItemClickListener(new AdapterView.OnItemClickListener()
			{
				@Override
				public void onItemClick(AdapterView<?> parent, View view, int position, long id)
				{
					Kvtml.Collect collect = (Kvtml.Collect) parent.getItemAtPosition(position);

					// Set the name of the just selected collection:
					nameOfSelectedCollection = collect.name;

					// Update the EditText view:
					nameOfCollectionView.setText(collect.name);
					commentOfCollectionView.setText(collect.comment);
				}
			}
		);

		// The list of items is prepared:
        loadEntries(collectList);

		// The list of items is added to the ListView object:
		list.setAdapter(sAdapter);

		// The text filter. The code will be here also if setTextFilterEnabled is set to false:
		mResultsNumText = (TextView) findViewById(R.id.text_results_num);
		mFilterListener = new FilterListener()
		{
			@Override
			public void onFilterComplete(int count)
			{
				if (mFilterText != null)
				{
					int ct = sAdapter.getCount();
					mResultsNumText.setText(Html.fromHtml(String.format("<b>Results: </b>%d", ct)));
					mResultsNumText.setVisibility(View.VISIBLE);
				}
				else
					mResultsNumText.setVisibility(View.GONE);
			}
		};

		EditText mEditTextFilter = null;
		mEditTextFilter = (EditText) findViewById(R.id.text_filter);
		mEditTextFilter.addTextChangedListener(new TextWatcher()
			{
				@Override
				public void onTextChanged(CharSequence s, int start, int before, int count)
				{
					if (s == null || s.toString().trim().length() == 0)
						mFilterText = null;
					else
						mFilterText = s.toString();
					sAdapter.getFilter().filter(mFilterText, mFilterListener);
				}

				@Override
				public void beforeTextChanged(CharSequence s, int start, int count, int after)
				{
				}

				@Override
				public void afterTextChanged(Editable s)
				{
				}
			}
		);
	}

	private void performAction(boolean bRemove)
	{
		// Considers the different type of collections managed by this dialog:
		if (mTypeOfCollection == SAVE_COLLECTION_OF_LESSONS)
			if (Config.lastData.existsCollectionOfLessons(nameOfCollectionView.getText().toString()))
				showConfirmationDialog(R.string.overwrite_collection, R.string.overwrite_collection_question, bRemove);
			else
				performActionAfterConfirmation(bRemove);

		else if (mTypeOfCollection == LOAD_COLLECTION_OF_LESSONS)
			if (Config.lastData.existsCollectionOfLessons(nameOfCollectionView.getText().toString()))
			{
				Config.lastData.loadCollectionOfLessons(nameOfCollectionView.getText().toString());

				// This to send information to the calling activity:
				final Intent data = new Intent();
				setResult(Activity.RESULT_OK, data);

				bShowMessageToSaveFile = false;
				finish();
			}
			else
				Toast.makeText(getApplicationContext(), getResources().getString(R.string.message_collection_does_not_exist), Toast.LENGTH_SHORT).show();

		else if (mTypeOfCollection == MODIFY_COLLECTION_OF_LESSONS)
			if (bRemove)
				if (Config.lastData.existsCollectionOfLessons(nameOfCollectionView.getText().toString()))
					showConfirmationDialog(R.string.remove_collection, R.string.remove_collection_question, bRemove);
				else
					Toast.makeText(getApplicationContext(), getResources().getString(R.string.message_collection_does_not_exist), Toast.LENGTH_SHORT).show();
			else
				if (nameOfSelectedCollection == null)
					Toast.makeText(getApplicationContext(), getResources().getString(R.string.message_a_collection_must_be_selected), Toast.LENGTH_SHORT).show();
				else
				{
					Config.lastData.getCollectionOfLessons(nameOfSelectedCollection).update(nameOfCollectionView.getText().toString(), commentOfCollectionView.getText().toString());
					initializeEditText();
					bShowMessageToSaveFile = true;
					recreate();
				}

		else if (mTypeOfCollection == SAVE_COLLECTION_OF_ENTRIES)
			if (Config.lastData.existsCollectionOfEntries(nameOfCollectionView.getText().toString()))
				showConfirmationDialog(R.string.overwrite_collection, R.string.overwrite_collection_question, bRemove);
			else
				performActionAfterConfirmation(bRemove);

		else if (mTypeOfCollection == PRACTICE_COLLECTION_OF_ENTRIES)
			if (Config.lastData.existsCollectionOfEntries(nameOfCollectionView.getText().toString()))
			{
				// This to send information to the calling activity:
				final Intent data = new Intent();
				data.putExtra("name_of_collection_of_entries", nameOfCollectionView.getText().toString());
				setResult(Activity.RESULT_OK, data);

				bShowMessageToSaveFile = false;
				finish();
			}
			else
				Toast.makeText(getApplicationContext(), getResources().getString(R.string.message_collection_does_not_exist), Toast.LENGTH_SHORT).show();

		else if (mTypeOfCollection == MODIFY_COLLECTION_OF_ENTRIES)
			if (bRemove)
				if (Config.lastData.existsCollectionOfEntries(nameOfCollectionView.getText().toString()))
					showConfirmationDialog(R.string.remove_collection, R.string.remove_collection_question, bRemove);
				else
					Toast.makeText(getApplicationContext(), getResources().getString(R.string.message_collection_does_not_exist), Toast.LENGTH_SHORT).show();
			else
				if (nameOfSelectedCollection == null)
					Toast.makeText(getApplicationContext(), getResources().getString(R.string.message_a_collection_must_be_selected), Toast.LENGTH_SHORT).show();
				else
				{
					Config.lastData.getCollectionOfEntries(nameOfSelectedCollection).update(nameOfCollectionView.getText().toString(), commentOfCollectionView.getText().toString());
					initializeEditText();
					bShowMessageToSaveFile = true;
					recreate();
				}
	}

	/**
	 * Show yes/no dialog for confirmation.
	 */
	private void showConfirmationDialog(int iTitle, int iMessage, final boolean bRemove)
	{
		String sMessage = getResources().getString(iMessage) + "\n\n" + nameOfCollectionView.getText().toString();

		new AlertDialog.Builder(this)
				.setTitle(iTitle)
				.setMessage(sMessage)
				.setCancelable(true)
				.setPositiveButton(R.string.button_yes,
						new DialogInterface.OnClickListener()
						{
							@Override
							public void onClick(DialogInterface dialog, int id)
							{
								performActionAfterConfirmation(bRemove);
							}
						}
				)
				.setNegativeButton(R.string.button_no,
						new DialogInterface.OnClickListener()
						{
							@Override
							public void onClick(DialogInterface dialog, int id)
							{
								dialog.cancel();
							}
						}
				)
				.create()
				.show();
	}

	private void performActionAfterConfirmation(boolean bRemove)
	{
		if (mTypeOfCollection == SAVE_COLLECTION_OF_LESSONS)
		{
			Config.lastData.saveCollectionOfLessons(nameOfCollectionView.getText().toString(), commentOfCollectionView.getText().toString());
			Toast.makeText(getApplicationContext(), getResources().getString(R.string.message_remember_save), Toast.LENGTH_LONG).show();
			bShowMessageToSaveFile = false;
			finish();
		}

		else if (mTypeOfCollection == MODIFY_COLLECTION_OF_LESSONS)
		{
			if (bRemove)
			{
				Config.lastData.removeCollectionOfLessons(nameOfCollectionView.getText().toString());
				initializeEditText();
				bShowMessageToSaveFile = true;
				recreate();
			}
		}

		else if (mTypeOfCollection == SAVE_COLLECTION_OF_ENTRIES)
		{
			// This to send information to the calling activity:
			final Intent data = new Intent();
			data.putExtra("name_of_collection_of_entries", nameOfCollectionView.getText().toString());
			data.putExtra("comment_of_collection_of_entries", commentOfCollectionView.getText().toString());
			setResult(Activity.RESULT_OK, data);

			Toast.makeText(getApplicationContext(), getResources().getString(R.string.message_remember_save), Toast.LENGTH_LONG).show();
			bShowMessageToSaveFile = false;
			finish();
		}

		else if (mTypeOfCollection == MODIFY_COLLECTION_OF_ENTRIES)
		{
			if (bRemove)
			{
				Config.lastData.removeCollectionOfEntries(nameOfCollectionView.getText().toString());
				initializeEditText();
				bShowMessageToSaveFile = true;
				recreate();
			}
		}
	}

	/**
	 * Set EditText for name and comment to empty string.
	 */
	private void initializeEditText()
	{
		// Update the EditText view:
		nameOfCollectionView.setText("");
		commentOfCollectionView.setText("");
	}

	/**
	 * TODO: it is possible to sort the list in other manners: for time.
	 */
	private void loadEntries(ArrayList<Kvtml.Collect> values)
	{
		if (data == null)
		{
			finish();
			return;
		}

		ArrayList<Kvtml.Collect> collections = null;
		if (mSortAlphabetically)
            collections = getSortedEntries(values);
		else
		{
			collections = new ArrayList<Kvtml.Collect>();
			collections.addAll(data.collectionsOfLessons.values());
		}

		if (collections == null)
		{
			Toast.makeText(this, R.string.error_loading_entries, Toast.LENGTH_LONG).show();
			finish();
			return;
		}

		Iterator<Kvtml.Collect> itera = collections.iterator();

		List<Kvtml.Collect> listItems = new ArrayList<Kvtml.Collect>();
		while (itera.hasNext())
			listItems.add(itera.next());

		sAdapter = new CollectionListAdapter(this, R.id.rowtext, listItems);
	}

	private ArrayList<Kvtml.Collect> getSortedEntries(ArrayList<Kvtml.Collect> values)
	{
		if (data == null)
			return null;

		SortedSet<Kvtml.Collect> sortedSet = null;
		Object[] sortedArray = null;

		// The ordination object:
		sortedSet = new TreeSet<Kvtml.Collect>(
			new Comparator<Kvtml.Collect>()
				{
					// The criteria in order to compare two CollectionOfLessons in alphabetic sense:
					@Override
					public int compare(Kvtml.Collect lhs, Kvtml.Collect rhs)
					{
						String s1 = lhs.name;
						String s2 = rhs.name;

						return s1.compareToIgnoreCase(s2);
					}
				}
		);

		try
		{
			sortedSet.addAll(values);
			sortedArray = sortedSet.toArray();
		}
		catch (Exception e)
		{
			return null;
		}

		if (sortedArray == null)
			return null;

		ArrayList<Kvtml.Collect> sortedCollections = new ArrayList<Kvtml.Collect>();
		Kvtml.Collect collection = null;

		for (int n = 0, mn = sortedArray.length; n < mn; n++)
		{
			collection = (Kvtml.Collect) sortedArray[n];
			if (collection == null)
				// XXX: must be reviewed: why not continue here?
				return null;
			sortedCollections.add(collection);
		}

		return sortedCollections;
	}
}