/*
 * Libero Vocab
 *     An app for Android systems which allows to do practice with kvtml
 *     vocabulary files.
 *     This program is a fork of another program called "Vocab Drill" by:
 *       - Károly Kiripolszky <karcsi@ekezet.com>
 *       - Matthias Völlinger <matthias.voellinger@gmx.de>
 *
 *     Copyright (C) 2019-2021  Lo Iacono Massimo (massimol@inventati.org)
 *
 *     This program is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package org.inventati.massimol.liberovocab.helpers;

import android.support.v7.app.AppCompatActivity;

import android.content.res.ColorStateList;
import android.graphics.Color;
import android.graphics.drawable.ColorDrawable;
import android.graphics.drawable.StateListDrawable;
import android.view.View;
import android.widget.TextView;

import org.inventati.massimol.liberovocab.R;
import org.inventati.massimol.liberovocab.Config;

import java.util.List;

public final class Gradient
{
	// The array containing the colors and brightness values:
	private static int[][][] sCache;

	// Used as indexes of the array:
	public final static int GRADIENT = 0;
	public final static int BRIGHTNESS = 1;

	// The colors for text, opposite to bright and dark background, for good readability:
	public static final int DARK_TEXT = 0xc8000000;
	public static final int BRIGHT_TEXT = 0xfcffffff;

	// Must be into the interval [0...100]:
	public static final int sBrightnessThreshold = 50;

	// The start and end colors of the gradient:
	private static final int color_start_RGB = Config.getColorStartForGradient();
	private static final int color_end_RGB = Config.getColorEndForGradient();

	/**
	 * Makes and returns an RGB gradient.
	 *
	 * @param steps Iteration count (must be greater than zero)
	 */
	public static int[] make(final int steps)
	{
		// The array to return:
		int[] colors = new int[steps];

		// In case steps = 1 the array returned will be composed only by the start color:
		if (steps == 1)
		{
			colors[0] = color_start_RGB;
			return colors;
		}

		// Here we handle the case which steps > 1 in:

		// The start color in the HSV color space:
		float[] color_start_HSV = new float[3];
		Color.colorToHSV(color_start_RGB, color_start_HSV);

		// The end color in the HSV color space:
		float[] color_end_HSV = new float[3];
		Color.colorToHSV(color_end_RGB, color_end_HSV);

		// The three dimensional vector for increment:
		float[] step = {(color_end_HSV[0] - color_start_HSV[0])/(steps -1),
						(color_end_HSV[1] - color_start_HSV[1])/(steps -1),
						(color_end_HSV[2] - color_start_HSV[2])/(steps -1)
					   };

		float[] new_color_HSV = new float[3];
		for (int i = 0; i < steps; i++)
		{
			// The i-th color of the gradient in the HSV space:
			new_color_HSV[0] = color_start_HSV[0] + i * step[0];
			new_color_HSV[1] = color_start_HSV[1] + i * step[1];
			new_color_HSV[2] = color_start_HSV[2] + i * step[2];

			// The i-th color of the gradient in the RGB space:
			colors[i] = Color.HSVToColor(255, new_color_HSV);
		}

		return colors;
	}

	public static int getBrightness(final int color)
	{
		float[] color_HSV = new float[3];
		Color.colorToHSV(color, color_HSV);

		return Math.round(color_HSV[2] * 100);
	}

	public static int[] getBrightnessArray(final int[] colors)
	{
		int[] ret = new int[colors.length];
		for (int i = 0; i < colors.length; i++)
			ret[i] = getBrightness(colors[i]);

		return ret;
	}

	public static void buildCache(final int steps)
	{
		if (sCache != null && steps <= sCache.length)
			return;

		int[][][] tmp = new int[steps][][];
		for (int i = 0; i < steps; i++)
		{
			tmp[i] = new int[2][];

			// +1 because cannot be zero
			tmp[i][GRADIENT] = make(i + 1);
			tmp[i][BRIGHTNESS] = getBrightnessArray(tmp[i][GRADIENT]);
		}

		sCache = tmp;
	}

	/**
	 * @param offset Start index in the colors array
	 */
	@SuppressWarnings("deprecation")
	public static void colorize(final List<View> views, final int offset)
	{
		final int steps = views.size();
		buildCache(steps + offset);

		final int[][] colors = sCache[(steps + offset) - 1];

		View v;
		StateListDrawable stlist;
		int btnbg;
		int[] fgcolors = new int[steps + offset];
		ColorDrawable activebgcolor;

		for (int i = 0; i < steps; i++)
		{
			v = views.get(i);
			btnbg = colors[GRADIENT][i + offset];
			fgcolors[i] = getReadableTextColorOverBackgroundColor(btnbg);

			if (0 < (i + offset))
				activebgcolor = new ColorDrawable(colors[GRADIENT][(i + offset) - 1]);
            else
                activebgcolor = new ColorDrawable(Color.TRANSPARENT);

			stlist = new StateListDrawable();
			stlist.addState(new int[]
									{
										android.R.attr.state_selected
									},
					        activebgcolor
							);

			stlist.addState(new int[]
									{
										android.R.attr.state_pressed
									},
							activebgcolor
							);

			stlist.addState(new int[]
									{
									},
							new ColorDrawable(btnbg)
							);

			v.setBackgroundDrawable(stlist);

			// In this moment we don't consider the first view (that with index 0):
			if (0 < i && v instanceof TextView)
				( (TextView) v ).setTextColor(new ColorStateList(
																new int[][]
																		{
																			new int[] { android.R.attr.state_selected },
																			new int[] { android.R.attr.state_pressed },
																			new int[0],
																		},

																new int[]
																		{
																			fgcolors[i - 1],
																			fgcolors[i - 1],
																			fgcolors[i],
																		}
																)
											 );
		}

        // Now we can consider the first view too:
		v = views.get(0);
        if (v instanceof TextView)
            ( (TextView) v ).setTextColor(new ColorStateList(
                                                            new int[][]
                                                                    {
                                                                            new int[] { android.R.attr.state_selected },
                                                                            new int[] { android.R.attr.state_pressed },
                                                                            new int[0],
                                                                    },

                                                            new int[]
                                                                    {
                                                                            fgcolors[steps - 1],
                                                                            fgcolors[steps - 1],
                                                                            fgcolors[0],
                                                                    }
                                                            )
                                         );
	}

	public static void colorize(final List<View> views)
	{
		colorize(views, 0);
	}

	public static int getReadableTextColorOverBackgroundColor(final int color)
	{
		if (getBrightness(color) < sBrightnessThreshold)
			return BRIGHT_TEXT;
		else
			return DARK_TEXT;
	}

	public static int getComplementary(final int color)
	{
		float[] color_HSV = new float[3];
		Color.colorToHSV(color, color_HSV);

		if (color_HSV[0] < 180)
			color_HSV[0] += 180;
		else
			color_HSV[0] -= 180;

		return Color.HSVToColor(color_HSV);
	}
}
