/*
 * Libero Vocab
 *     An app for Android systems which allows to do practice with kvtml
 *     vocabulary files.
 *     This program is a fork of another program called "Vocab Drill" by:
 *       - Károly Kiripolszky <karcsi@ekezet.com>
 *       - Matthias Völlinger <matthias.voellinger@gmx.de>
 *
 *     Copyright (C) 2019-2021  Lo Iacono Massimo (massimol@inventati.org)
 *
 *     This program is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package org.inventati.massimol.liberovocab.kvtml;

import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Locale;
import java.util.Set;
import java.text.SimpleDateFormat;

final public class Kvtml
{
	final public static int START = 1;
	final public static int END = 0;

	final public static String DOCTYPE = " kvtml PUBLIC \"kvtml2.dtd\" \"http://edu.kde.org/kvtml/kvtml2.dtd\"";

	final public static String VERSION = "2.0";

	final public static String ROOT_TAG = "kvtml";
	final public static String VERSION_TAG = "version";

	final public static String INFORMATION_TAG = "information";
	final public static String GENERATOR_TAG = "generator";
	final public static String TITLE_TAG = "title";
	final public static String DATE_TAG = "date";
	final public static String AUTHOR_TAG = "author";
	final public static String CONTACT_TAG = "contact";
	final public static String LICENSE_TAG = "license";
	final public static String CATEGORY_TAG = "category";

	final public static String IDENTIFIERS_TAG = "identifiers";
	final public static String IDENTIFIER_TAG = "identifier";
	final public static String ID_TAG = "id";
	final public static String NAME_TAG = "name";
	final public static String LOCALE_TAG = "locale";

	final public static String ENTRIES_TAG = "entries";
	final public static String ENTRY_TAG = "entry";
	final public static String COMMENT_TAG = "comment";
	final public static String TRANSLATION_TAG = "translation";
	final public static String TEXT_TAG = "text";
	final public static String SOUND_TAG = "sound";
	final public static String IMAGE_TAG = "image";

	final public static String GRADE_TAG = "grade";
	final public static String CURRENT_GRADE_TAG = "currentgrade";
	final public static String COUNT_TAG = "count";
	final public static String ERROR_COUNT_TAG = "errorcount";
	final public static String GRADE_DATE_TAG = "date";
	final public static String GRADE_DATE_DATE_FORMAT = "yyyy-MM-dd'T'HH:mm:ss";

	final public static String LESSONS_TAG = "lessons";
	final public static String CONTAINER_TAG = "container";
	final public static String INPRACTICE_TAG = "inpractice";

	final public static String COLLECTIONS_OF_LESSONS_TAG = "collections_of_lessons";
	final public static String COLLECTION_OF_LESSONS_TAG = "collection_of_lessons";

	final public static String COLLECTIONS_OF_ENTRIES_TAG = "collections_of_entries";
	final public static String COLLECTION_OF_ENTRIES_TAG = "collection_of_entries";

	public class Node
	{
		public String id;
	}

	public class Identifier extends Node
	{
		public String name;
		public String locale;
	}

	public class Grade
	{
		public int currentGrade;
		public int count;
		public int errorCount;
		public String date;

		public Grade()
		{
			currentGrade = 0;
			count = 0;
			errorCount = 0;
			date = "";
		}
	}

	public class Translation extends Node
	{
		public String text;
		public String sound;
		public String image;
		public String comment;
		public Grade grade;

		public Translation()
		{
			text = "";
			sound = "";
			image = "";
			comment = "";
			grade = new Grade();
		}

		public void setGrade(Grade iGrade)
		{
			grade = iGrade;
		}
	}

	public class Entry extends Node
	{
		// For efficiency in search we collect the Translations inside an Entry
		// using an HashMap object that contains pairs (Translation.id, Translation):
		public HashMap<String, Translation> translations;

		public Entry()
		{
			translations = new HashMap<String, Translation>();
		}
	}

	public class Lesson extends Node
	{
		public String name;

		// If the lesson is marked as in practice:
		public Boolean inpractice;

		// This set contains the ids for each entry in the lesson.
		public Set<String> idOfEntries;

		// This set contains the ids for each collectionOfLessons which the lesson belongs to.
		public Set<String> idOfCollectionsOfLessons;

		// A lesson can contains other lessons, that are his sublessons.
		// For efficiency in search we collect the Lessons inside a Lesson
		// using an HashMap contains pairs (Lesson.id, Lesson):
		public HashMap<String, Lesson> subLessons;

		public Lesson()
		{
			idOfEntries = new HashSet<String>();
			idOfCollectionsOfLessons = new HashSet<String>();
			subLessons = new HashMap<String, Lesson>();
		}
	}

	public class Collect extends Node
	{
		public String name;
		public String comment;
		public String date;

		Collect()
		{
			name = "";
			comment = "";
			date = "";
		}

		Collect(String name, String comment)
		{
			this.name = name;
			this.comment = comment;

			SimpleDateFormat sDF = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss", Locale.getDefault());
			Calendar now = Calendar.getInstance();
			date = sDF.format(now.getTime());
		}

		public void update()
		{
			SimpleDateFormat sDF = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss", Locale.getDefault());
			Calendar now = Calendar.getInstance();
			date = sDF.format(now.getTime());
		}

		public void update(String comment)
		{
			this.comment = comment;
            update();
		}

		public void update(String name, String comment)
		{
			this.name = name;
			update(comment);
		}
	}

	public class CollectionOfLessons extends Collect
	{
		/**
		 * Constructor.
		 */
		public CollectionOfLessons()
		{
			super();
		}

		/**
		 * Constructor. This new collection of lesson is conceived to be added to the
		 * collections of lessons of a Kvtml object witch is passed as parameter.
		 * The id will be automatically set incrementing of one the number of collections
		 * of lessons already present in the Kvtml object.
		 * The date will be set the same as the current date in the format "yyyy-MM-dd HH:mm:ss".
		 *
		 * @param kvtml This new collection of lesson will be added to the collections of
		 *              lessons of this Kvtml object passed as parameter.
		 * @param name The name of the collection.
		 * @param comment The comment of the collection.
		 */
		public CollectionOfLessons(Kvtml kvtml, String name, String comment)
		{
			super(name, comment);
			id = Integer.toString(kvtml.collectionsOfLessons.values().size());
		}
	}

	public class CollectionOfEntries extends Collect
	{
		// This set contains the ids for each entry in the lesson.
		public Set<String> idOfEntries;

		public CollectionOfEntries()
		{
			super();
			idOfEntries = new HashSet<String>();
		}

		/**
		 * Constructor. This new collection of entries is conceived to be added to the
		 * collections of entries of a Kvtml object witch is passed as parameter.
		 * The id will be automatically set incrementing of one the number of collections
		 * of entries already present in the Kvtml object.
		 * The date will be set the same as the current date in the format "yyyy-MM-dd HH:mm:ss".
		 *
		 * @param kvtml This new collection of entries will be added to the collections of
		 *              entries of this Kvtml object passed as parameter.
		 * @param name The name of the collection.
		 * @param comment The comment of the collection.
		 * @param idOfEntries the set of entries in the collection.
		 */
		public CollectionOfEntries(Kvtml kvtml, String name, String comment, HashSet<String> idOfEntries)
		{
			super(name, comment);
			id = Integer.toString(kvtml.collectionsOfEntries.values().size());
			this.idOfEntries = idOfEntries;
		}

        public void update(HashSet<String> idOfEntries)
        {
            this.idOfEntries = idOfEntries;
            update();
        }

		public void update(String comment, HashSet<String> idOfEntries)
		{
			this.idOfEntries = idOfEntries;
			update(comment);
		}

		public void update(String name, String comment, HashSet<String> idOfEntries)
		{
			this.idOfEntries = idOfEntries;
			update(name, comment);
		}
	}

	public String version;

	public String generator;
	public String title;
	public Date date;
	public String author;
	public String contact;
	public String license;
	public String comment;
	public String category;

	// For efficiency in search we collect the identifiers inside this
	// Kvtml object using an HashMap contains pairs (Identifier.id, Identifier):
	public HashMap<String, Identifier> identifiers;

	// For efficiency in search we collect the entries inside this
	// Kvtml object using an HashMap contains pairs (Entry.id, Entry):
	public HashMap<String, Entry> entries;

	// For efficiency in search we collect the lessons inside this
	// Kvtml object using an HashMap contains pairs (Lesson.id, Lesson):
	public HashMap<String, Lesson> lessons;

	// For efficiency in search we collect the collections of lessons inside this
	// Kvtml object using an HashMap contains pairs (CollectionOfLessons.id, CollectionOfLessons):
	public HashMap<String, CollectionOfLessons> collectionsOfLessons;

	// For efficiency in search we collect the dynamic collections of entries inside this
	// Kvtml object using an HashMap contains pairs (CollectionOfEntries.id, CollectionOfEntries):
	public HashMap<String, CollectionOfEntries> collectionsOfEntries;

	public Kvtml()
	{
		version = null;
		identifiers = new HashMap<String, Identifier>();
		entries = new HashMap<String, Entry>();
		lessons = new HashMap<String, Lesson>();
		collectionsOfLessons = new HashMap<String, CollectionOfLessons>();
		collectionsOfEntries = new HashMap<String, CollectionOfEntries>();
	}

	/**
	 * Recursively checks if there is at least one lesson with inpractice = true in a
	 * collection of lessons passed as parameter.
	 */
	public static boolean areInpracticeLessons(Collection<Lesson> colLessons)
	{
		if (colLessons.isEmpty())
			return false;
		else
		{
			for (Kvtml.Lesson lesson : colLessons)
			{
				if (lesson.inpractice)
					return true;
				else if (areInpracticeLessons(lesson.subLessons.values()))
					return true;
			}
			return false;
		}
	}

	/**
	 * Checks if there is at least one lesson to be practiced in this Kvtml instance.
	 */
	public boolean areInpracticeLessons()
	{
		return areInpracticeLessons(lessons.values());
	}

	/**
	 * Recursively add the id, passed as parameter, of a collection of lessons to all the
	 * lessons, member of a Collection<Lesson> passed also as parameter, with inpractice = true.
	 *
	 * @param colLessons The Collection<Lesson> in witch look for lessons with inpractice = true.
	 * @param id The id of the collection of lessons.
	 */
	private void addIdOfCollectionOfLessonsToInpracticeLessons(Collection<Lesson> colLessons, String id)
	{
		if (colLessons.isEmpty())
			return;

		for (Kvtml.Lesson lesson : colLessons)
		{
			if (lesson.inpractice)
				lesson.idOfCollectionsOfLessons.add(id);

			addIdOfCollectionOfLessonsToInpracticeLessons(lesson.subLessons.values(), id);
		}
	}

	/**
	 * Recursively remove the id, passed as parameter, of a collection of lessons to all the
	 * lessons, member of a Collection<Lesson> passed also as parameter, which have that id.
	 *
	 * @param colLessons The Collection<Lesson> in witch look for lessons which have the id passed
	 *                   as parameter.
	 * @param id The id of the collection of lessons.
	 */
	private void removeIdOfCollectionOfLessonsToInpracticeLessons(Collection<Lesson> colLessons, String id)
	{
		if (colLessons.isEmpty())
			return;

		for (Kvtml.Lesson lesson : colLessons)
		{
			if (lesson.idOfCollectionsOfLessons.contains(id))
				lesson.idOfCollectionsOfLessons.remove(id);

			removeIdOfCollectionOfLessonsToInpracticeLessons(lesson.subLessons.values(), id);
		}
	}

	/**
	 * Recursively set the flag inpractice = true in the lessons, member of a Collection<Lesson>
	 * passed as parameter, witch are referred by the collection of lessons with the id passed
	 * also as parameter.
	 *
	 * @param colLessons The Collection<Lesson> in witch look for lessons referred by the
	 *                   collection of lessons.
	 * @param id The id of the collection of lessons.
	 */
	private void setInpracticeToLessonsReferredByCollectionOfLessons(Collection<Lesson> colLessons, String id)
	{
		for (Kvtml.Lesson lesson : colLessons)
		{
			lesson.inpractice = lesson.idOfCollectionsOfLessons.contains(id);

			setInpracticeToLessonsReferredByCollectionOfLessons(lesson.subLessons.values(), id);
		}
	}

	/**
	 * In order to save, in a new collection of lessons, the reference to the subset of lessons
	 * with inpractice = true and add this new collection of lessons to the collections of lessons.
	 * Note that, because we do an upstream control, the fact this method is called implies that
	 * there is sure at least a lesson with inpractice = true.
	 *
	 * @param name The name of the collection.
	 * @param comment The comment of the collection.
	 */
	public void saveCollectionOfLessons(String name, String comment)
	{
		// We control if an other collection of lessons with the same name already exists and
		// in this case we substitute its comment and date. Otherwise a new collection of
		// lessons will be created.
		CollectionOfLessons colLessons = getCollectionOfLessons(name);

		if (colLessons != null)
		{
			// We update the existed collection of lessons object:
			colLessons.update(comment);

			// We remove the id of the found collection of lessons to all the lessons which
			// have that id.
			removeIdOfCollectionOfLessonsToInpracticeLessons(lessons.values(), colLessons.id);

			// Now we look for lessons with inpractice = true in order to add to them the id of the
			// found collection of lessons.
			addIdOfCollectionOfLessonsToInpracticeLessons(lessons.values(), colLessons.id);
		}
		else
		{
			// We create a new collection of lessons object:
			colLessons = new CollectionOfLessons(this, name, comment);

			// Now we look for lessons with inpractice = true in order to add to them the id of the
			// new collection of lessons just created.
			addIdOfCollectionOfLessonsToInpracticeLessons(lessons.values(), colLessons.id);

			// We finally add the new collection of lessons to the collections of lessons:
			collectionsOfLessons.put(colLessons.id, colLessons);
		}
	}

	/**
	 * To reset the subset of lessons with inpractice = true at the lessons member of the
	 * collection of lessons passed as parameter.
	 * @param name The name of the collection of lessons to load.
	 * @return False if doesn't exist any collection of lessons with the same name than passed.
	 *         Otherwise True.
	 */
	public boolean loadCollectionOfLessons(String name)
	{
		if (collectionsOfLessons.values().isEmpty())
			return false;

		for (Kvtml.CollectionOfLessons colLessons : collectionsOfLessons.values())
		{
			if (colLessons.name.equals(name))
			{
				// We have found a collection of lesson with the same name than passed.
				// We can proceed:

				setInpracticeToLessonsReferredByCollectionOfLessons(lessons.values(), colLessons.id);
				return true;
			}
		}
		return false;
	}

	/**
	 * Whether a collection of lessons witch name is egual to the
	 * string passed as parameter exists.
	 *
	 * @param name The name of the collection to find.
	 * @return true if a collection of lessons with the given name exists otherwise false.
	 */
	public boolean existsCollectionOfLessons(String name)
	{
		for (CollectionOfLessons colLessons : collectionsOfLessons.values())
		{
			if (colLessons.name.equals(name))
				return true;
		}

		return false;
	}

	/**
	 * In order to get the first collection of lessons witch name is egual to the
	 * string passed as parameter.
	 *
	 * @param name The name of the collection to find.
	 * @return The first collection of lesson found witch name is egual to the string
	 *         passed as parameter. If any collection of lesson doesn't exist with this name,
	 *         it will return null.
	 */
	public CollectionOfLessons getCollectionOfLessons(String name)
	{
		for (CollectionOfLessons colLessons : collectionsOfLessons.values())
		{
			if (colLessons.name.equals(name))
				return colLessons;
		}

		return null;
	}

	/**
	 * In order to remove the first collection of lessons witch name is egual to the
	 * string passed as parameter.
	 *
	 * @param name The name of the collection to remove.
	 * @return The id of the collection of lessons removed or null if any collection of
	 *         lessons doesn't exist with this name.
	 */
	public String removeCollectionOfLessons(String name)
	{
		// We control if an collection of lessons with the same name exists:
		CollectionOfLessons colLessons = getCollectionOfLessons(name);

		if (colLessons != null)
		{
			// We remove the found collection of lessons from the collections of lessons:
			collectionsOfLessons.remove(colLessons.id);

			// We remove the id of the found collection of lessons to all the lessons which
			// have that id.
			removeIdOfCollectionOfLessonsToInpracticeLessons(lessons.values(), colLessons.id);

			return colLessons.id;
		}
		else
			return null;
	}

	/**
	 * In order to save, in a new collection of entries, the ids to the subset of entries
	 * witch is practised and add this new collection of entries to the collections of entries.
	 *
	 * @param name The name of the collection.
	 * @param comment The comment of the collection.
	 * @param idOfEntries The id of the entries.
	 */
	public void saveCollectionOfEntries(String name, String comment, HashSet<String> idOfEntries)
	{
	    // We control if an other collection of entries with the same name already exists and
        // in this case we substitute its comment, date and idOfEntries. Otherwise a new
        // collection of entries will be created.
        CollectionOfEntries collEntries = getCollectionOfEntries(name);

        if (collEntries != null)
            // We update the existed collection of entries object:
            collEntries.update(comment, idOfEntries);
        else
        {
            // We create a new collection of entries object:
            collEntries = new CollectionOfEntries(this, name, comment, idOfEntries);

            // We add the new collection of entries to the collections of entries:
            collectionsOfEntries.put(collEntries.id, collEntries);
        }
	}

	/**
	 * Whether a collection of entries witch name is egual to the
	 * string passed as parameter exists.
	 *
	 * @param name The name of the collection to find.
	 * @return true if a collection of entries with the given name exists, otherwise false.
	 */
	public boolean existsCollectionOfEntries(String name)
	{
		for (CollectionOfEntries collEntries : collectionsOfEntries.values())
		{
			if (collEntries.name.equals(name))
				return true;
		}

		return false;
	}

    /**
     * In order to get the first collection of entries witch name is egual to the
     * string passed as parameter.
     *
     * @param name The name of the collection to find.
     * @return The first collection of entries found witch name is egual to the string
     *         passed as parameter. If any collection of entries doesn't exist with this name,
     *         it will return null.
     */
    public CollectionOfEntries getCollectionOfEntries(String name)
    {
        for (CollectionOfEntries collEntries : collectionsOfEntries.values())
        {
            if (collEntries.name.equals(name))
                return collEntries;
        }

        return null;
    }

    /**
     * In order to remove the first collection of entries witch name is egual to the
     * string passed as parameter.
     *
     * @param name The name of the collection to remove.
     * @return The id of the collection of entries removed or null if any collection of
     *         entries doesn't exist with this name.
     */
    public String removeCollectionOfEntries(String name)
    {
        // We control if an collection of entries with the same name exists:
        CollectionOfEntries collEntries = getCollectionOfEntries(name);

        if (collEntries != null)
        {
            // We remove the found collection of entries from the collections of entries:
            collectionsOfEntries.remove(collEntries.id);

            shiftCollectionsOfEntries(collEntries.id);
            return collEntries.id;
        }
        else
            return null;
    }

    /**
     * In order to ensure that the ids of the collections of entries respect an
     * entire sequence of integer without holes, we shift downward the ids superior
     * than that passed, supposing the id passed corresponds to an hole.
     * Typically this method will be called after a removal.
     *
     * @param id The id corresponding to the hole.
     * @return The number of collections of entries shifted downward.
     */
    private int shiftCollectionsOfEntries(String id)
    {
    	CollectionOfEntries collTemp;

		int iHole = Integer.valueOf(id);
		int numShifted = collectionsOfEntries.size() - iHole;
		for (int i = iHole + 1 ; i <= collectionsOfEntries.size() ; i++)
        {
			collTemp = collectionsOfEntries.get(String.valueOf(i));
			collTemp.id = String.valueOf(i - 1);

			collectionsOfEntries.remove(String.valueOf(i));
			collectionsOfEntries.put(String.valueOf(i - 1), collTemp);
		}
        return numShifted;
    }

	public Kvtml.Identifier newIdentifier()
	{
		return new Kvtml.Identifier();
	}

	public Kvtml.Grade newGrade()
	{
		return new Kvtml.Grade();
	}

	public Kvtml.Entry newEntry()
	{
		return new Kvtml.Entry();
	}

	public Kvtml.Translation newTranslation()
	{
		return new Kvtml.Translation();
	}

	public Kvtml.Lesson newLesson()
	{
		return new Kvtml.Lesson();
	}

	public Kvtml.CollectionOfLessons newCollectionOfLessons()
	{
		return new Kvtml.CollectionOfLessons();
	}

	public Kvtml.CollectionOfEntries newCollectionOfEntries()
	{
		return new Kvtml.CollectionOfEntries();
	}

	/**
	 * Clones a KVTML object containing the specified entries. Useful for creating
	 * test documents at runtime.
	 *
	 * @param clonedEntries
	 * @return A new KVTML document object.
	 */
	public Kvtml cloneFromEntries(HashMap<String, Kvtml.Entry> clonedEntries)
	{
		Kvtml clone = new Kvtml();

		clone.version = VERSION;
		clone.generator = new String(generator);
		clone.title = new String(title);
		clone.date = (Date) date.clone();
		clone.identifiers.putAll(identifiers);
		clone.entries.putAll(clonedEntries);
		clone.lessons.putAll(lessons);
		clone.collectionsOfLessons.putAll(collectionsOfLessons);
		clone.collectionsOfEntries.putAll(collectionsOfEntries);

		return clone;
	}
}