/*
 * Libero Vocab
 *     An app for Android systems which allows to do practice with kvtml
 *     vocabulary files.
 *     This program is a fork of another program called "Vocab Drill" by:
 *       - Károly Kiripolszky <karcsi@ekezet.com>
 *       - Matthias Völlinger <matthias.voellinger@gmx.de>
 *
 *     Copyright (C) 2019-2021  Lo Iacono Massimo (massimol@inventati.org)
 *
 *     This program is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package org.inventati.massimol.liberovocab.kvtml;

import org.xml.sax.InputSource;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.ParserConfigurationException;

import java.io.ByteArrayInputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Locale;
import java.util.Stack;

final public class KvtmlParserWithSax
{
	/**
	 * SAXParser instance.
	 */
	private SAXParser mParser;

	/**
	 * Kvtml object containing the parser output.
	 */
	private Kvtml mKvtml;

	/**
	 * Input stream reader.
	 */
	private KvtmlFileReader mFileReader;

	/**
	 * Path to the input KVTML file.
	 */
	private String mFilename;

	/*
	 *           =======================================
	 *           Start of the inner class:  KvtmlHandler
	 *           =======================================
	 */

	/**
	 * Handler class for the Kvtml parser.
	 */
	final private class KvtmlHandler extends DefaultHandler
	{
		private Kvtml mKvtml;

		private StringBuilder element_text = null;

		private Kvtml.Identifier identifier = null;
		private Kvtml.Entry entry = null;
		private Kvtml.Translation translation = null;
		private Kvtml.Grade grade = null;
		private Kvtml.CollectionOfLessons collectionOfLessons = null;
		private Kvtml.CollectionOfEntries collectionOfEntries = null;

		private Kvtml.Lesson lesson = null;
		private Kvtml.Lesson lessonParent = null;
		private Stack<Kvtml.Lesson> stackOfLesson = new Stack<>();
		int depthOfStack = 0;
		int idStack = 0;

		boolean bInformation = false;
		boolean bIdentifiers = false;
		boolean bEntries = false;
		boolean bLessons = false;
		boolean bCollections_of_lessons = false;
		boolean bCollections_of_entries = false;

		/**
		 * KvtmlHandler constructor.
		 *
		 * @param kvtml
		 */
		public KvtmlHandler(Kvtml kvtml)
		{
			this.mKvtml = kvtml;
		}

		/*
		 * Start of an element.
		 */
		@Override
		public void startElement(String uri, String localName, String qName, Attributes attributes)
		{
			// The cases witch a section is starting:
			if (qName.equalsIgnoreCase(Kvtml.ROOT_TAG))
				mKvtml.version = attributes.getValue(Kvtml.VERSION_TAG);
			else if (qName.equalsIgnoreCase(Kvtml.INFORMATION_TAG))
				bInformation = true;
			else if (qName.equalsIgnoreCase(Kvtml.IDENTIFIERS_TAG))
				bIdentifiers = true;
			else if (qName.equalsIgnoreCase(Kvtml.ENTRIES_TAG))
				bEntries = true;
			else if (qName.equalsIgnoreCase(Kvtml.LESSONS_TAG))
				bLessons = true;
			else if (qName.equalsIgnoreCase(Kvtml.COLLECTIONS_OF_LESSONS_TAG))
				bCollections_of_lessons = true;
			else if (qName.equalsIgnoreCase(Kvtml.COLLECTIONS_OF_ENTRIES_TAG))
				bCollections_of_entries = true;

				// The cases witch the parsing process is already inside a section:
			else if (bInformation)
				informationParsing(Kvtml.START, qName);
			else if (bIdentifiers)
				identifiersParsing(Kvtml.START, qName, attributes);
			else if (bEntries)
				entriesParsing(Kvtml.START, qName, attributes);
			else if (bLessons)
				lessonsParsing(Kvtml.START, qName, attributes);
			else if (bCollections_of_lessons)
				collectionsOfLessonsParsing(Kvtml.START, qName, attributes);
			else if (bCollections_of_entries)
				collectionsOfEntriesParsing(Kvtml.START, qName, attributes);

			element_text = new StringBuilder();
		}

		/*
		 * End of an element.
		 */
		@Override
		public void endElement(String uri, String localName, String qName)
		{
			// The cases witch a section is ending:
			if (qName.equalsIgnoreCase(Kvtml.INFORMATION_TAG))
				bInformation = false;
			else if (qName.equalsIgnoreCase(Kvtml.IDENTIFIERS_TAG))
				bIdentifiers = false;
			else if (qName.equalsIgnoreCase(Kvtml.ENTRIES_TAG))
				bEntries = false;
			else if (qName.equalsIgnoreCase(Kvtml.LESSONS_TAG))
				bLessons = false;
			else if (qName.equalsIgnoreCase(Kvtml.COLLECTIONS_OF_LESSONS_TAG))
				bCollections_of_lessons = false;
			else if (qName.equalsIgnoreCase(Kvtml.COLLECTIONS_OF_ENTRIES_TAG))
				bCollections_of_entries = false;

				// The cases witch the parsing process is already inside a section:
			else if (bInformation)
				informationParsing(Kvtml.END, qName);
			else if (bIdentifiers)
				identifiersParsing(Kvtml.END, qName, null);
			else if (bEntries)
				entriesParsing(Kvtml.END, qName, null);
			else if (bLessons)
				lessonsParsing(Kvtml.END, qName, null);
			else if (bCollections_of_lessons)
				collectionsOfLessonsParsing(Kvtml.END, qName, null);
			else if (bCollections_of_entries)
				collectionsOfEntriesParsing(Kvtml.END, qName, null);
		}

		/*
		 * For parsing the information section of the Kvtml file.
		 */
		private void informationParsing(int startOrEnd, String qName)
		{
			switch (startOrEnd) {
				case Kvtml.START:
					element_text = new StringBuilder();
					break;

				case Kvtml.END:
					if (qName.equalsIgnoreCase(Kvtml.GENERATOR_TAG))
						mKvtml.generator = element_text.toString();
					else if (qName.equalsIgnoreCase(Kvtml.TITLE_TAG))
						mKvtml.title = element_text.toString();
					else if (qName.equalsIgnoreCase(Kvtml.DATE_TAG))
						try
						{
							DateFormat df = new SimpleDateFormat("yyyy-MM-dd", Locale.getDefault());
							mKvtml.date = df.parse(element_text.toString());
						}
						catch (ParseException e)
						{
						}
					else if (qName.equalsIgnoreCase(Kvtml.AUTHOR_TAG))
						mKvtml.author = element_text.toString();
					else if (qName.equalsIgnoreCase(Kvtml.CONTACT_TAG))
						mKvtml.contact = element_text.toString();
					else if (qName.equalsIgnoreCase(Kvtml.LICENSE_TAG))
						mKvtml.license = element_text.toString();
					else if (qName.equalsIgnoreCase(Kvtml.COMMENT_TAG))
						mKvtml.comment = element_text.toString();
					else if (qName.equalsIgnoreCase(Kvtml.CATEGORY_TAG))
						mKvtml.category = element_text.toString();
					break;
			}
		}

		/*
		 * For parsing the identifiers section of the Kvtml file.
		 */
		private void identifiersParsing(int startOrEnd, String qName, Attributes attributes)
		{
			switch (startOrEnd)
			{
				case Kvtml.START:
					if (qName.equalsIgnoreCase(Kvtml.IDENTIFIER_TAG))
					{
						identifier = mKvtml.newIdentifier();
						identifier.id = attributes.getValue(Kvtml.ID_TAG);
					}

					element_text = new StringBuilder();
					break;

				case Kvtml.END:
					if (qName.equalsIgnoreCase(Kvtml.IDENTIFIER_TAG))
						mKvtml.identifiers.put(identifier.id, identifier);

					else if (qName.equalsIgnoreCase(Kvtml.NAME_TAG))
						identifier.name = element_text.toString();
					else if (qName.equalsIgnoreCase(Kvtml.LOCALE_TAG))
						identifier.locale = element_text.toString();
					break;
			}
		}

		/*
		 * For parsing the entries section of the Kvtml file.
		 */
		private void entriesParsing(int startOrEnd, String qName, Attributes attributes)
		{
			switch (startOrEnd) {
				case Kvtml.START:
					if (qName.equalsIgnoreCase(Kvtml.ENTRY_TAG))
					{
						entry = mKvtml.newEntry();
						entry.id = attributes.getValue(Kvtml.ID_TAG);
					}
					else if (qName.equalsIgnoreCase(Kvtml.TRANSLATION_TAG))
					{
						translation = mKvtml.newTranslation();
						translation.id = attributes.getValue(Kvtml.ID_TAG);
					}
					else if (qName.equalsIgnoreCase(Kvtml.GRADE_TAG))
						grade = mKvtml.newGrade();

					element_text = new StringBuilder();
					break;

				case Kvtml.END:
					if (qName.equalsIgnoreCase(Kvtml.GRADE_TAG))
					{
						if (grade.date == null)
							break;
						translation.setGrade(grade);
					}
					else if (qName.equalsIgnoreCase(Kvtml.TRANSLATION_TAG))
					{
						if (translation.text == null)
							break;
						if (0 == translation.text.length())
							break;
						entry.translations.put(translation.id, translation);
					}
					else if (qName.equalsIgnoreCase(Kvtml.ENTRY_TAG))
					{
						if (entry != null)
							if (entry.translations != null && entry.translations.size() > 0)
								mKvtml.entries.put(entry.id, entry);
					}
					else if (qName.equalsIgnoreCase(Kvtml.TEXT_TAG))
						translation.text = element_text.toString();
					else if (qName.equalsIgnoreCase(Kvtml.SOUND_TAG))
						translation.sound = element_text.toString();
					else if (qName.equalsIgnoreCase(Kvtml.IMAGE_TAG))
						translation.image = element_text.toString();
					else if (qName.equalsIgnoreCase(Kvtml.COMMENT_TAG))
						translation.comment = element_text.toString();
					else if (qName.equalsIgnoreCase(Kvtml.CURRENT_GRADE_TAG))
						grade.currentGrade = Integer.valueOf(element_text.toString());
					else if (qName.equalsIgnoreCase(Kvtml.COUNT_TAG))
						grade.count = Integer.valueOf(element_text.toString());
					else if (qName.equalsIgnoreCase(Kvtml.ERROR_COUNT_TAG))
						grade.errorCount = Integer.valueOf(element_text.toString());
					else if (qName.equalsIgnoreCase(Kvtml.GRADE_DATE_TAG))
						grade.date = element_text.toString();
					break;
			}
		}

		/*
		 * For parsing the lessons section of the Kvtml file.
		 */
		private void lessonsParsing(int startOrEnd, String qName, Attributes attributes)
		{
			/*
			 * We will use a stack of lessons to handle possible tree
			 * structures of lessons and considering that the entries
			 * of a lesson might get parsed after sublessons are handled:
			 */

			switch (startOrEnd)
			{
				case Kvtml.START:
					if (qName.equalsIgnoreCase(Kvtml.CONTAINER_TAG))
					{
						// Containers are lessons therefore we create a new lesson:
						lesson = mKvtml.newLesson();

						if (lessonParent == null)
						{
							lesson.id = Integer.toString(idStack);
							idStack++;
						}
						else
						{
							lesson.id = Integer.toString(lessonParent.subLessons.size());

							// The stack of lessons is updated adding the parent lesson:
							stackOfLesson.push(lessonParent);
						}

						// The parent lesson is now this lesson:
						lessonParent = lesson;
						depthOfStack++;
					}
					else if (qName.equalsIgnoreCase(Kvtml.ENTRY_TAG))
						lesson.idOfEntries.add(attributes.getValue(Kvtml.ID_TAG));
					else if (qName.equalsIgnoreCase(Kvtml.COLLECTION_OF_LESSONS_TAG))
						lesson.idOfCollectionsOfLessons.add(attributes.getValue(Kvtml.ID_TAG));

					element_text = new StringBuilder();
					break;

				case Kvtml.END:
					if (qName.equalsIgnoreCase(Kvtml.CONTAINER_TAG))
					{
						if (lesson.inpractice == null)
							lesson.inpractice = false;

						depthOfStack--;
						if (depthOfStack == 0)
						{
							mKvtml.lessons.put(lesson.id, lesson);
							lessonParent = null;
						}
						else
						{
							lessonParent = stackOfLesson.pop();
							lessonParent.subLessons.put(lesson.id, lesson);
							lesson = lessonParent;
						}
					}
					else if (qName.equalsIgnoreCase(Kvtml.NAME_TAG))
						lesson.name = element_text.toString();
					else if (qName.equalsIgnoreCase(Kvtml.INPRACTICE_TAG))
						lesson.inpractice = Boolean.valueOf(element_text.toString());
					break;
			}
		}

		/*
		 * For parsing the collections of lessons section of the Kvtml file.
		 */
		private void collectionsOfLessonsParsing(int startOrEnd, String qName, Attributes attributes)
		{
			switch (startOrEnd)
			{
				case Kvtml.START:
					if (qName.equalsIgnoreCase(Kvtml.COLLECTION_OF_LESSONS_TAG))
					{
						collectionOfLessons = mKvtml.newCollectionOfLessons();
						collectionOfLessons.id = attributes.getValue(Kvtml.ID_TAG);
					}

					element_text = new StringBuilder();
					break;

				case Kvtml.END:
					if (qName.equalsIgnoreCase(Kvtml.COLLECTION_OF_LESSONS_TAG))
						mKvtml.collectionsOfLessons.put(collectionOfLessons.id, collectionOfLessons);

					else if (qName.equalsIgnoreCase(Kvtml.NAME_TAG))
						collectionOfLessons.name = element_text.toString();
					else if (qName.equalsIgnoreCase(Kvtml.COMMENT_TAG))
						collectionOfLessons.comment = element_text.toString();
					else if (qName.equalsIgnoreCase(Kvtml.DATE_TAG))
						collectionOfLessons.date = element_text.toString();
					break;
			}
		}

		/*
		 * For parsing the collections of entries section of the Kvtml file.
		 */
		private void collectionsOfEntriesParsing(int startOrEnd, String qName, Attributes attributes)
		{
			switch (startOrEnd)
			{
				case Kvtml.START:
					if (qName.equalsIgnoreCase(Kvtml.COLLECTION_OF_ENTRIES_TAG))
					{
						collectionOfEntries = mKvtml.newCollectionOfEntries();
						collectionOfEntries.id = attributes.getValue(Kvtml.ID_TAG);
					}
					else if (qName.equalsIgnoreCase(Kvtml.ENTRY_TAG))
						collectionOfEntries.idOfEntries.add(attributes.getValue(Kvtml.ID_TAG));

					element_text = new StringBuilder();
					break;

				case Kvtml.END:
					if (qName.equalsIgnoreCase(Kvtml.COLLECTION_OF_ENTRIES_TAG))
						mKvtml.collectionsOfEntries.put(collectionOfEntries.id, collectionOfEntries);

					else if (qName.equalsIgnoreCase(Kvtml.NAME_TAG))
						collectionOfEntries.name = element_text.toString();
					else if (qName.equalsIgnoreCase(Kvtml.COMMENT_TAG))
						collectionOfEntries.comment = element_text.toString();
					else if (qName.equalsIgnoreCase(Kvtml.DATE_TAG))
						collectionOfEntries.date = element_text.toString();
					break;
			}
		}

		/*
		 * In order to get the element_text of an element.
		 */
		@Override
		public void characters(char ch[], int start, int length)
		{
			element_text.append(new String(ch, start, length));
		}

		/*
		 * In order to prevent the error message:
		 * "The markup declarations contained or pointed to by the document type declaration must be well-formed"
		 */
		@Override
		public InputSource resolveEntity(String publicId, String systemId)
		{
			return new InputSource(new ByteArrayInputStream(new byte[0]));
		}
	}

	/*
	 *           =====================================
	 *           End of the inner class:  KvtmlHandler
	 *           =====================================
	 */

	/**
	 * KvtmlParserWithSax constructor.
	 *
	 * @param filename
	 * @throws ParserConfigurationException
	 * @throws SAXException
	 * @throws IOException
	 */
	public KvtmlParserWithSax(String filename) throws ParserConfigurationException, SAXException, IOException
	{
		mFilename = filename;
		mFileReader = new KvtmlFileReader(mFilename);

		// Create the mParser object:
		SAXParserFactory factory = SAXParserFactory.newInstance();
		factory.setValidating(false);
		mParser = factory.newSAXParser();

		// Create the empty Kvtml object witch will container the structure of the
		// Kvtml vocabulary:
		mKvtml = new Kvtml();
	}

	/**
	 * Main parsing function.
	 *
	 * @return mKvtml
	 * @throws SAXException
	 * @throws IOException
	 */
	public Kvtml parse() throws SAXException, IOException
	{
		KvtmlHandler handler = new KvtmlHandler(mKvtml);
		mParser.parse(new FileInputStream(mFilename), handler);

		// Before return the Kvtml object, we ensure that certain values will not be null:
		convertNullValuesInSpaceInKvtmlObject();

		return mKvtml;
	}

	private void convertNullValuesInSpaceInKvtmlObject()
	{
		mKvtml.generator = mKvtml.generator == null ? "" : mKvtml.generator;
		mKvtml.title = mKvtml.title == null ? "" : mKvtml.title;
		mKvtml.author = mKvtml.author == null ? "" : mKvtml.author;
		mKvtml.contact = mKvtml.contact == null ? "" : mKvtml.contact;
		mKvtml.license = mKvtml.license == null ? "" : mKvtml.license;
		mKvtml.comment = mKvtml.comment == null ? "" : mKvtml.comment;
		mKvtml.category = mKvtml.category == null ? "" : mKvtml.category;
	}

	public KvtmlFileReader getFileReader()
	{
		return mFileReader;
	}
}