/*
 * Libero Vocab
 *     An app for Android systems which allows to do practice with kvtml
 *     vocabulary files.
 *     This program is a fork of another program called "Vocab Drill" by:
 *       - Károly Kiripolszky <karcsi@ekezet.com>
 *       - Matthias Völlinger <matthias.voellinger@gmx.de>
 *
 *     Copyright (C) 2019-2021  Lo Iacono Massimo (massimol@inventati.org)
 *
 *     This program is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package org.inventati.massimol.liberovocab.widgets;

import android.content.Context;
import android.content.res.TypedArray;
import android.preference.DialogPreference;
import android.util.AttributeSet;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.SeekBar;
import android.widget.SeekBar.OnSeekBarChangeListener;
import android.widget.TextView;

import org.inventati.massimol.liberovocab.R;

public class SeekBarPreference extends DialogPreference implements OnSeekBarChangeListener
{
	private static final String PREFERENCE_NS = "http://schemas.android.com/apk/res-auto";
	private static final String ANDROID_NS = "http://schemas.android.com/apk/res/android";

	private static final String ATTR_DEFAULT_VALUE = "defaultValue";
	private static final String ATTR_MIN_VALUE = "minValue";
	private static final String ATTR_MAX_VALUE = "maxValue";

	private SeekBar mSeekBar = null;

	private TextView mValueText = null;
	private String mSummaryText = null;
	private int mMinValue = 0;
	private int mMaxValue = 10;
	private int mProgress = mMinValue;
	private int mDefaultValue = mMinValue;

	public SeekBarPreference(Context context, AttributeSet attrs)
	{
		super(context, attrs);
		mMinValue = attrs.getAttributeIntValue(PREFERENCE_NS, ATTR_MIN_VALUE, mMinValue);
		mMaxValue = attrs.getAttributeIntValue(PREFERENCE_NS, ATTR_MAX_VALUE, mMaxValue);
		mProgress = attrs.getAttributeIntValue(ANDROID_NS, ATTR_DEFAULT_VALUE, getPersistedInt(mDefaultValue));
		setPersistent(true);
	}

	@Override
	protected View onCreateDialogView()
	{
		LayoutInflater inflater =
			(LayoutInflater) getContext().getSystemService(Context.LAYOUT_INFLATER_SERVICE);
		final View dlg = inflater.inflate(R.layout.dialog_seekbar_preference, null);

		mValueText = (TextView) dlg.findViewById(R.id.seek_preference_text);
		mValueText.setText(String.valueOf(mProgress));
		mSeekBar = (SeekBar) dlg.findViewById(R.id.seek_preference_bar);
		mSeekBar.setMax(mMaxValue - mMinValue);
		// -mMinValue because onProgressChanged is triggered
		mSeekBar.setProgress(mProgress - mMinValue);
		mSeekBar.setOnSeekBarChangeListener(this);

		return dlg;
	}

	@Override
	public void onProgressChanged(SeekBar seekBar, int progress, boolean fromUser)
	{
		mProgress = progress + mMinValue;
		mValueText.setText(String.valueOf(mProgress));
		callChangeListener(mProgress);
	}

	public void setSummaryPrefixText(int res)
	{
		mSummaryText = getContext().getResources().getString(res);
	}

	@Override
	public void setDefaultValue(Object defaultValue)
	{
		if (defaultValue == null)
			return;
		mProgress = mDefaultValue = (Integer) defaultValue;
		persistInt(mProgress);
		if (mSeekBar != null)
			mSeekBar.setProgress(mDefaultValue);
	}

	@Override
	protected void onDialogClosed(boolean positiveResult)
	{
		super.onDialogClosed(positiveResult);
		if (!positiveResult)
			return;
		if (shouldPersist())
			persistInt(mProgress);
		notifyChanged();
	}

	@Override
	public CharSequence getSummary()
	{
		return String.format("%s %s", mSummaryText, mProgress);
	}

	@Override
	public void onStartTrackingTouch(SeekBar seekBar)
	{
	}

	@Override
	public void onStopTrackingTouch(SeekBar seekBar)
	{
	}

	@Override
	protected Object onGetDefaultValue(TypedArray a, int index)
	{
		return Integer.valueOf(mDefaultValue);
	}
}