/*
   Copyright 2010 Johan Hilding

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

     http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
 */


package org.johanhil.flygtider.flightwatch;

import java.util.ArrayList;
import java.util.List;

import org.johanhil.flygtider.provider.FlightInfo;

import android.content.Context;
import android.content.Intent;
import android.database.Cursor;

public class FlightWatcher {
	
	private FlightWatchDatabaseAdapter flightWatchDbAdapter;
	private Context context;
	
	public FlightWatcher(Context context)
	{
	    this.context = context;
		this.flightWatchDbAdapter = new FlightWatchDatabaseAdapter(context).open();
	}
	
	public void add(FlightInfo flightInfo)
	{
		flightWatchDbAdapter.add(flightInfo);
		
		if (! FlightWatchBackgroundService.isRunning())
		{
		    final Intent serviceIntent = new Intent(context, FlightWatchBackgroundService.class);
		    context.startService(serviceIntent);
		}
	}
	
	public void remove(FlightInfo flightInfo)
	{
		flightWatchDbAdapter.deleteByFlightInfo(flightInfo);
		
		if (FlightWatchBackgroundService.isRunning() && getCount() == 0)
		{
		    final Intent serviceIntent = new Intent(context, FlightWatchBackgroundService.class);
            context.stopService(serviceIntent);
		}
	}
	
	public boolean isWatched(FlightInfo flightInfo)
	{
		Cursor cursor = flightWatchDbAdapter.fetchByFlightInfo(flightInfo);
		if (cursor.getCount() > 0)
		{
			return true;
		}
		else
		{
			return false;
		}
	}
	
	public int getCount()
	{
	    Cursor cursor = flightWatchDbAdapter.fetchAll();
	    return cursor.getCount();
	}
	
	public List<FlightInfo> getWatchedFlights()
	{
		Cursor cursor = flightWatchDbAdapter.fetchAll();
		List<FlightInfo> retList = new ArrayList<FlightInfo>();
		
		int dateColumn = cursor.getColumnIndexOrThrow(FlightWatchDatabaseAdapter.KEY_DATE);
		int departingAirportColumn = cursor.getColumnIndexOrThrow(FlightWatchDatabaseAdapter.KEY_DEPARTING_AIRPORT);
		int arrivingAirportColumn = cursor.getColumnIndexOrThrow(FlightWatchDatabaseAdapter.KEY_ARRIVING_AIRPORT);
		int arrivingColumn = cursor.getColumnIndex(FlightWatchDatabaseAdapter.KEY_ARRIVING);
		int flightNumberColumn = cursor.getColumnIndex(FlightWatchDatabaseAdapter.KEY_FLIGHT_NUMBER);
		int noteColumn = cursor.getColumnIndexOrThrow(FlightWatchDatabaseAdapter.KEY_NOTE);
		
		if (! cursor.moveToFirst())
			return retList;
		
		while (! cursor.isAfterLast())
		{
			FlightInfo flightInfo = new FlightInfo();
			flightInfo.setDate(cursor.getString(dateColumn));
			flightInfo.setDepartingAirport(cursor.getString(departingAirportColumn));
			flightInfo.setArrivingAirport(cursor.getString(arrivingAirportColumn));
			if (cursor.getInt(arrivingColumn) == 1)
			{
				flightInfo.setArriving(true);
			}
			else
			{
				flightInfo.setArriving(false);
			}
			flightInfo.setFlightNr(cursor.getString(flightNumberColumn));
			flightInfo.setNotes(cursor.getString(noteColumn));
			
			retList.add(flightInfo);
			cursor.moveToNext();
		}
		
		return retList;
	}
	
	public void close()
	{
		flightWatchDbAdapter.close();
	}
}
