/*
   Copyright 2010 Johan Hilding

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

     http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
 */


package org.johanhil.flygtider.ui;

import java.util.List;

import org.johanhil.flygtider.provider.FlightInfo;
import org.johanhil.flygtider.provider.FlightStatus;
import org.johanhil.flygtider.R;

import android.content.Context;
import android.graphics.Color;
import android.text.format.Time;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ArrayAdapter;
import android.widget.TextView;

public class FlightInfoAdapter extends ArrayAdapter<FlightInfo> {
	public static final int SIMPLE = 1; 
	public static final int DETAILED = 2;
	public static final int WATCHED_FLIGHTS = 3;
	
	private List<FlightInfo> flights;
	Context context;
	int style;
	
	/**
	 * The constructor for the adapter.
	 * @param style Style is either SIMPLE ([time] [airport]      [status]), 
	 * 								DETAILED ([date] [time] [flightnr] [airport]) or
	 * 								WATCHED_FLIGHTS ([dep_airport] -> [arr_airport])
	 * @param flights The list of flights to be used a source.
	 */
	public FlightInfoAdapter(Context context, int textViewResourceId, int style, List<FlightInfo> flights) 
	{
        super(context, textViewResourceId, flights);
        this.flights = flights;
        this.context = context;
        this.style = style;
	}

    public View getView(int position, View oldView, ViewGroup parent) {
        View v = oldView;
        
        if (v == null) {
            LayoutInflater vi = (LayoutInflater) context.getSystemService(Context.LAYOUT_INFLATER_SERVICE);
            v = vi.inflate(R.layout.flightinfoitem, null);
        }

        FlightInfo flightInfo = flights.get(position);
        
        if (flightInfo == null)
        {
        	return null;
        }
        
        TextView info = (TextView) v.findViewById(R.id.flightinfo);
        TextView status = (TextView) v.findViewById(R.id.flightstatus);

        if (info != null && status != null) {
        	FlightStatus flightStatus = FlightStatus.getStatus(flightInfo);
        	
    		status.setTextColor(getStatusTextColor(flightStatus));
    		status.setText(getStatusString(flightStatus));

        	StringBuffer infoBuffer = new StringBuffer();
        	
        	switch (style)
        	{
        	case FlightInfoAdapter.SIMPLE:
        		infoBuffer.append(flightInfo.getTime());
        		infoBuffer.append(" ");

        		if (flightInfo.isArriving())
        		{
        			infoBuffer.append(flightInfo.getDepartingAirport());
        		}
        		else
        		{
        			infoBuffer.append(flightInfo.getArrivingAirport());
        		}

        		info.setText(infoBuffer);
        		info.setTextColor(getInfoTextColor(flightStatus));
        		break;
        	
        	case FlightInfoAdapter.DETAILED:
        		// [date] [time] [flightnr] [airport]
        	    StringBuffer date = new StringBuffer();
        	    date.append(flightInfo.getDate().substring(6)).append('/').append(flightInfo.getDate().substring(3,5));
        	    infoBuffer.append(date).append(" ");
        		infoBuffer.append(flightInfo.getTime()).append(" ");
        		infoBuffer.append(flightInfo.getFlightNr()).append(" ");
                
        		if (flightInfo.isArriving())
                {
                    infoBuffer.append(flightInfo.getDepartingAirport());
                }
                else
                {
                    infoBuffer.append(flightInfo.getArrivingAirport());
                }
        		info.setText(infoBuffer);
        		info.setTextColor(getInfoTextColor(flightStatus));
        		
        		// TODO see below
        		status.setText("");
        		break;
        	
        	case FlightInfoAdapter.WATCHED_FLIGHTS:
        		infoBuffer.append(flightInfo.getDepartingAirport()).append(" ");
        		infoBuffer.append('\u2192').append(" ");
        		infoBuffer.append(flightInfo.getArrivingAirport());
        		
        		info.setText(infoBuffer);
        		info.setTextColor(Color.LTGRAY);
        		// TODO this ... could be a .setVisibility() but then we'd have to re-set its visibility... nggh.
        		status.setText("");
        		
        		break;
        	}
        }
        
        return v;
    }
    
    private CharSequence formatTime(Time t)
    {
    	StringBuffer sb = new StringBuffer();
    	if (t.hour < 10)
    		sb.append("0");
    	sb.append(t.hour);
    	sb.append(":");
    	
    	if (t.minute < 10)
    		sb.append("0");
    	sb.append(t.minute);
    	
    	return sb;
    }
    
    private CharSequence getStatusString(FlightStatus flightStatus)
    {
    	switch (flightStatus.getStatus())
    	{
    	case FlightStatus.NO_INFO:
    		return "";
    	case FlightStatus.BOARDING:
    		return "Boarding";
    	case FlightStatus.CANCELLED:
    		return "Inställd";
    	case FlightStatus.DELAYED:
    		return formatTime(flightStatus.getNewTime());
    	case FlightStatus.EARLY:
    		return formatTime(flightStatus.getNewTime());
    	case FlightStatus.LANDED:
    		return "Landat";
    	case FlightStatus.ON_TIME:
    		return formatTime(flightStatus.getNewTime());
    	case FlightStatus.STARTED:
    		return "Startat";
    	default:
    		return "";
    	}
    }
    
    private int getStatusTextColor(FlightStatus flightStatus)
    {
    	switch (flightStatus.getStatus())
    	{
    	case FlightStatus.CANCELLED:
    		return Color.RED;
    	case FlightStatus.DELAYED:
    		return Color.RED;
        case FlightStatus.INFO_COMING:
            return Color.RED;
    	case FlightStatus.EARLY:
    		return Color.GREEN;
    	case FlightStatus.LANDED:
    		return Color.GRAY;
    	case FlightStatus.ON_TIME:
    		return Color.LTGRAY;
    	case FlightStatus.STARTED:
    		return Color.GRAY;
    	default:
    		return Color.LTGRAY;
    	}
    }
    
    private int getInfoTextColor(FlightStatus flightStatus)
    {
    	switch (flightStatus.getStatus())
    	{
    	case FlightStatus.LANDED:
    		return Color.GRAY;
    	case FlightStatus.STARTED:
    		return Color.GRAY;
    	default:
    		return Color.LTGRAY;
    	}
    }
}
