/*
   Copyright 2010 Johan Hilding

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

     http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
 */

package org.johanhil.flygtider.ui;

import java.util.List;

import org.johanhil.flygtider.FlightInfoActivity;
import org.johanhil.flygtider.R;
import org.johanhil.flygtider.flightwatch.FlightWatcher;
import org.johanhil.flygtider.provider.FlightInfo;

import android.app.Activity;
import android.content.Intent;
import android.view.ContextMenu;
import android.view.Menu;
import android.view.MenuItem;
import android.view.View;
import android.view.ContextMenu.ContextMenuInfo;
import android.view.MenuItem.OnMenuItemClickListener;
import android.widget.AdapterView;
import android.widget.Toast;
import android.widget.AdapterView.AdapterContextMenuInfo;

public class FlightListItemClickHandler implements
		AdapterView.OnItemClickListener, View.OnCreateContextMenuListener, OnMenuItemClickListener 
{
	private FlightWatcher flightWatcher;
	private List<FlightInfo> flights;
	private FlightInfoAdapter adapter; // we use this to update the list.
	private Activity activity; // this has to be an activity as we use it to start other activities.
	private boolean needRefresh;
	
	private static final int SHOW_ID = 0;
	private static final int WATCH_UNWATCH_ID = 1;
	
	/**
	 * 
	 * @param activity
	 * @param flightWatcher
	 * @param flights
	 * @param needRefresh True if the flightInfo's in the flights needs a refresh. 
	 */
	public FlightListItemClickHandler(Activity activity, FlightWatcher flightWatcher, List<FlightInfo> flights, boolean needRefresh) {
		this.flightWatcher = flightWatcher;
		this.flights = flights;
		this.activity = activity;
		this.needRefresh = needRefresh;
	}
	
	public void setWatchedFlights(List<FlightInfo> flights)
	{
		this.flights = flights;
	}
	
    public void setAdapter(FlightInfoAdapter adapter)
    {
        this.adapter = adapter;
    }
	
	@Override
	public void onItemClick(AdapterView<?> parent, View v, int position, long id) {
		FlightInfo flightInfo = flights.get(position);
		showFlightInfo(flightInfo);
	}

	@Override
	public void onCreateContextMenu(ContextMenu menu, View v,
			ContextMenuInfo menuInfo) {
		
		AdapterContextMenuInfo info = (AdapterContextMenuInfo) menuInfo;
		FlightInfo flightInfo = flights.get(info.position);
		
		if (flightInfo.isArriving())
		{
			menu.setHeaderTitle(flightInfo.getDepartingAirport());
		}
		else
		{
			menu.setHeaderTitle(flightInfo.getArrivingAirport());
		}
		
		int label;
		if (flightWatcher.isWatched(flightInfo))
		{
			label = R.string.unwatch;
		}
		else
		{
			label = R.string.watch;
		}
		
		menu.add(Menu.NONE, SHOW_ID, Menu.NONE, R.string.show).setOnMenuItemClickListener(this);
		menu.add(Menu.NONE, WATCH_UNWATCH_ID, Menu.NONE, label).setOnMenuItemClickListener(this);
	}
	
	public boolean onMenuItemClick(MenuItem item)
	{
		AdapterContextMenuInfo info = (AdapterContextMenuInfo) item.getMenuInfo();
		FlightInfo flightInfo = flights.get(info.position);
		
		switch (item.getItemId())
		{
		case SHOW_ID:
			showFlightInfo(flightInfo);
			break;
		case WATCH_UNWATCH_ID:
			if (flightWatcher.isWatched(flightInfo))
			{
			    Toast.makeText(activity, R.string.watch_stopped, Toast.LENGTH_SHORT).show();
				flightWatcher.remove(flightInfo);
				
				// make the flight disappear from the watched flights List as well
				if (adapter != null)
				{
				    adapter.remove(flightInfo);
				    adapter.notifyDataSetChanged();
				}
			}
			else
			{
			    Toast.makeText(activity, R.string.watch_started, Toast.LENGTH_SHORT).show();
				flightWatcher.add(flightInfo);
			}
			break;
		}
		
		return true;
	}
	
    private void showFlightInfo(FlightInfo flightInfo)
    {
       	// start a new activity that shows info for the specific flight
    	Intent flightInfoActivity = new Intent(activity, FlightInfoActivity.class);
    	
    	//  we have to pass along the FlightInfo object for the clicked item.
    	flightInfoActivity.putExtra("flightinfo", flightInfo);
    	flightInfoActivity.putExtra("refresh", needRefresh);
    	    	
    	activity.startActivity(flightInfoActivity);
    }
}
