// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/TauFinder.hh"
#include "Rivet/Tools/Utils.hh"

namespace Rivet {


  /// @brief High-mass Drell-Yan prototype at 13 TeV
  class ATLAS_2025_I2905252 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(ATLAS_2025_I2905252);
    //@}


    /// @name Analysis methods
    //@{

    /// Book histograms and initialise projections before the run
    void init() {

      // Basic final-state projection for hadronic taus
      declare(TauFinder(TauDecay::HADRONIC, Cuts::abseta < 4.9), "hadtaus");

      book(_e, "mll");
      book(_h, "_mll", _e.binning().edges<0>());

    }

    /// Perform the per-event analysis
    void analyze(const Event& event) {

      // visible.
      const vector<Particle> hadtaus = apply<TauFinder>(event, "hadtaus").taus();

      FourMomentum ditau;
      vector<double> taus;
      for (const Particle& tau : hadtaus) {
        FourMomentum l;
        for (const Particle& p : tau.children()) {
          if (p.isNeutrino())  continue;
          ditau += p.mom();
          l += p.mom();
        }
        if ((l.abseta() > 1.37) && (l.abseta() < 1.52 || l.abseta() > 2.47))  continue;
        if (l.pT() < 20*GeV) continue;
        taus.push_back(l.pT());
      }
      if (taus.size() != 2) vetoEvent;

      std::sort(taus.begin(), taus.end(), std::greater<double>());
      if (taus[0] < 90*GeV) vetoEvent;
      if (taus[1] < 60*GeV) vetoEvent;

      const double mll = ditau.mass();
      if (mll <= 100*GeV)  vetoEvent;

      //histogram filling
      _h->fill(mll/GeV);

    }

    /// Normalise histograms etc., after the run
    void finalize() {

      scale(_h, crossSectionPerEvent()/femtobarn);
      *_e = _h->mkEstimate(_e->path(), "stats", true, 1.0);

    }

    //@}

    /// @name Histograms
    //@{
    Histo1DPtr _h;
    Estimate1DPtr _e;
    //@}

  };

  RIVET_DECLARE_PLUGIN(ATLAS_2025_I2905252);

}
