// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet
{
  /// @brief J/psi production in pPb collisions at $\sqrt{s_{NN}} = 5$ TeV
  class LHCB_2013_I1251899 : public Analysis
  {

  public:
    RIVET_DEFAULT_ANALYSIS_CTOR(LHCB_2013_I1251899);

    void init()
    {
      for (size_t i = 0; i < 2; ++i)
      {
        book(_int_fw_y_pt[i], 1, 1, i + 1);
        book(_int_bw_y_pt[i], 2, 1, i + 1);
        book(_fw_y_hist[i], 3, 1, i + 1);
        book(_bw_y_hist[i], 4, 1, i + 1);
        string hname = "d0" + std::to_string(i + 5) + "-x01-y0";
        book(_diff_fw_y_pt_hist[i], {1.5, 2.0, 2.5, 3.0, 3.5, 4.0}, 
          {hname + "1", hname + "2", hname + "3", hname + "4", hname + "5"});
        book(_fw_bw_ratio[i], 9, 1, i + 1);
        // Define temporary histograms with transverse-momentum intervals to compute
        // forward-to-backward ratios
        book(_fw_red_y_pt_hist[i], "TMP/_fw_red_y_pt_hist" + std::to_string(i+1), refData(9, 1, i + 1));
        book(_bw_red_y_pt_hist[i], "TMP/_bw_red_y_pt_hist" + std::to_string(i+1), refData(9, 1, i + 1));
      }

      // Select J/psi mesons
      declare(UnstableParticles(
                  (Cuts::abspid == PID::JPSI) &&
                  (Cuts::pT < 14. * GeV) &&
                  (Cuts::rapIn(1.5, 4.0) || Cuts::rapIn(-5.0, -2.5))),
              "UPs");
    }

    void analyze(const Event &ev)
    {
      Particles prompt_jpsi_mesons;
      Particles from_b_jpsi_mesons;
      size_t idx = -1; // 0 for prompt, 1 for from-b

      // Apply UnstableParticles projection to get entire decay chain of every particle
      const UnstableParticles &unst_parts = apply<UnstableParticles>(ev, "UPs");
      for (const Particle &part : unst_parts.particles())
      {
        double y = part.rapidity();
        double pt = part.pT();

        idx = (part.fromBottom()) ? 1 : 0;  // histogram index depending on promptness

        if (y > 0) { // forward region J/psi
          _int_fw_y_pt[idx]->fill(pt);
          _fw_y_hist[idx]->fill(y);
          _diff_fw_y_pt_hist[idx]->fill(y, pt);
          _fw_red_y_pt_hist[idx]->fill(pt);
        } else { // backward region J/psi
          _int_bw_y_pt[idx]->fill(pt);
          _bw_y_hist[idx]->fill(y);
          _bw_red_y_pt_hist[idx]->fill(pt);
        };
      }

    }

    void finalize()
    {
      // Compute scale factor with inelastic cross-section from input file and sum of
      // weights (corresponds to number of events in input file)
      const double scale_fact = crossSection() / microbarn / sumOfWeights();

      // Apply scale factor (histogram scaled by interval widths when running plotting
      // command [results in differential cross-section])
      for(size_t i = 0; i < 2; ++i) {
        scale(_int_fw_y_pt[i], scale_fact);
        scale(_int_bw_y_pt[i], scale_fact);
        scale(_fw_y_hist[i], scale_fact);
        scale(_bw_y_hist[i], scale_fact);
        _diff_fw_y_pt_hist[i]->divByGroupWidth();
        _diff_fw_y_pt_hist[i]->scaleW(scale_fact);
        // Compute forward-to-backward ratios
        divide(_fw_red_y_pt_hist[i], _bw_red_y_pt_hist[i], _fw_bw_ratio[i]);
      }
    
    }

    Histo1DPtr _int_fw_y_pt[2]; // 0 for prompt, 1 for from-b
    Histo1DPtr _int_bw_y_pt[2];
    Histo1DPtr _fw_y_hist[2];
    Histo1DPtr _bw_y_hist[2];
    Histo1DGroupPtr _diff_fw_y_pt_hist[2];
    Estimate1DPtr _fw_bw_ratio[2];
    Histo1DPtr _fw_red_y_pt_hist[2];
    Histo1DPtr _bw_red_y_pt_hist[2];
  };

  RIVET_DECLARE_PLUGIN(LHCB_2013_I1251899);
}
