/*
For general Scribus (>=1.3.2) copyright and licensing information please refer
to the COPYING file provided with the program. Following this notice may exist
a copyright and/or license notice that predates the release of Scribus 1.3.2
for which a new license (GPL+exception) is in place.
*/
#include <memory>

#include "importcvg.h"
#include "importcvgplugin.h"

#include "commonstrings.h"
#include "prefscontext.h"
#include "prefsfile.h"
#include "prefsmanager.h"
#include "scpage.h"
#include "scraction.h"
#include "scribuscore.h"
#include "undomanager.h"
#include "util_formats.h"

#include "ui/customfdialog.h"
#include "ui/scmwmenumanager.h"

int importcvg_getPluginAPIVersion()
{
	return PLUGIN_API_VERSION;
}

ScPlugin* importcvg_getPlugin()
{
	auto* plug = new ImportCvgPlugin();
	Q_CHECK_PTR(plug);
	return plug;
}

void importcvg_freePlugin(ScPlugin* plugin)
{
	ImportCvgPlugin* plug = qobject_cast<ImportCvgPlugin*>(plugin);
	Q_ASSERT(plug);
	delete plug;
}

ImportCvgPlugin::ImportCvgPlugin() :
	importAction(new ScrAction(ScrAction::DLL, "", QKeySequence(), this))
{
	// Set action info in languageChange, so we only have to do it in one
	// place. This includes registering file format support.
	registerFormats();
	languageChange();
}

void ImportCvgPlugin::languageChange()
{
	importAction->setText( tr("Import Calamus Vector Graphics..."));
	FileFormat* fmt = getFormatByExt("cvg");
	fmt->trName = FormatsManager::instance()->nameOfFormat(FormatsManager::CVG); // Human readable name
	fmt->filter = FormatsManager::instance()->extensionsForFormat(FormatsManager::CVG); // QFileDialog filter
}

ImportCvgPlugin::~ImportCvgPlugin()
{
	unregisterAll();
}

QString ImportCvgPlugin::fullTrName() const
{
	return QObject::tr("Calamus Vector Graphics Importer");
}


const ScActionPlugin::AboutData* ImportCvgPlugin::getAboutData() const
{
	auto* about = new AboutData;
	about->authors = "Franz Schmid <franz@scribus.info>";
	about->shortDescription = tr("Imports Calamus Vector Graphics Files");
	about->description = tr("Imports most Calamus Vector Graphics files into the current document, converting their vector data into Scribus objects.");
	about->license = "GPL";
	Q_CHECK_PTR(about);
	return about;
}

void ImportCvgPlugin::deleteAboutData(const AboutData* about) const
{
	Q_ASSERT(about);
	delete about;
}

void ImportCvgPlugin::registerFormats()
{
	FileFormat fmt(this);
	fmt.trName = FormatsManager::instance()->nameOfFormat(FormatsManager::CVG); // Human readable name
	fmt.formatId = 0;
	fmt.filter = FormatsManager::instance()->extensionsForFormat(FormatsManager::CVG); // QFileDialog filter
	fmt.fileExtensions = QStringList() << "cvg";
	fmt.load = true;
	fmt.save = false;
	fmt.thumb = true;
	fmt.mimeTypes = FormatsManager::instance()->mimetypeOfFormat(FormatsManager::CVG); // MIME types
	fmt.priority = 64; // Priority
	registerFormat(fmt);
}

bool ImportCvgPlugin::fileSupported(QIODevice* /* file */, const QString & fileName) const
{
	return true;
}

bool ImportCvgPlugin::loadFile(const QString & fileName, const FileFormat &, int flags, int /*index*/)
{
	// There's only one format to handle, so we just call importFile(...)
	return importFile(fileName, flags);
}

bool ImportCvgPlugin::importFile(QString fileName, int flags)
{
	if (!checkFlags(flags))
		return false;
	if (fileName.isEmpty())
	{
		flags |= lfInteractive;
		PrefsContext* prefs = PrefsManager::instance().prefsFile->getPluginContext("importcvg");
		QString wdir = prefs->get("wdir", ".");
		CustomFDialog diaf(ScCore->primaryMainWindow(), wdir, QObject::tr("Open"), tr("All Supported Formats")+" (*.cvg *.CVG);;All Files (*)");
		if (!diaf.exec())
			return true;
		fileName = diaf.selectedFile();
		prefs->set("wdir", fileName.left(fileName.lastIndexOf("/")));
	}

	m_Doc = ScCore->primaryMainWindow()->doc;

	UndoTransaction activeTransaction;
	bool emptyDoc = (m_Doc == nullptr);
	bool hasCurrentPage = (m_Doc && m_Doc->currentPage());

	TransactionSettings trSettings;
	trSettings.targetName   = hasCurrentPage ? m_Doc->currentPage()->getUName() : "";
	trSettings.targetPixmap = Um::IImageFrame;
	trSettings.actionName   = Um::ImportCVG;
	trSettings.description  = fileName;
	trSettings.actionPixmap = Um::IXFIG;
	if (emptyDoc || !(flags & lfInteractive) || !(flags & lfScripted))
		UndoManager::instance()->setUndoEnabled(false);
	if (UndoManager::undoEnabled())
		activeTransaction = UndoManager::instance()->beginTransaction(trSettings);

	auto dia = std::make_unique<CvgPlug>(m_Doc, flags);
	Q_CHECK_PTR(dia);
	dia->importFile(fileName, trSettings, flags, !(flags & lfScripted));

	if (activeTransaction)
		activeTransaction.commit();
	if (emptyDoc || !(flags & lfInteractive) || !(flags & lfScripted))
		UndoManager::instance()->setUndoEnabled(true);

	return true;
}

QImage ImportCvgPlugin::readThumbnail(const QString& fileName)
{
	if (fileName.isEmpty())
		return QImage();
	UndoManager::instance()->setUndoEnabled(false);
	m_Doc = nullptr;
	auto dia = std::make_unique<CvgPlug>(m_Doc, lfCreateThumbnail);
	Q_CHECK_PTR(dia);
	QImage ret = dia->readThumbnail(fileName);
	UndoManager::instance()->setUndoEnabled(true);
	return ret;
}
