# arborium-toml

[![Part of Arborium](https://img.shields.io/badge/🌳_Arborium-grammar_collection-228B22)](https://github.com/bearcove/arborium)
[![crates.io](https://img.shields.io/crates/v/arborium-toml.svg)](https://crates.io/crates/arborium-toml)
[![docs.rs](https://img.shields.io/docsrs/arborium-toml)](https://docs.rs/arborium-toml)
[![license](https://img.shields.io/crates/l/arborium-toml.svg)](https://github.com/bearcove/arborium)

[TOML](https://en.wikipedia.org/wiki/TOML) grammar for [tree-sitter](https://tree-sitter.github.io/).

> Tom's Obvious, Minimal Language for configuration; official spec at <a href="https://toml.io/en/">toml.io</a>.


| | |
|---|---|
| **Inventor** | Tom Preston-Werner |
| **Year** | 2013 |

## Usage

```rust,ignore
use arborium_toml::language;
use tree_sitter_patched_arborium::Parser;

let mut parser = Parser::new();
parser.set_language(&language()).expect("Error loading toml grammar");

let source_code = "/* your toml code here */";
let tree = parser.parse(source_code, None).unwrap();
println!("{}", tree.root_node().to_sexp());
```

For most use cases, prefer the main [arborium](https://crates.io/crates/arborium) crate:

```toml
[dependencies]
arborium = { version = "*", features = ["toml"] }
```

Or use this crate directly:

```toml
[dependencies]
arborium-toml = "*"
```

## About Arborium

This crate is part of [**Arborium**](https://github.com/bearcove/arborium), a collection of tree-sitter grammars packaged as Rust crates, maintained by [Amos Wenger](https://fasterthanli.me).

This crate packages a `parser.c` file generated by the [tree-sitter CLI](https://tree-sitter.github.io/tree-sitter/creating-parsers#generating-the-parser). It exports a `language()` function returning the tree-sitter `Language` for **toml**.

**Why separate crates?** Tree-sitter parsers can be large (some exceed 1MB of generated C). Splitting languages into separate crates means you only compile what you need, and parallel/incremental builds are more effective.

## Build Requirements

- **Rust**: MSRV is 1.85.0 (Rust 2024 edition)
- **C compiler**: This crate compiles C code via the [`cc`](https://crates.io/crates/cc) crate

### Supported Targets

- All major platforms: Linux, macOS, Windows
- WebAssembly: `wasm32-unknown-unknown` (with custom sysroot)

## Upstream Grammar

| | |
|---|---|
| **Repository** | [tree-sitter-grammars/tree-sitter-toml](https://github.com/tree-sitter-grammars/tree-sitter-toml) |
| **Commit** | `64b56832c2cffe41758f28e05c756a3a98d16f41` |
| **License** | MIT |

To regenerate after upstream updates:

```sh
cargo xtask gen --name toml
```

**Found a grammar bug?** Please report it to the [upstream repository](https://github.com/tree-sitter-grammars/tree-sitter-toml/issues).

## Safety

This crate uses `unsafe` for FFI bindings to the tree-sitter C library. The unsafe surface is minimal and limited to what tree-sitter requires. The generated parser code is produced by tree-sitter's own code generator.

## Versioning

Crate versions track Arborium releases, not upstream grammar versions. The upstream commit is recorded above. A major version bump occurs when grammar changes affect node names or structure (breaking for query consumers).

## License

- **Generated parser code** (`parser.c`): **MIT** (from upstream grammar)
- **Arborium glue code**: **MIT OR Apache-2.0**

## Contributing

- Issues: [github.com/bearcove/arborium/issues](https://github.com/bearcove/arborium/issues)
- Repository: [github.com/bearcove/arborium](https://github.com/bearcove/arborium)