// -*- mode: rust; -*-
//
// This file is part of curve25519-dalek.
// Copyright (c) 2016-2021 isis lovecruft
// Copyright (c) 2016-2019 Henry de Valence
// See LICENSE for licensing information.
//
// Authors:
// - isis agora lovecruft <isis@patternsinthevoid.net>
// - Henry de Valence <hdevalence@hdevalence.ca>

//! This module contains backend-specific constant values, such as the 64-bit limbs of curve constants.

use super::field::FieldElement51;
use super::scalar::Scalar52;
use crate::edwards::EdwardsPoint;

#[cfg(feature = "precomputed-tables")]
use crate::{
    backend::serial::curve_models::AffineNielsPoint,
    edwards::EdwardsBasepointTable,
    window::{LookupTable, NafLookupTable8},
};

/// The value of minus one, equal to `-&FieldElement::ONE`
pub(crate) const MINUS_ONE: FieldElement51 = FieldElement51::from_limbs([
    2251799813685228,
    2251799813685247,
    2251799813685247,
    2251799813685247,
    2251799813685247,
]);

/// Edwards `d` value, equal to `-121665/121666 mod p`.
pub(crate) const EDWARDS_D: FieldElement51 = FieldElement51::from_limbs([
    929955233495203,
    466365720129213,
    1662059464998953,
    2033849074728123,
    1442794654840575,
]);

/// Edwards `2*d` value, equal to `2*(-121665/121666) mod p`.
pub(crate) const EDWARDS_D2: FieldElement51 = FieldElement51::from_limbs([
    1859910466990425,
    932731440258426,
    1072319116312658,
    1815898335770999,
    633789495995903,
]);

/// One minus edwards `d` value squared, equal to `(1 - (-121665/121666) mod p) pow 2`
pub(crate) const ONE_MINUS_EDWARDS_D_SQUARED: FieldElement51 = FieldElement51::from_limbs([
    1136626929484150,
    1998550399581263,
    496427632559748,
    118527312129759,
    45110755273534,
]);

/// Edwards `d` value minus one squared, equal to `(((-121665/121666) mod p) - 1) pow 2`
pub(crate) const EDWARDS_D_MINUS_ONE_SQUARED: FieldElement51 = FieldElement51::from_limbs([
    1507062230895904,
    1572317787530805,
    683053064812840,
    317374165784489,
    1572899562415810,
]);

/// `= sqrt(a*d - 1)`, where `a = -1 (mod p)`, `d` are the Edwards curve parameters.
pub(crate) const SQRT_AD_MINUS_ONE: FieldElement51 = FieldElement51::from_limbs([
    2241493124984347,
    425987919032274,
    2207028919301688,
    1220490630685848,
    974799131293748,
]);

/// `= 1/sqrt(a-d)`, where `a = -1 (mod p)`, `d` are the Edwards curve parameters.
pub(crate) const INVSQRT_A_MINUS_D: FieldElement51 = FieldElement51::from_limbs([
    278908739862762,
    821645201101625,
    8113234426968,
    1777959178193151,
    2118520810568447,
]);

/// Precomputed value of one of the square roots of -1 (mod p)
pub(crate) const SQRT_M1: FieldElement51 = FieldElement51::from_limbs([
    1718705420411056,
    234908883556509,
    2233514472574048,
    2117202627021982,
    765476049583133,
]);

/// `APLUS2_OVER_FOUR` is (A+2)/4. (This is used internally within the Montgomery ladder.)
pub(crate) const APLUS2_OVER_FOUR: FieldElement51 =
    FieldElement51::from_limbs([121666, 0, 0, 0, 0]);

/// `MONTGOMERY_A` is equal to 486662, which is a constant of the curve equation
/// for Curve25519 in its Montgomery form. (This is used internally within the
/// Elligator map.)
pub(crate) const MONTGOMERY_A: FieldElement51 = FieldElement51::from_limbs([486662, 0, 0, 0, 0]);

/// `MONTGOMERY_A_NEG` is equal to -486662. (This is used internally within the
/// Elligator map.)
pub(crate) const MONTGOMERY_A_NEG: FieldElement51 = FieldElement51::from_limbs([
    2251799813198567,
    2251799813685247,
    2251799813685247,
    2251799813685247,
    2251799813685247,
]);

/// `L` is the order of base point, i.e. 2^252 + 27742317777372353535851937790883648493
pub(crate) const L: Scalar52 = Scalar52([
    0x0002631a5cf5d3ed,
    0x000dea2f79cd6581,
    0x000000000014def9,
    0x0000000000000000,
    0x0000100000000000,
]);

/// `L` * `LFACTOR` = -1 (mod 2^52)
pub(crate) const LFACTOR: u64 = 0x51da312547e1b;

/// `R` = R % L where R = 2^260
pub(crate) const R: Scalar52 = Scalar52([
    0x000f48bd6721e6ed,
    0x0003bab5ac67e45a,
    0x000fffffeb35e51b,
    0x000fffffffffffff,
    0x00000fffffffffff,
]);

/// `RR` = (R^2) % L where R = 2^260
pub(crate) const RR: Scalar52 = Scalar52([
    0x0009d265e952d13b,
    0x000d63c715bea69f,
    0x0005be65cb687604,
    0x0003dceec73d217f,
    0x000009411b7c309a,
]);

/// The Ed25519 basepoint, as an `EdwardsPoint`.
///
/// This is called `_POINT` to distinguish it from
/// `ED25519_BASEPOINT_TABLE`, which should be used for scalar
/// multiplication (it's much faster).
pub const ED25519_BASEPOINT_POINT: EdwardsPoint = EdwardsPoint {
    X: FieldElement51::from_limbs([
        1738742601995546,
        1146398526822698,
        2070867633025821,
        562264141797630,
        587772402128613,
    ]),
    Y: FieldElement51::from_limbs([
        1801439850948184,
        1351079888211148,
        450359962737049,
        900719925474099,
        1801439850948198,
    ]),
    Z: FieldElement51::from_limbs([1, 0, 0, 0, 0]),
    T: FieldElement51::from_limbs([
        1841354044333475,
        16398895984059,
        755974180946558,
        900171276175154,
        1821297809914039,
    ]),
};

/// The 8-torsion subgroup \\(\mathcal E \[8\]\\).
///
/// In the case of Curve25519, it is cyclic; the \\(i\\)-th element of
/// the array is \\(\[i\]P\\), where \\(P\\) is a point of order \\(8\\)
/// generating \\(\mathcal E\[8\]\\).
///
/// Thus \\(\mathcal E\[4\]\\) is the points indexed by `0,2,4,6`, and
/// \\(\mathcal E\[2\]\\) is the points indexed by `0,4`.
pub const EIGHT_TORSION: [EdwardsPoint; 8] = EIGHT_TORSION_INNER_DOC_HIDDEN;

/// Inner item used to hide limb constants from cargo doc output.
#[doc(hidden)]
pub const EIGHT_TORSION_INNER_DOC_HIDDEN: [EdwardsPoint; 8] = [
    EdwardsPoint {
        X: FieldElement51::from_limbs([0, 0, 0, 0, 0]),
        Y: FieldElement51::from_limbs([1, 0, 0, 0, 0]),
        Z: FieldElement51::from_limbs([1, 0, 0, 0, 0]),
        T: FieldElement51::from_limbs([0, 0, 0, 0, 0]),
    },
    EdwardsPoint {
        X: FieldElement51::from_limbs([
            358744748052810,
            1691584618240980,
            977650209285361,
            1429865912637724,
            560044844278676,
        ]),
        Y: FieldElement51::from_limbs([
            84926274344903,
            473620666599931,
            365590438845504,
            1028470286882429,
            2146499180330972,
        ]),
        Z: FieldElement51::from_limbs([1, 0, 0, 0, 0]),
        T: FieldElement51::from_limbs([
            1448326834587521,
            1857896831960481,
            1093722731865333,
            1677408490711241,
            1915505153018406,
        ]),
    },
    EdwardsPoint {
        X: FieldElement51::from_limbs([
            533094393274173,
            2016890930128738,
            18285341111199,
            134597186663265,
            1486323764102114,
        ]),
        Y: FieldElement51::from_limbs([0, 0, 0, 0, 0]),
        Z: FieldElement51::from_limbs([1, 0, 0, 0, 0]),
        T: FieldElement51::from_limbs([0, 0, 0, 0, 0]),
    },
    EdwardsPoint {
        X: FieldElement51::from_limbs([
            358744748052810,
            1691584618240980,
            977650209285361,
            1429865912637724,
            560044844278676,
        ]),
        Y: FieldElement51::from_limbs([
            2166873539340326,
            1778179147085316,
            1886209374839743,
            1223329526802818,
            105300633354275,
        ]),
        Z: FieldElement51::from_limbs([1, 0, 0, 0, 0]),
        T: FieldElement51::from_limbs([
            803472979097708,
            393902981724766,
            1158077081819914,
            574391322974006,
            336294660666841,
        ]),
    },
    EdwardsPoint {
        X: FieldElement51::from_limbs([0, 0, 0, 0, 0]),
        Y: FieldElement51::from_limbs([
            2251799813685228,
            2251799813685247,
            2251799813685247,
            2251799813685247,
            2251799813685247,
        ]),
        Z: FieldElement51::from_limbs([1, 0, 0, 0, 0]),
        T: FieldElement51::from_limbs([0, 0, 0, 0, 0]),
    },
    EdwardsPoint {
        X: FieldElement51::from_limbs([
            1893055065632419,
            560215195444267,
            1274149604399886,
            821933901047523,
            1691754969406571,
        ]),
        Y: FieldElement51::from_limbs([
            2166873539340326,
            1778179147085316,
            1886209374839743,
            1223329526802818,
            105300633354275,
        ]),
        Z: FieldElement51::from_limbs([1, 0, 0, 0, 0]),
        T: FieldElement51::from_limbs([
            1448326834587521,
            1857896831960481,
            1093722731865333,
            1677408490711241,
            1915505153018406,
        ]),
    },
    EdwardsPoint {
        X: FieldElement51::from_limbs([
            1718705420411056,
            234908883556509,
            2233514472574048,
            2117202627021982,
            765476049583133,
        ]),
        Y: FieldElement51::from_limbs([0, 0, 0, 0, 0]),
        Z: FieldElement51::from_limbs([1, 0, 0, 0, 0]),
        T: FieldElement51::from_limbs([0, 0, 0, 0, 0]),
    },
    EdwardsPoint {
        X: FieldElement51::from_limbs([
            1893055065632419,
            560215195444267,
            1274149604399886,
            821933901047523,
            1691754969406571,
        ]),
        Y: FieldElement51::from_limbs([
            84926274344903,
            473620666599931,
            365590438845504,
            1028470286882429,
            2146499180330972,
        ]),
        Z: FieldElement51::from_limbs([1, 0, 0, 0, 0]),
        T: FieldElement51::from_limbs([
            803472979097708,
            393902981724766,
            1158077081819914,
            574391322974006,
            336294660666841,
        ]),
    },
];

/// Table containing precomputed multiples of the Ed25519 basepoint \\(B = (x, 4/5)\\).
#[cfg(feature = "precomputed-tables")]
pub static ED25519_BASEPOINT_TABLE: &EdwardsBasepointTable =
    &ED25519_BASEPOINT_TABLE_INNER_DOC_HIDDEN;

/// Inner constant, used to avoid filling the docs with precomputed points.
#[doc(hidden)]
#[cfg(feature = "precomputed-tables")]
static ED25519_BASEPOINT_TABLE_INNER_DOC_HIDDEN: EdwardsBasepointTable = EdwardsBasepointTable([
    LookupTable([
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3540182452943730,
                2497478415033846,
                2521227595762870,
                1462984067271729,
                2389212253076811,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                62697248952638,
                204681361388450,
                631292143396476,
                338455783676468,
                1213667448819585,
            ]),
            xy2d: FieldElement51::from_limbs([
                301289933810280,
                1259582250014073,
                1422107436869536,
                796239922652654,
                1953934009299142,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3632771708514775,
                790832306631235,
                2067202295274102,
                1995808275510000,
                1566530869037010,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                463307831301544,
                432984605774163,
                1610641361907204,
                750899048855000,
                1894842303421586,
            ]),
            xy2d: FieldElement51::from_limbs([
                748439484463711,
                1033211726465151,
                1396005112841647,
                1611506220286469,
                1972177495910992,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1601611775252272,
                1720807796594148,
                1132070835939856,
                3512254832574799,
                2147779492816910,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                316559037616741,
                2177824224946892,
                1459442586438991,
                1461528397712656,
                751590696113597,
            ]),
            xy2d: FieldElement51::from_limbs([
                1850748884277385,
                1200145853858453,
                1068094770532492,
                672251375690438,
                1586055907191707,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                934282339813791,
                1846903124198670,
                1172395437954843,
                1007037127761661,
                1830588347719256,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1694390458783935,
                1735906047636159,
                705069562067493,
                648033061693059,
                696214010414170,
            ]),
            xy2d: FieldElement51::from_limbs([
                1121406372216585,
                192876649532226,
                190294192191717,
                1994165897297032,
                2245000007398739,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                769950342298400,
                2384754244604994,
                3095885746880802,
                3225892188161580,
                2977876099231263,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                425251763115706,
                608463272472562,
                442562545713235,
                837766094556764,
                374555092627893,
            ]),
            xy2d: FieldElement51::from_limbs([
                1086255230780037,
                274979815921559,
                1960002765731872,
                929474102396301,
                1190409889297339,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1388594989461809,
                316767091099457,
                2646098655878230,
                1230079486801004,
                1440737038838979,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                7380825640100,
                146210432690483,
                304903576448906,
                1198869323871120,
                997689833219095,
            ]),
            xy2d: FieldElement51::from_limbs([
                1181317918772081,
                114573476638901,
                262805072233344,
                265712217171332,
                294181933805782,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2916800678241215,
                2065379846933858,
                2622030924071124,
                2602788184473875,
                1233371373142984,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2019367628972465,
                676711900706637,
                110710997811333,
                1108646842542025,
                517791959672113,
            ]),
            xy2d: FieldElement51::from_limbs([
                965130719900578,
                247011430587952,
                526356006571389,
                91986625355052,
                2157223321444601,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                4320419353804412,
                4218074731744053,
                957728544705548,
                729906502578991,
                2411634706750414,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2073601412052185,
                31021124762708,
                264500969797082,
                248034690651703,
                1030252227928288,
            ]),
            xy2d: FieldElement51::from_limbs([
                551790716293402,
                1989538725166328,
                801169423371717,
                2052451893578887,
                678432056995012,
            ]),
        },
    ]),
    LookupTable([
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1368953770187805,
                3042147450398169,
                2689308289352409,
                2142576377050579,
                1932081720066286,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                953638594433374,
                1092333936795051,
                1419774766716690,
                805677984380077,
                859228993502513,
            ]),
            xy2d: FieldElement51::from_limbs([
                1200766035879111,
                20142053207432,
                1465634435977050,
                1645256912097844,
                295121984874596,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1735718747031538,
                1248237894295956,
                1204753118328107,
                976066523550493,
                2317743583219840,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1060098822528990,
                1586825862073490,
                212301317240126,
                1975302711403555,
                666724059764335,
            ]),
            xy2d: FieldElement51::from_limbs([
                1091990273418756,
                1572899409348578,
                80968014455247,
                306009358661350,
                1520450739132526,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3732317023121341,
                1511153322193951,
                3496143672676420,
                2556587964178488,
                2620936670181690,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2151330273626164,
                762045184746182,
                1688074332551515,
                823046109005759,
                907602769079491,
            ]),
            xy2d: FieldElement51::from_limbs([
                2047386910586836,
                168470092900250,
                1552838872594810,
                340951180073789,
                360819374702533,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1982622644432037,
                2014393600336956,
                2380709022489462,
                3869592437614438,
                2357094095599062,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                980234343912898,
                1712256739246056,
                588935272190264,
                204298813091998,
                841798321043288,
            ]),
            xy2d: FieldElement51::from_limbs([
                197561292938973,
                454817274782871,
                1963754960082318,
                2113372252160468,
                971377527342673,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2416499262514576,
                2254927265442919,
                3451304785234000,
                1766155447043651,
                1899238924683527,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                732262946680281,
                1674412764227063,
                2182456405662809,
                1350894754474250,
                558458873295247,
            ]),
            xy2d: FieldElement51::from_limbs([
                2103305098582922,
                1960809151316468,
                715134605001343,
                1454892949167181,
                40827143824949,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1239289043050193,
                1744654158124578,
                758702410031698,
                4048562808759936,
                2253402870349013,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2232056027107988,
                987343914584615,
                2115594492994461,
                1819598072792159,
                1119305654014850,
            ]),
            xy2d: FieldElement51::from_limbs([
                320153677847348,
                939613871605645,
                641883205761567,
                1930009789398224,
                329165806634126,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3232730304159378,
                1242488692177892,
                1251446316964684,
                1086618677993530,
                1961430968465772,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                276821765317453,
                1536835591188030,
                1305212741412361,
                61473904210175,
                2051377036983058,
            ]),
            xy2d: FieldElement51::from_limbs([
                833449923882501,
                1750270368490475,
                1123347002068295,
                185477424765687,
                278090826653186,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                794524995833413,
                1849907304548286,
                2305148486158393,
                1272368559505216,
                1147304168324779,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1504846112759364,
                1203096289004681,
                562139421471418,
                274333017451844,
                1284344053775441,
            ]),
            xy2d: FieldElement51::from_limbs([
                483048732424432,
                2116063063343382,
                30120189902313,
                292451576741007,
                1156379271702225,
            ]),
        },
    ]),
    LookupTable([
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3180171966714267,
                2147692869914563,
                1455665844462196,
                1986737809425946,
                2437006863943337,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                137732961814206,
                706670923917341,
                1387038086865771,
                1965643813686352,
                1384777115696347,
            ]),
            xy2d: FieldElement51::from_limbs([
                481144981981577,
                2053319313589856,
                2065402289827512,
                617954271490316,
                1106602634668125,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2948097833334040,
                3145099472726142,
                1148636718636008,
                2278533891034865,
                2203955659340680,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                657390353372855,
                998499966885562,
                991893336905797,
                810470207106761,
                343139804608786,
            ]),
            xy2d: FieldElement51::from_limbs([
                791736669492960,
                934767652997115,
                824656780392914,
                1759463253018643,
                361530362383518,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2022541353055578,
                4346500076272714,
                3802807888710933,
                2494585331103411,
                2947785218648809,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1287487199965223,
                2215311941380308,
                1552928390931986,
                1664859529680196,
                1125004975265243,
            ]),
            xy2d: FieldElement51::from_limbs([
                677434665154918,
                989582503122485,
                1817429540898386,
                1052904935475344,
                1143826298169798,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2619066141993637,
                2570231002607651,
                2947429167440602,
                2885885471266079,
                2276381426249673,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                773360688841258,
                1815381330538070,
                363773437667376,
                539629987070205,
                783280434248437,
            ]),
            xy2d: FieldElement51::from_limbs([
                180820816194166,
                168937968377394,
                748416242794470,
                1227281252254508,
                1567587861004268,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2730575372268893,
                2062896624554806,
                2951191072970647,
                2609899222113120,
                1277310261461760,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1984740906540026,
                1079164179400229,
                1056021349262661,
                1659958556483663,
                1088529069025527,
            ]),
            xy2d: FieldElement51::from_limbs([
                580736401511151,
                1842931091388998,
                1177201471228238,
                2075460256527244,
                1301133425678027,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1515728832059163,
                1575261009617579,
                1510246567196186,
                2442877836294952,
                2368461529974388,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1295295738269652,
                1714742313707026,
                545583042462581,
                2034411676262552,
                1513248090013606,
            ]),
            xy2d: FieldElement51::from_limbs([
                230710545179830,
                30821514358353,
                760704303452229,
                390668103790604,
                573437871383156,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3421179921230875,
                2514967047430861,
                4274701112739695,
                3071700566936367,
                4275698278559832,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2102254323485823,
                1570832666216754,
                34696906544624,
                1993213739807337,
                70638552271463,
            ]),
            xy2d: FieldElement51::from_limbs([
                894132856735058,
                548675863558441,
                845349339503395,
                1942269668326667,
                1615682209874691,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3539470031223082,
                1222355136884919,
                1846481788678694,
                1150426571265110,
                1613523400722047,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                793388516527298,
                1315457083650035,
                1972286999342417,
                1901825953052455,
                338269477222410,
            ]),
            xy2d: FieldElement51::from_limbs([
                550201530671806,
                778605267108140,
                2063911101902983,
                115500557286349,
                2041641272971022,
            ]),
        },
    ]),
    LookupTable([
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                717255318455100,
                519313764361315,
                2080406977303708,
                541981206705521,
                774328150311600,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                261715221532238,
                1795354330069993,
                1496878026850283,
                499739720521052,
                389031152673770,
            ]),
            xy2d: FieldElement51::from_limbs([
                1997217696294013,
                1717306351628065,
                1684313917746180,
                1644426076011410,
                1857378133465451,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3727234538477877,
                2328731709971226,
                3368528843456914,
                2002544139318041,
                2977347647489186,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2022306639183567,
                726296063571875,
                315345054448644,
                1058733329149221,
                1448201136060677,
            ]),
            xy2d: FieldElement51::from_limbs([
                1710065158525665,
                1895094923036397,
                123988286168546,
                1145519900776355,
                1607510767693874,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2813405189107769,
                1071733543815036,
                2383296312486238,
                1946868434569998,
                3079937947649451,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1548495173745801,
                442310529226540,
                998072547000384,
                553054358385281,
                644824326376171,
            ]),
            xy2d: FieldElement51::from_limbs([
                1445526537029440,
                2225519789662536,
                914628859347385,
                1064754194555068,
                1660295614401091,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3451490036797185,
                2275827949507588,
                2318438102929588,
                2309425969971222,
                2816893781664854,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                876926774220824,
                554618976488214,
                1012056309841565,
                839961821554611,
                1414499340307677,
            ]),
            xy2d: FieldElement51::from_limbs([
                703047626104145,
                1266841406201770,
                165556500219173,
                486991595001879,
                1011325891650656,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1622861044480487,
                1156394801573634,
                4120932379100752,
                2578903799462977,
                2095342781472283,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                334886927423922,
                489511099221528,
                129160865966726,
                1720809113143481,
                619700195649254,
            ]),
            xy2d: FieldElement51::from_limbs([
                1646545795166119,
                1758370782583567,
                714746174550637,
                1472693650165135,
                898994790308209,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2585203586724508,
                2547572356138185,
                1693106465353609,
                912330357530760,
                2723035471635610,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1811196219982022,
                1068969825533602,
                289602974833439,
                1988956043611592,
                863562343398367,
            ]),
            xy2d: FieldElement51::from_limbs([
                906282429780072,
                2108672665779781,
                432396390473936,
                150625823801893,
                1708930497638539,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                925664675702309,
                2273216662253932,
                4083236455546587,
                601157008940112,
                2623617868729744,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1479786007267725,
                1738881859066675,
                68646196476567,
                2146507056100328,
                1247662817535471,
            ]),
            xy2d: FieldElement51::from_limbs([
                52035296774456,
                939969390708103,
                312023458773250,
                59873523517659,
                1231345905848899,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2895154920100990,
                2541986621181021,
                2013561737429022,
                2571447883196794,
                2645536492181409,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                129358342392716,
                1932811617704777,
                1176749390799681,
                398040349861790,
                1170779668090425,
            ]),
            xy2d: FieldElement51::from_limbs([
                2051980782668029,
                121859921510665,
                2048329875753063,
                1235229850149665,
                519062146124755,
            ]),
        },
    ]),
    LookupTable([
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3859970785658325,
                2667608874045675,
                1350468408164765,
                2038620059057678,
                3278704299674360,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1837656083115103,
                1510134048812070,
                906263674192061,
                1821064197805734,
                565375124676301,
            ]),
            xy2d: FieldElement51::from_limbs([
                578027192365650,
                2034800251375322,
                2128954087207123,
                478816193810521,
                2196171989962750,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1633188840273120,
                3104586986058956,
                1548762607215795,
                1266275218902681,
                3359018017010381,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                462189358480054,
                1784816734159228,
                1611334301651368,
                1303938263943540,
                707589560319424,
            ]),
            xy2d: FieldElement51::from_limbs([
                1038829280972848,
                38176604650029,
                753193246598573,
                1136076426528122,
                595709990562434,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3660251634545082,
                2194984964010832,
                2198361797561729,
                1061962440055713,
                1645147963442934,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                4701053362120,
                1647641066302348,
                1047553002242085,
                1923635013395977,
                206970314902065,
            ]),
            xy2d: FieldElement51::from_limbs([
                1750479161778571,
                1362553355169293,
                1891721260220598,
                966109370862782,
                1024913988299801,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2464498862816952,
                1117950018299774,
                1873945661751056,
                3655602735669306,
                2382695896337945,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                636808533673210,
                1262201711667560,
                390951380330599,
                1663420692697294,
                561951321757406,
            ]),
            xy2d: FieldElement51::from_limbs([
                520731594438141,
                1446301499955692,
                273753264629267,
                1565101517999256,
                1019411827004672,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3178327305714638,
                3443653291096626,
                734233225181170,
                2435838701226518,
                4042225960010590,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1464651961852572,
                1483737295721717,
                1519450561335517,
                1161429831763785,
                405914998179977,
            ]),
            xy2d: FieldElement51::from_limbs([
                996126634382301,
                796204125879525,
                127517800546509,
                344155944689303,
                615279846169038,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2990523894660505,
                2188666632415295,
                1961313708559162,
                1506545807547587,
                3403101452654988,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                622917337413835,
                1218989177089035,
                1284857712846592,
                970502061709359,
                351025208117090,
            ]),
            xy2d: FieldElement51::from_limbs([
                2067814584765580,
                1677855129927492,
                2086109782475197,
                235286517313238,
                1416314046739645,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2838644076315587,
                2559244195637442,
                458399356043425,
                2853867838192310,
                3280348017100490,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                678489922928203,
                2016657584724032,
                90977383049628,
                1026831907234582,
                615271492942522,
            ]),
            xy2d: FieldElement51::from_limbs([
                301225714012278,
                1094837270268560,
                1202288391010439,
                644352775178361,
                1647055902137983,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1210746697896459,
                1416608304244708,
                2938287290903104,
                3496931005119382,
                3303038150540984,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1135604073198207,
                1683322080485474,
                769147804376683,
                2086688130589414,
                900445683120379,
            ]),
            xy2d: FieldElement51::from_limbs([
                1971518477615628,
                401909519527336,
                448627091057375,
                1409486868273821,
                1214789035034363,
            ]),
        },
    ]),
    LookupTable([
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1364039144731711,
                1897497433586190,
                2203097701135459,
                2397261210496499,
                1349844460790698,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1045230323257973,
                818206601145807,
                630513189076103,
                1672046528998132,
                807204017562437,
            ]),
            xy2d: FieldElement51::from_limbs([
                439961968385997,
                386362664488986,
                1382706320807688,
                309894000125359,
                2207801346498567,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3480804500082836,
                3172443782216110,
                2375775707596425,
                2933223806901024,
                1400559197080972,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2003766096898049,
                170074059235165,
                1141124258967971,
                1485419893480973,
                1573762821028725,
            ]),
            xy2d: FieldElement51::from_limbs([
                729905708611432,
                1270323270673202,
                123353058984288,
                426460209632942,
                2195574535456672,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1271140255321216,
                2044363183174497,
                2303925201319937,
                3696920060379952,
                3194341800024331,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1761608437466135,
                583360847526804,
                1586706389685493,
                2157056599579261,
                1170692369685772,
            ]),
            xy2d: FieldElement51::from_limbs([
                871476219910823,
                1878769545097794,
                2241832391238412,
                548957640601001,
                690047440233174,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2548994545820755,
                1366347803776819,
                3552985325930849,
                561849853336293,
                1533554921345731,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                999628998628371,
                1132836708493400,
                2084741674517453,
                469343353015612,
                678782988708035,
            ]),
            xy2d: FieldElement51::from_limbs([
                2189427607417022,
                699801937082607,
                412764402319267,
                1478091893643349,
                2244675696854460,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3964091869651792,
                2456213404310121,
                3657538451018088,
                2660781114515010,
                3112882032961968,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                508561155940631,
                966928475686665,
                2236717801150132,
                424543858577297,
                2089272956986143,
            ]),
            xy2d: FieldElement51::from_limbs([
                221245220129925,
                1156020201681217,
                491145634799213,
                542422431960839,
                828100817819207,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2405556784925632,
                1299874139923976,
                2644898978945750,
                1058234455773021,
                996989038681183,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                559086812798481,
                573177704212711,
                1629737083816402,
                1399819713462595,
                1646954378266038,
            ]),
            xy2d: FieldElement51::from_limbs([
                1887963056288059,
                228507035730124,
                1468368348640282,
                930557653420194,
                613513962454686,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1224529808187534,
                1577022856702685,
                2206946542980843,
                625883007765001,
                2531730607197406,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1076287717051609,
                1114455570543035,
                187297059715481,
                250446884292121,
                1885187512550540,
            ]),
            xy2d: FieldElement51::from_limbs([
                902497362940219,
                76749815795675,
                1657927525633846,
                1420238379745202,
                1340321636548352,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1129576631190765,
                3533793823712575,
                996844254743017,
                2509676177174497,
                3402650555740265,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                628740660038789,
                1943038498527841,
                467786347793886,
                1093341428303375,
                235413859513003,
            ]),
            xy2d: FieldElement51::from_limbs([
                237425418909360,
                469614029179605,
                1512389769174935,
                1241726368345357,
                441602891065214,
            ]),
        },
    ]),
    LookupTable([
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3988217766743784,
                726531315520507,
                1833335034432527,
                1629442561574747,
                2876218732971333,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1960754663920689,
                497040957888962,
                1909832851283095,
                1271432136996826,
                2219780368020940,
            ]),
            xy2d: FieldElement51::from_limbs([
                1537037379417136,
                1358865369268262,
                2130838645654099,
                828733687040705,
                1999987652890901,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                629042105241795,
                1098854999137608,
                887281544569320,
                3674901833560025,
                2259711072636808,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1811562332665373,
                1501882019007673,
                2213763501088999,
                359573079719636,
                36370565049116,
            ]),
            xy2d: FieldElement51::from_limbs([
                218907117361280,
                1209298913016966,
                1944312619096112,
                1130690631451061,
                1342327389191701,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1369976867854685,
                1396479602419169,
                4017456468084104,
                2203659200586298,
                3250127649802489,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2230701885562825,
                1348173180338974,
                2172856128624598,
                1426538746123771,
                444193481326151,
            ]),
            xy2d: FieldElement51::from_limbs([
                784210426627951,
                918204562375674,
                1284546780452985,
                1324534636134684,
                1872449409642708,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2571438643225542,
                2848082470493653,
                2037902696412607,
                1557219121643918,
                341938082688094,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1901860206695915,
                2004489122065736,
                1625847061568236,
                973529743399879,
                2075287685312905,
            ]),
            xy2d: FieldElement51::from_limbs([
                1371853944110545,
                1042332820512553,
                1949855697918254,
                1791195775521505,
                37487364849293,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                687200189577836,
                1082536651125675,
                2896024754556794,
                2592723009743198,
                2595381160432643,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2082717129583892,
                27829425539422,
                145655066671970,
                1690527209845512,
                1865260509673478,
            ]),
            xy2d: FieldElement51::from_limbs([
                1059729620568824,
                2163709103470266,
                1440302280256872,
                1769143160546397,
                869830310425069,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3861316033464273,
                777277757338816,
                2101121130363987,
                550762194946473,
                1905542338659364,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2024821921041576,
                426948675450149,
                595133284085473,
                471860860885970,
                600321679413000,
            ]),
            xy2d: FieldElement51::from_limbs([
                598474602406721,
                1468128276358244,
                1191923149557635,
                1501376424093216,
                1281662691293476,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1721138489890688,
                1264336102277790,
                2684864359106535,
                1359988423149465,
                3813671107094695,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                719520245587143,
                393380711632345,
                132350400863381,
                1543271270810729,
                1819543295798660,
            ]),
            xy2d: FieldElement51::from_limbs([
                396397949784152,
                1811354474471839,
                1362679985304303,
                2117033964846756,
                498041172552279,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1812471844975748,
                1856491995543149,
                126579494584102,
                3288044672967868,
                1975108050082549,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                650623932407995,
                1137551288410575,
                2125223403615539,
                1725658013221271,
                2134892965117796,
            ]),
            xy2d: FieldElement51::from_limbs([
                522584000310195,
                1241762481390450,
                1743702789495384,
                2227404127826575,
                1686746002148897,
            ]),
        },
    ]),
    LookupTable([
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                427904865186293,
                1703211129693455,
                1585368107547509,
                3688784302429584,
                3012988348299225,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                318101947455002,
                248138407995851,
                1481904195303927,
                309278454311197,
                1258516760217879,
            ]),
            xy2d: FieldElement51::from_limbs([
                1275068538599310,
                513726919533379,
                349926553492294,
                688428871968420,
                1702400196000666,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3313663849950481,
                3213411074010628,
                2573659446386085,
                3297400443644764,
                1985130202504037,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1558816436882417,
                1962896332636523,
                1337709822062152,
                1501413830776938,
                294436165831932,
            ]),
            xy2d: FieldElement51::from_limbs([
                818359826554971,
                1862173000996177,
                626821592884859,
                573655738872376,
                1749691246745455,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1988022651432119,
                3333911312271288,
                1834020786104820,
                3706626690108935,
                692929915223121,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2146513703733331,
                584788900394667,
                464965657279958,
                2183973639356127,
                238371159456790,
            ]),
            xy2d: FieldElement51::from_limbs([
                1129007025494441,
                2197883144413266,
                265142755578169,
                971864464758890,
                1983715884903702,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1291366624493056,
                2633256531874362,
                1711482489312443,
                1815233647702022,
                3144079596677715,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                444548969917454,
                1452286453853356,
                2113731441506810,
                645188273895859,
                810317625309512,
            ]),
            xy2d: FieldElement51::from_limbs([
                2242724082797924,
                1373354730327868,
                1006520110883049,
                2147330369940688,
                1151816104883620,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3997520014069025,
                4163522956860564,
                2056329390702073,
                2607026987995097,
                3131032608056347,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                163723479936298,
                115424889803150,
                1156016391581227,
                1894942220753364,
                1970549419986329,
            ]),
            xy2d: FieldElement51::from_limbs([
                681981452362484,
                267208874112496,
                1374683991933094,
                638600984916117,
                646178654558546,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2265178468539480,
                2358037120714814,
                1944412051589650,
                4093776581610705,
                2482502633520820,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                260683893467075,
                854060306077237,
                913639551980112,
                4704576840123,
                280254810808712,
            ]),
            xy2d: FieldElement51::from_limbs([
                715374893080287,
                1173334812210491,
                1806524662079626,
                1894596008000979,
                398905715033393,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2751826223412909,
                3848231101880618,
                1420380351989369,
                3237011375206737,
                392444930785632,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2096421546958141,
                1922523000950363,
                789831022876840,
                427295144688779,
                320923973161730,
            ]),
            xy2d: FieldElement51::from_limbs([
                1927770723575450,
                1485792977512719,
                1850996108474547,
                551696031508956,
                2126047405475647,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2112099158080129,
                2994370617594963,
                2258284371762679,
                1951119898618915,
                2344890196388664,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                383905201636970,
                859946997631870,
                855623867637644,
                1017125780577795,
                794250831877809,
            ]),
            xy2d: FieldElement51::from_limbs([
                77571826285752,
                999304298101753,
                487841111777762,
                1038031143212339,
                339066367948762,
            ]),
        },
    ]),
    LookupTable([
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2926794589205781,
                2517835660016036,
                826951213393477,
                1405007746162285,
                1781791018620876,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1001412661522686,
                348196197067298,
                1666614366723946,
                888424995032760,
                580747687801357,
            ]),
            xy2d: FieldElement51::from_limbs([
                1939560076207777,
                1409892634407635,
                552574736069277,
                383854338280405,
                190706709864139,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2177087163428741,
                1439255351721944,
                3459870654068041,
                2230616362004768,
                1396886392021913,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                676962063230039,
                1880275537148808,
                2046721011602706,
                888463247083003,
                1318301552024067,
            ]),
            xy2d: FieldElement51::from_limbs([
                1466980508178206,
                617045217998949,
                652303580573628,
                757303753529064,
                207583137376902,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3762856566592150,
                2357202940576524,
                2745234706458093,
                1091943425335975,
                1802717338077427,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1853982405405128,
                1878664056251147,
                1528011020803992,
                1019626468153565,
                1128438412189035,
            ]),
            xy2d: FieldElement51::from_limbs([
                1963939888391106,
                293456433791664,
                697897559513649,
                985882796904380,
                796244541237972,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2668570812315008,
                2641455366112301,
                1314476859406755,
                1749382513022778,
                3413705412424739,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1428358296490651,
                1027115282420478,
                304840698058337,
                441410174026628,
                1819358356278573,
            ]),
            xy2d: FieldElement51::from_limbs([
                204943430200135,
                1554861433819175,
                216426658514651,
                264149070665950,
                2047097371738319,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1934415182909015,
                1393285083565062,
                2768209145458208,
                3409490548679139,
                2372839480279515,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                662035583584445,
                286736105093098,
                1131773000510616,
                818494214211439,
                472943792054479,
            ]),
            xy2d: FieldElement51::from_limbs([
                665784778135882,
                1893179629898606,
                808313193813106,
                276797254706413,
                1563426179676396,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                945205108984213,
                2778077376644543,
                1324180513733565,
                1666970227868664,
                2405347422974421,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2031433403516252,
                203996615228162,
                170487168837083,
                981513604791390,
                843573964916831,
            ]),
            xy2d: FieldElement51::from_limbs([
                1476570093962618,
                838514669399805,
                1857930577281364,
                2017007352225784,
                317085545220047,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1461557121912823,
                1600674043318359,
                2157134900399597,
                1670641601940616,
                2379565397488531,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1293543509393474,
                2143624609202546,
                1058361566797508,
                214097127393994,
                946888515472729,
            ]),
            xy2d: FieldElement51::from_limbs([
                357067959932916,
                1290876214345711,
                521245575443703,
                1494975468601005,
                800942377643885,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2817916472785262,
                820247422481739,
                994464017954148,
                2578957425371613,
                2344391131796991,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                617256647603209,
                1652107761099439,
                1857213046645471,
                1085597175214970,
                817432759830522,
            ]),
            xy2d: FieldElement51::from_limbs([
                771808161440705,
                1323510426395069,
                680497615846440,
                851580615547985,
                1320806384849017,
            ]),
        },
    ]),
    LookupTable([
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1219260086131896,
                2898968820282063,
                2331400938444953,
                2161724213426747,
                2656661710745446,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1327968293887866,
                1335500852943256,
                1401587164534264,
                558137311952440,
                1551360549268902,
            ]),
            xy2d: FieldElement51::from_limbs([
                417621685193956,
                1429953819744454,
                396157358457099,
                1940470778873255,
                214000046234152,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1268047918491954,
                2172375426948536,
                1533916099229249,
                1761293575457130,
                3842422480712013,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1627072914981959,
                2211603081280073,
                1912369601616504,
                1191770436221309,
                2187309757525860,
            ]),
            xy2d: FieldElement51::from_limbs([
                1149147819689533,
                378692712667677,
                828475842424202,
                2218619146419342,
                70688125792186,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3551539230764990,
                3690416477138006,
                3788528892189659,
                2053896748919837,
                3260220846276494,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2040723824657366,
                399555637875075,
                632543375452995,
                872649937008051,
                1235394727030233,
            ]),
            xy2d: FieldElement51::from_limbs([
                2211311599327900,
                2139787259888175,
                938706616835350,
                12609661139114,
                2081897930719789,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1324994503390431,
                2588782144267879,
                1183998925654176,
                3343454479598522,
                2300527487656566,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1845522914617879,
                1222198248335542,
                150841072760134,
                1927029069940982,
                1189913404498011,
            ]),
            xy2d: FieldElement51::from_limbs([
                1079559557592645,
                2215338383666441,
                1903569501302605,
                49033973033940,
                305703433934152,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2346453219102138,
                3637921163538246,
                3313930291577009,
                2288353761164521,
                3085469462634093,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1432015813136298,
                440364795295369,
                1395647062821501,
                1976874522764578,
                934452372723352,
            ]),
            xy2d: FieldElement51::from_limbs([
                1296625309219774,
                2068273464883862,
                1858621048097805,
                1492281814208508,
                2235868981918946,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1490330266465551,
                1858795661361448,
                3688040948655011,
                2546373032584894,
                3459939824714180,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1282462923712748,
                741885683986255,
                2027754642827561,
                518989529541027,
                1826610009555945,
            ]),
            xy2d: FieldElement51::from_limbs([
                1525827120027511,
                723686461809551,
                1597702369236987,
                244802101764964,
                1502833890372311,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2365421849929742,
                3485539881431101,
                2925909765963743,
                2114345180342964,
                2418564326541511,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2041668749310338,
                2184405322203901,
                1633400637611036,
                2110682505536899,
                2048144390084644,
            ]),
            xy2d: FieldElement51::from_limbs([
                503058759232932,
                760293024620937,
                2027152777219493,
                666858468148475,
                1539184379870952,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1916168475367211,
                3167426246226591,
                883217071712574,
                363427871374304,
                1976029821251593,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                678039535434506,
                570587290189340,
                1605302676614120,
                2147762562875701,
                1706063797091704,
            ]),
            xy2d: FieldElement51::from_limbs([
                1439489648586438,
                2194580753290951,
                832380563557396,
                561521973970522,
                584497280718389,
            ]),
        },
    ]),
    LookupTable([
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2439789269177838,
                681223515948274,
                1933493571072456,
                1872921007304880,
                2739962177820919,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1413466089534451,
                410844090765630,
                1397263346404072,
                408227143123410,
                1594561803147811,
            ]),
            xy2d: FieldElement51::from_limbs([
                2102170800973153,
                719462588665004,
                1479649438510153,
                1097529543970028,
                1302363283777685,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3193865531532443,
                3321113493038208,
                2007341951411050,
                2322773230131539,
                1419433790163705,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1146565545556377,
                1661971299445212,
                406681704748893,
                564452436406089,
                1109109865829139,
            ]),
            xy2d: FieldElement51::from_limbs([
                2214421081775077,
                1165671861210569,
                1890453018796184,
                3556249878661,
                442116172656317,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3005630360306059,
                1666955059895018,
                1530775289309243,
                3371786842789394,
                2164156153857579,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                615171919212796,
                1523849404854568,
                854560460547503,
                2067097370290715,
                1765325848586042,
            ]),
            xy2d: FieldElement51::from_limbs([
                1094538949313667,
                1796592198908825,
                870221004284388,
                2025558921863561,
                1699010892802384,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1951351290725195,
                1916457206844795,
                2449824998123274,
                1909076887557594,
                1938542290318919,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1014323197538413,
                869150639940606,
                1756009942696599,
                1334952557375672,
                1544945379082874,
            ]),
            xy2d: FieldElement51::from_limbs([
                764055910920305,
                1603590757375439,
                146805246592357,
                1843313433854297,
                954279890114939,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                80113526615731,
                764536758732259,
                3306939158785481,
                2721052465444637,
                2869697326116762,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                74497112547268,
                740094153192149,
                1745254631717581,
                727713886503130,
                1283034364416928,
            ]),
            xy2d: FieldElement51::from_limbs([
                525892105991110,
                1723776830270342,
                1476444848991936,
                573789489857760,
                133864092632978,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2794411533877810,
                1986812262899320,
                1162535242465837,
                2733298779828712,
                2796400347268869,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                64123227344372,
                1239927720647794,
                1360722983445904,
                222610813654661,
                62429487187991,
            ]),
            xy2d: FieldElement51::from_limbs([
                1793193323953132,
                91096687857833,
                70945970938921,
                2158587638946380,
                1537042406482111,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1895854577604590,
                3646695522634664,
                1728548428495943,
                3392664713925397,
                2815445147288308,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                141358280486863,
                91435889572504,
                1087208572552643,
                1829599652522921,
                1193307020643647,
            ]),
            xy2d: FieldElement51::from_limbs([
                1611230858525381,
                950720175540785,
                499589887488610,
                2001656988495019,
                88977313255908,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3440880315164906,
                2184348804772596,
                3292618539427567,
                2018318290311833,
                1712060030915354,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                873966876953756,
                1090638350350440,
                1708559325189137,
                672344594801910,
                1320437969700239,
            ]),
            xy2d: FieldElement51::from_limbs([
                1508590048271766,
                1131769479776094,
                101550868699323,
                428297785557897,
                561791648661744,
            ]),
        },
    ]),
    LookupTable([
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3008217384184691,
                2489682092917849,
                2136263418594015,
                1701968045454886,
                2955512998822720,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1781187809325462,
                1697624151492346,
                1381393690939988,
                175194132284669,
                1483054666415238,
            ]),
            xy2d: FieldElement51::from_limbs([
                2175517777364616,
                708781536456029,
                955668231122942,
                1967557500069555,
                2021208005604118,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3366935780292116,
                2476017186636029,
                915967306279221,
                593866251291540,
                2813546907893254,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1443163092879439,
                391875531646162,
                2180847134654632,
                464538543018753,
                1594098196837178,
            ]),
            xy2d: FieldElement51::from_limbs([
                850858855888869,
                319436476624586,
                327807784938441,
                740785849558761,
                17128415486016,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2132756334090048,
                2788047633840893,
                2300706964962114,
                2860273011285942,
                3513489358708031,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1525176236978354,
                974205476721062,
                293436255662638,
                148269621098039,
                137961998433963,
            ]),
            xy2d: FieldElement51::from_limbs([
                1121075518299410,
                2071745529082111,
                1265567917414828,
                1648196578317805,
                496232102750820,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2374121042985030,
                3274721891178932,
                2001275453369483,
                2017441881607947,
                3245005694463250,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                654925550560074,
                1168810995576858,
                575655959430926,
                905758704861388,
                496774564663534,
            ]),
            xy2d: FieldElement51::from_limbs([
                1954109525779738,
                2117022646152485,
                338102630417180,
                1194140505732026,
                107881734943492,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1714785840001267,
                4288299832366837,
                1876380234251965,
                2056717182974196,
                1645855254384642,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                106431476499341,
                62482972120563,
                1513446655109411,
                807258751769522,
                538491469114,
            ]),
            xy2d: FieldElement51::from_limbs([
                2002850762893643,
                1243624520538135,
                1486040410574605,
                2184752338181213,
                378495998083531,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                922510868424903,
                1089502620807680,
                402544072617374,
                1131446598479839,
                1290278588136533,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1867998812076769,
                715425053580701,
                39968586461416,
                2173068014586163,
                653822651801304,
            ]),
            xy2d: FieldElement51::from_limbs([
                162892278589453,
                182585796682149,
                75093073137630,
                497037941226502,
                133871727117371,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                4166396390264918,
                1608999621851577,
                1987629837704609,
                1519655314857977,
                1819193753409464,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1949315551096831,
                1069003344994464,
                1939165033499916,
                1548227205730856,
                1933767655861407,
            ]),
            xy2d: FieldElement51::from_limbs([
                1730519386931635,
                1393284965610134,
                1597143735726030,
                416032382447158,
                1429665248828629,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                360275475604546,
                2799635544748326,
                2467160717872776,
                2848446553564254,
                2584509464110332,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                47602113726801,
                1522314509708010,
                437706261372925,
                814035330438027,
                335930650933545,
            ]),
            xy2d: FieldElement51::from_limbs([
                1291597595523886,
                1058020588994081,
                402837842324045,
                1363323695882781,
                2105763393033193,
            ]),
        },
    ]),
    LookupTable([
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2361321796251793,
                3967057562270386,
                1112231216891515,
                2046641005101484,
                2386048970842261,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2156991030936798,
                2227544497153325,
                1869050094431622,
                754875860479115,
                1754242344267058,
            ]),
            xy2d: FieldElement51::from_limbs([
                1846089562873800,
                98894784984326,
                1412430299204844,
                171351226625762,
                1100604760929008,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2335972195815721,
                2751510784385293,
                425749630620777,
                1762872794206857,
                2864642415813208,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                868309334532756,
                1703010512741873,
                1952690008738057,
                4325269926064,
                2071083554962116,
            ]),
            xy2d: FieldElement51::from_limbs([
                523094549451158,
                401938899487815,
                1407690589076010,
                2022387426254453,
                158660516411257,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                612867287630009,
                2700012425789062,
                2823428891104443,
                1466796750919375,
                1728478129663858,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1723848973783452,
                2208822520534681,
                1718748322776940,
                1974268454121942,
                1194212502258141,
            ]),
            xy2d: FieldElement51::from_limbs([
                1254114807944608,
                977770684047110,
                2010756238954993,
                1783628927194099,
                1525962994408256,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2484263871921055,
                1948628555342433,
                1835348780427694,
                1031609499437291,
                2316271920603621,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                767338676040683,
                754089548318405,
                1523192045639075,
                435746025122062,
                512692508440385,
            ]),
            xy2d: FieldElement51::from_limbs([
                1255955808701983,
                1700487367990941,
                1166401238800299,
                1175121994891534,
                1190934801395380,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2600943821853521,
                1337012557669161,
                1475912332999108,
                3573418268585706,
                2299411105589567,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                877519947135419,
                2172838026132651,
                272304391224129,
                1655143327559984,
                886229406429814,
            ]),
            xy2d: FieldElement51::from_limbs([
                375806028254706,
                214463229793940,
                572906353144089,
                572168269875638,
                697556386112979,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1168827102357825,
                823864273033637,
                4323338565789945,
                788062026895923,
                2851378154428610,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1948116082078088,
                2054898304487796,
                2204939184983900,
                210526805152138,
                786593586607626,
            ]),
            xy2d: FieldElement51::from_limbs([
                1915320147894736,
                156481169009469,
                655050471180417,
                592917090415421,
                2165897438660879,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1726336468579724,
                1119932070398949,
                1929199510967666,
                2285718602008207,
                1836837863503149,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                829996854845988,
                217061778005138,
                1686565909803640,
                1346948817219846,
                1723823550730181,
            ]),
            xy2d: FieldElement51::from_limbs([
                384301494966394,
                687038900403062,
                2211195391021739,
                254684538421383,
                1245698430589680,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1247567493562669,
                4229981908141095,
                2435671288478202,
                806570235643434,
                2540261331753164,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1449077384734201,
                38285445457996,
                2136537659177832,
                2146493000841573,
                725161151123125,
            ]),
            xy2d: FieldElement51::from_limbs([
                1201928866368855,
                800415690605445,
                1703146756828343,
                997278587541744,
                1858284414104014,
            ]),
        },
    ]),
    LookupTable([
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2608268623334125,
                3034173730618399,
                1718002439402869,
                3644022065904502,
                663171266061950,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                759628738230460,
                1012693474275852,
                353780233086498,
                246080061387552,
                2030378857679162,
            ]),
            xy2d: FieldElement51::from_limbs([
                2040672435071076,
                888593182036908,
                1298443657189359,
                1804780278521327,
                354070726137060,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1894938527423184,
                3715012855162525,
                2726210319182898,
                2499094776718546,
                877975941029127,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                207937160991127,
                12966911039119,
                820997788283092,
                1010440472205286,
                1701372890140810,
            ]),
            xy2d: FieldElement51::from_limbs([
                218882774543183,
                533427444716285,
                1233243976733245,
                435054256891319,
                1509568989549904,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                4140638349397055,
                3303977572025869,
                3465353617009382,
                2420981822812579,
                2715174081801119,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                299137589460312,
                1594371588983567,
                868058494039073,
                257771590636681,
                1805012993142921,
            ]),
            xy2d: FieldElement51::from_limbs([
                1806842755664364,
                2098896946025095,
                1356630998422878,
                1458279806348064,
                347755825962072,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1402334161391744,
                3811883484731547,
                1008585416617746,
                1147797150908892,
                1420416683642459,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                665506704253369,
                273770475169863,
                799236974202630,
                848328990077558,
                1811448782807931,
            ]),
            xy2d: FieldElement51::from_limbs([
                1468412523962641,
                771866649897997,
                1931766110147832,
                799561180078482,
                524837559150077,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2223212657821831,
                2882216061048914,
                2144451165500327,
                3068710944633039,
                3276150872095279,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1266603897524861,
                156378408858100,
                1275649024228779,
                447738405888420,
                253186462063095,
            ]),
            xy2d: FieldElement51::from_limbs([
                2022215964509735,
                136144366993649,
                1800716593296582,
                1193970603800203,
                871675847064218,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1862751661970309,
                851596246739884,
                1519315554814041,
                3794598280232697,
                3669775149586767,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1228168094547481,
                334133883362894,
                587567568420081,
                433612590281181,
                603390400373205,
            ]),
            xy2d: FieldElement51::from_limbs([
                121893973206505,
                1843345804916664,
                1703118377384911,
                497810164760654,
                101150811654673,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2710146069631716,
                2542709749304591,
                1452768413850678,
                2802722688939463,
                1537286854336537,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                584322311184395,
                380661238802118,
                114839394528060,
                655082270500073,
                2111856026034852,
            ]),
            xy2d: FieldElement51::from_limbs([
                996965581008991,
                2148998626477022,
                1012273164934654,
                1073876063914522,
                1688031788934939,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3175286832534829,
                2085106799623354,
                2779882615305384,
                1606206360876187,
                2987706905397772,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1697697887804317,
                1335343703828273,
                831288615207040,
                949416685250051,
                288760277392022,
            ]),
            xy2d: FieldElement51::from_limbs([
                1419122478109648,
                1325574567803701,
                602393874111094,
                2107893372601700,
                1314159682671307,
            ]),
        },
    ]),
    LookupTable([
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2201150872731785,
                2180241023425241,
                2349463270108411,
                1633405770247823,
                3100744856129234,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1173339555550611,
                818605084277583,
                47521504364289,
                924108720564965,
                735423405754506,
            ]),
            xy2d: FieldElement51::from_limbs([
                830104860549448,
                1886653193241086,
                1600929509383773,
                1475051275443631,
                286679780900937,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3828911108518224,
                3282698983453994,
                2396700729978777,
                4216472406664814,
                2820189914640497,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                278388655910247,
                487143369099838,
                927762205508727,
                181017540174210,
                1616886700741287,
            ]),
            xy2d: FieldElement51::from_limbs([
                1191033906638969,
                940823957346562,
                1606870843663445,
                861684761499847,
                658674867251089,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1875032594195527,
                1427106132796197,
                2976536204647406,
                3153660325729987,
                2887068310954007,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                622869792298357,
                1903919278950367,
                1922588621661629,
                1520574711600434,
                1087100760174640,
            ]),
            xy2d: FieldElement51::from_limbs([
                25465949416618,
                1693639527318811,
                1526153382657203,
                125943137857169,
                145276964043999,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2466539671654587,
                920212862967914,
                4191701364657517,
                3463662605460468,
                2336897329405367,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2006245852772938,
                734762734836159,
                254642929763427,
                1406213292755966,
                239303749517686,
            ]),
            xy2d: FieldElement51::from_limbs([
                1619678837192149,
                1919424032779215,
                1357391272956794,
                1525634040073113,
                1310226789796241,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3292563523447371,
                1704449869235351,
                2857062884141577,
                1998838089036354,
                1312142911487502,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1996723311435669,
                1844342766567060,
                985455700466044,
                1165924681400960,
                311508689870129,
            ]),
            xy2d: FieldElement51::from_limbs([
                43173156290518,
                2202883069785309,
                1137787467085917,
                1733636061944606,
                1394992037553852,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                670078326344559,
                2807454838744604,
                2723759199967685,
                2141455487356408,
                849015953823125,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2197214573372804,
                794254097241315,
                1030190060513737,
                267632515541902,
                2040478049202624,
            ]),
            xy2d: FieldElement51::from_limbs([
                1812516004670529,
                1609256702920783,
                1706897079364493,
                258549904773295,
                996051247540686,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1540374301420565,
                1764656898914615,
                1810104162020396,
                3175608592848336,
                2916189887881826,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1323460699404750,
                1262690757880991,
                871777133477900,
                1060078894988977,
                1712236889662886,
            ]),
            xy2d: FieldElement51::from_limbs([
                1696163952057966,
                1391710137550823,
                608793846867416,
                1034391509472039,
                1780770894075012,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1367603834210822,
                4383788460268472,
                890353773628143,
                1908908219165595,
                2522636708938139,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                597536315471731,
                40375058742586,
                1942256403956049,
                1185484645495932,
                312666282024145,
            ]),
            xy2d: FieldElement51::from_limbs([
                1919411405316294,
                1234508526402192,
                1066863051997083,
                1008444703737597,
                1348810787701552,
            ]),
        },
    ]),
    LookupTable([
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2102881477513865,
                3822074379630609,
                1573617900503707,
                2270462449417831,
                2232324307922097,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1853931367696942,
                8107973870707,
                350214504129299,
                775206934582587,
                1752317649166792,
            ]),
            xy2d: FieldElement51::from_limbs([
                1417148368003523,
                721357181628282,
                505725498207811,
                373232277872983,
                261634707184480,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2186733281493248,
                2250694917008620,
                1014829812957440,
                2731797975137637,
                2335366007561721,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1268116367301224,
                560157088142809,
                802626839600444,
                2210189936605713,
                1129993785579988,
            ]),
            xy2d: FieldElement51::from_limbs([
                615183387352312,
                917611676109240,
                878893615973325,
                978940963313282,
                938686890583575,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                522024729211672,
                3296859129001056,
                1892245413707789,
                1907891107684253,
                2059998109500714,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1799679152208884,
                912132775900387,
                25967768040979,
                432130448590461,
                274568990261996,
            ]),
            xy2d: FieldElement51::from_limbs([
                98698809797682,
                2144627600856209,
                1907959298569602,
                811491302610148,
                1262481774981493,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1791451399743152,
                1713538728337276,
                2370149810942738,
                1882306388849953,
                158235232210248,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1217809823321928,
                2173947284933160,
                1986927836272325,
                1388114931125539,
                12686131160169,
            ]),
            xy2d: FieldElement51::from_limbs([
                1650875518872272,
                1136263858253897,
                1732115601395988,
                734312880662190,
                1252904681142109,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2624786269799113,
                2777230729143418,
                2116279931702134,
                2753222527273063,
                1907002872974924,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                803147181835288,
                868941437997146,
                316299302989663,
                943495589630550,
                571224287904572,
            ]),
            xy2d: FieldElement51::from_limbs([
                227742695588364,
                1776969298667369,
                628602552821802,
                457210915378118,
                2041906378111140,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                815000523470260,
                3164885502413555,
                3303859931956420,
                1345536665214222,
                541623413135555,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1580216071604333,
                1877997504342444,
                857147161260913,
                703522726778478,
                2182763974211603,
            ]),
            xy2d: FieldElement51::from_limbs([
                1870080310923419,
                71988220958492,
                1783225432016732,
                615915287105016,
                1035570475990230,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2982787564515398,
                857613889540279,
                1083813157271766,
                1002817255970169,
                1719228484436074,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                377616581647602,
                1581980403078513,
                804044118130621,
                2034382823044191,
                643844048472185,
            ]),
            xy2d: FieldElement51::from_limbs([
                176957326463017,
                1573744060478586,
                528642225008045,
                1816109618372371,
                1515140189765006,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1888911448245718,
                3638910709296328,
                4176303607751676,
                1731539523700948,
                2230378382645454,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                443392177002051,
                233793396845137,
                2199506622312416,
                1011858706515937,
                974676837063129,
            ]),
            xy2d: FieldElement51::from_limbs([
                1846351103143623,
                1949984838808427,
                671247021915253,
                1946756846184401,
                1929296930380217,
            ]),
        },
    ]),
    LookupTable([
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                849646212451983,
                1410198775302919,
                2325567699868943,
                1641663456615811,
                3014056086137659,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                692017667358279,
                723305578826727,
                1638042139863265,
                748219305990306,
                334589200523901,
            ]),
            xy2d: FieldElement51::from_limbs([
                22893968530686,
                2235758574399251,
                1661465835630252,
                925707319443452,
                1203475116966621,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3053098849470395,
                3985092410411378,
                1664508947088595,
                2719548934677170,
                3899298398220870,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                903105258014366,
                427141894933047,
                561187017169777,
                1884330244401954,
                1914145708422219,
            ]),
            xy2d: FieldElement51::from_limbs([
                1344191060517578,
                1960935031767890,
                1518838929955259,
                1781502350597190,
                1564784025565682,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2925523165433334,
                1979969272514922,
                3427087126180756,
                1187589090978665,
                1881897672213940,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1917185587363432,
                1098342571752737,
                5935801044414,
                2000527662351839,
                1538640296181569,
            ]),
            xy2d: FieldElement51::from_limbs([
                2495540013192,
                678856913479236,
                224998292422872,
                219635787698590,
                1972465269000940,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                271413961212179,
                3604851875156899,
                2596511104968730,
                2014925838520661,
                2006221033113941,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                194583029968109,
                514316781467765,
                829677956235672,
                1676415686873082,
                810104584395840,
            ]),
            xy2d: FieldElement51::from_limbs([
                1980510813313589,
                1948645276483975,
                152063780665900,
                129968026417582,
                256984195613935,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1860190562533083,
                1936576191345085,
                2712900106391212,
                1811043097042829,
                3209286562992083,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                796664815624365,
                1543160838872951,
                1500897791837765,
                1667315977988401,
                599303877030711,
            ]),
            xy2d: FieldElement51::from_limbs([
                1151480509533204,
                2136010406720455,
                738796060240027,
                319298003765044,
                1150614464349587,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1731069268103131,
                2987442261301335,
                1364750481334267,
                2669032653668119,
                3178908082812908,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1017222050227968,
                1987716148359,
                2234319589635701,
                621282683093392,
                2132553131763026,
            ]),
            xy2d: FieldElement51::from_limbs([
                1567828528453324,
                1017807205202360,
                565295260895298,
                829541698429100,
                307243822276582,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                249079270936229,
                1501514259790706,
                3199709537890096,
                944551802437486,
                2804458577667728,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2089966982947227,
                1854140343916181,
                2151980759220007,
                2139781292261749,
                158070445864917,
            ]),
            xy2d: FieldElement51::from_limbs([
                1338766321464554,
                1906702607371284,
                1519569445519894,
                115384726262267,
                1393058953390992,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3616421371950629,
                3764188048593604,
                1926731583198685,
                2041482526432505,
                3172200936019022,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1884844597333588,
                601480070269079,
                620203503079537,
                1079527400117915,
                1202076693132015,
            ]),
            xy2d: FieldElement51::from_limbs([
                840922919763324,
                727955812569642,
                1303406629750194,
                522898432152867,
                294161410441865,
            ]),
        },
    ]),
    LookupTable([
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2605560604520539,
                1598361541848742,
                3374705511887547,
                4174333403844152,
                2670907514351827,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                359856369838236,
                180914355488683,
                861726472646627,
                218807937262986,
                575626773232501,
            ]),
            xy2d: FieldElement51::from_limbs([
                755467689082474,
                909202735047934,
                730078068932500,
                936309075711518,
                2007798262842972,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1609384177904054,
                2614544999293875,
                1335318541768200,
                3052765584121496,
                2799677792952659,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                984339177776787,
                815727786505884,
                1645154585713747,
                1659074964378553,
                1686601651984156,
            ]),
            xy2d: FieldElement51::from_limbs([
                1697863093781930,
                599794399429786,
                1104556219769607,
                830560774794755,
                12812858601017,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1168737550514982,
                897832437380552,
                463140296333799,
                2554364413707795,
                2008360505135500,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1856930662813910,
                678090852002597,
                1920179140755167,
                1259527833759868,
                55540971895511,
            ]),
            xy2d: FieldElement51::from_limbs([
                1158643631044921,
                476554103621892,
                178447851439725,
                1305025542653569,
                103433927680625,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2176793111709008,
                3828525530035639,
                2009350167273522,
                2012390194631546,
                2125297410909580,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                825403285195098,
                2144208587560784,
                1925552004644643,
                1915177840006985,
                1015952128947864,
            ]),
            xy2d: FieldElement51::from_limbs([
                1807108316634472,
                1534392066433717,
                347342975407218,
                1153820745616376,
                7375003497471,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3234860815484973,
                2683011703586488,
                2201903782961092,
                3069193724749589,
                2214616493042166,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                228567918409756,
                865093958780220,
                358083886450556,
                159617889659320,
                1360637926292598,
            ]),
            xy2d: FieldElement51::from_limbs([
                234147501399755,
                2229469128637390,
                2175289352258889,
                1397401514549353,
                1885288963089922,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3363562226636810,
                2504649386192636,
                3300514047508588,
                2397910909286693,
                1237505378776769,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1113790697840279,
                1051167139966244,
                1045930658550944,
                2011366241542643,
                1686166824620755,
            ]),
            xy2d: FieldElement51::from_limbs([
                1054097349305049,
                1872495070333352,
                182121071220717,
                1064378906787311,
                100273572924182,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3558210666856834,
                1627717417672446,
                2302783034773665,
                1109249951172249,
                3122001602766640,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                104233794644221,
                1548919791188248,
                2224541913267306,
                2054909377116478,
                1043803389015153,
            ]),
            xy2d: FieldElement51::from_limbs([
                216762189468802,
                707284285441622,
                190678557969733,
                973969342604308,
                1403009538434867,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3530824104723725,
                2596576648903557,
                2525521909702446,
                4086000250496689,
                634517197663803,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                343805853118335,
                1302216857414201,
                566872543223541,
                2051138939539004,
                321428858384280,
            ]),
            xy2d: FieldElement51::from_limbs([
                470067171324852,
                1618629234173951,
                2000092177515639,
                7307679772789,
                1117521120249968,
            ]),
        },
    ]),
    LookupTable([
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2529951391976704,
                1810282338562946,
                1771599529530998,
                3635459223356879,
                2937173228157088,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                577009397403102,
                1791440261786291,
                2177643735971638,
                174546149911960,
                1412505077782326,
            ]),
            xy2d: FieldElement51::from_limbs([
                893719721537457,
                1201282458018197,
                1522349501711173,
                58011597740583,
                1130406465887139,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                412607348255434,
                1280455764199780,
                2233277987330768,
                2265979894086913,
                2583384512102412,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                262483770854550,
                990511055108216,
                526885552771698,
                571664396646158,
                354086190278723,
            ]),
            xy2d: FieldElement51::from_limbs([
                1820352417585487,
                24495617171480,
                1547899057533253,
                10041836186225,
                480457105094042,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2023310314989233,
                2889705151211129,
                2106474638900686,
                2809620524769320,
                1687858215057825,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1144168702609745,
                604444390410187,
                1544541121756138,
                1925315550126027,
                626401428894002,
            ]),
            xy2d: FieldElement51::from_limbs([
                1922168257351784,
                2018674099908659,
                1776454117494445,
                956539191509034,
                36031129147635,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2796444352433270,
                1039872944430373,
                3128550222815858,
                2962457525011798,
                3468752501170219,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                58242421545916,
                2035812695641843,
                2118491866122923,
                1191684463816273,
                46921517454099,
            ]),
            xy2d: FieldElement51::from_limbs([
                272268252444639,
                1374166457774292,
                2230115177009552,
                1053149803909880,
                1354288411641016,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1857910905368338,
                1754729879288912,
                3137745277795125,
                1516096106802165,
                1602902393369811,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1193437069800958,
                901107149704790,
                999672920611411,
                477584824802207,
                364239578697845,
            ]),
            xy2d: FieldElement51::from_limbs([
                886299989548838,
                1538292895758047,
                1590564179491896,
                1944527126709657,
                837344427345298,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3006358179063534,
                1712186480903617,
                3955456640022779,
                3002110732175033,
                2770795853936147,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1309847803895382,
                1462151862813074,
                211370866671570,
                1544595152703681,
                1027691798954090,
            ]),
            xy2d: FieldElement51::from_limbs([
                803217563745370,
                1884799722343599,
                1357706345069218,
                2244955901722095,
                730869460037413,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2941099284981214,
                1831210565161070,
                3626987155270686,
                3358084791231418,
                1893781834054268,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                696351368613042,
                1494385251239250,
                738037133616932,
                636385507851544,
                927483222611406,
            ]),
            xy2d: FieldElement51::from_limbs([
                1949114198209333,
                1104419699537997,
                783495707664463,
                1747473107602770,
                2002634765788641,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1607325776830197,
                2782683755100581,
                1451089452727894,
                3833490970768671,
                496100432831153,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1068900648804224,
                2006891997072550,
                1134049269345549,
                1638760646180091,
                2055396084625778,
            ]),
            xy2d: FieldElement51::from_limbs([
                2222475519314561,
                1870703901472013,
                1884051508440561,
                1344072275216753,
                1318025677799069,
            ]),
        },
    ]),
    LookupTable([
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                155711679280637,
                681100400509288,
                389811735211209,
                2135723811340709,
                2660533024889373,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                7813206966729,
                194444201427550,
                2071405409526507,
                1065605076176312,
                1645486789731291,
            ]),
            xy2d: FieldElement51::from_limbs([
                16625790644959,
                1647648827778410,
                1579910185572704,
                436452271048548,
                121070048451050,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3289062842237779,
                2820185594063076,
                2549752917829677,
                3810384325616458,
                2238221839292470,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                190565267697443,
                672855706028058,
                338796554369226,
                337687268493904,
                853246848691734,
            ]),
            xy2d: FieldElement51::from_limbs([
                1763863028400139,
                766498079432444,
                1321118624818005,
                69494294452268,
                858786744165651,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3543856582248253,
                1456632109855637,
                3352431060735432,
                1386133165675320,
                3484698163879000,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                366253102478259,
                525676242508811,
                1449610995265438,
                1183300845322183,
                185960306491545,
            ]),
            xy2d: FieldElement51::from_limbs([
                28315355815982,
                460422265558930,
                1799675876678724,
                1969256312504498,
                1051823843138725,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2408714813047231,
                3857948219405196,
                1665208410108429,
                2569443092377519,
                1383783705665319,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                54684536365732,
                2210010038536222,
                1194984798155308,
                535239027773705,
                1516355079301361,
            ]),
            xy2d: FieldElement51::from_limbs([
                1484387703771650,
                198537510937949,
                2186282186359116,
                617687444857508,
                647477376402122,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2147715541830533,
                2751832352131065,
                2898179830570073,
                2604027669016369,
                1488268620408051,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                159386186465542,
                1877626593362941,
                618737197060512,
                1026674284330807,
                1158121760792685,
            ]),
            xy2d: FieldElement51::from_limbs([
                1744544377739822,
                1964054180355661,
                1685781755873170,
                2169740670377448,
                1286112621104591,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2333777063470241,
                3919742931398333,
                3920783633320113,
                1605016835177614,
                1353960708075544,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1602253788689063,
                439542044889886,
                2220348297664483,
                657877410752869,
                157451572512238,
            ]),
            xy2d: FieldElement51::from_limbs([
                1029287186166717,
                65860128430192,
                525298368814832,
                1491902500801986,
                1461064796385400,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2660016802414475,
                2121095722306988,
                913562102267595,
                1879708920318308,
                2492861262121979,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1185483484383269,
                1356339572588553,
                584932367316448,
                102132779946470,
                1792922621116791,
            ]),
            xy2d: FieldElement51::from_limbs([
                1966196870701923,
                2230044620318636,
                1425982460745905,
                261167817826569,
                46517743394330,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2358877405280588,
                3136759755857592,
                2279106683482647,
                2224911448949389,
                3216151871930471,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1730194207717538,
                431790042319772,
                1831515233279467,
                1372080552768581,
                1074513929381760,
            ]),
            xy2d: FieldElement51::from_limbs([
                1450880638731607,
                1019861580989005,
                1229729455116861,
                1174945729836143,
                826083146840706,
            ]),
        },
    ]),
    LookupTable([
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1899935429242705,
                1602068751520477,
                940583196550370,
                2334230882739107,
                1540863155745695,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2136688454840028,
                2099509000964294,
                1690800495246475,
                1217643678575476,
                828720645084218,
            ]),
            xy2d: FieldElement51::from_limbs([
                765548025667841,
                462473984016099,
                998061409979798,
                546353034089527,
                2212508972466858,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2298375097456408,
                3144370785258318,
                1281983193144089,
                1491520128287375,
                75847005908304,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1801436127943107,
                1734436817907890,
                1268728090345068,
                167003097070711,
                2233597765834956,
            ]),
            xy2d: FieldElement51::from_limbs([
                1997562060465113,
                1048700225534011,
                7615603985628,
                1855310849546841,
                2242557647635213,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1161017320376250,
                2744424393854291,
                2169815802355236,
                3228296595417790,
                1770879511019628,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1357044908364776,
                729130645262438,
                1762469072918979,
                1365633616878458,
                181282906404941,
            ]),
            xy2d: FieldElement51::from_limbs([
                1080413443139865,
                1155205815510486,
                1848782073549786,
                622566975152580,
                124965574467971,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1184526762066993,
                247622751762817,
                2943928830891604,
                3071818503097743,
                2188697339828084,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2020536369003019,
                202261491735136,
                1053169669150884,
                2056531979272544,
                778165514694311,
            ]),
            xy2d: FieldElement51::from_limbs([
                237404399610207,
                1308324858405118,
                1229680749538400,
                720131409105291,
                1958958863624906,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2767383321724075,
                2269456792542436,
                1717918437373988,
                1568052070792483,
                2298775616809171,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                281527309158085,
                36970532401524,
                866906920877543,
                2222282602952734,
                1289598729589882,
            ]),
            xy2d: FieldElement51::from_limbs([
                1278207464902042,
                494742455008756,
                1262082121427081,
                1577236621659884,
                1888786707293291,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                353042527954210,
                1830056151907359,
                1111731275799225,
                2426760769524072,
                404312815582674,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2064251142068628,
                1666421603389706,
                1419271365315441,
                468767774902855,
                191535130366583,
            ]),
            xy2d: FieldElement51::from_limbs([
                1716987058588002,
                1859366439773457,
                1767194234188234,
                64476199777924,
                1117233614485261,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3236091949205521,
                2386938060636506,
                2220652137473166,
                1722843421165029,
                2442282371698157,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                298845952651262,
                1166086588952562,
                1179896526238434,
                1347812759398693,
                1412945390096208,
            ]),
            xy2d: FieldElement51::from_limbs([
                1143239552672925,
                906436640714209,
                2177000572812152,
                2075299936108548,
                325186347798433,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2972824668060020,
                2936287674948563,
                3625238557779406,
                2193186935276994,
                1387043709851261,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                418098668140962,
                715065997721283,
                1471916138376055,
                2168570337288357,
                937812682637044,
            ]),
            xy2d: FieldElement51::from_limbs([
                1043584187226485,
                2143395746619356,
                2209558562919611,
                482427979307092,
                847556718384018,
            ]),
        },
    ]),
    LookupTable([
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1248731221520740,
                1465200936117687,
                2792603306395388,
                2304778448366139,
                2513234303861356,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1057329623869501,
                620334067429122,
                461700859268034,
                2012481616501857,
                297268569108938,
            ]),
            xy2d: FieldElement51::from_limbs([
                1055352180870759,
                1553151421852298,
                1510903185371259,
                1470458349428097,
                1226259419062731,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3744788603986897,
                3042126439258578,
                3441906842094992,
                3641194565844440,
                3872208010289441,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                47000654413729,
                1004754424173864,
                1868044813557703,
                173236934059409,
                588771199737015,
            ]),
            xy2d: FieldElement51::from_limbs([
                30498470091663,
                1082245510489825,
                576771653181956,
                806509986132686,
                1317634017056939,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2672107869436803,
                3745154677001249,
                2417006535213335,
                4136645508605033,
                2065456951573058,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1115636332012334,
                1854340990964155,
                83792697369514,
                1972177451994021,
                457455116057587,
            ]),
            xy2d: FieldElement51::from_limbs([
                1698968457310898,
                1435137169051090,
                1083661677032510,
                938363267483709,
                340103887207182,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1995325341336555,
                911500251774648,
                2415810569088940,
                855378419194761,
                3825401211214090,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                241719380661528,
                310028521317150,
                1215881323380194,
                1408214976493624,
                2141142156467363,
            ]),
            xy2d: FieldElement51::from_limbs([
                1315157046163473,
                727368447885818,
                1363466668108618,
                1668921439990361,
                1398483384337907,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2326829491984875,
                3267188020145720,
                1849729037055211,
                4191614430138232,
                2696204044080201,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2053597130993710,
                2024431685856332,
                2233550957004860,
                2012407275509545,
                872546993104440,
            ]),
            xy2d: FieldElement51::from_limbs([
                1217269667678610,
                599909351968693,
                1390077048548598,
                1471879360694802,
                739586172317596,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3970118453066023,
                1560510726633957,
                3156262694845170,
                1418028351780051,
                2346204163137185,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2132502667405250,
                214379346175414,
                1502748313768060,
                1960071701057800,
                1353971822643138,
            ]),
            xy2d: FieldElement51::from_limbs([
                319394212043702,
                2127459436033571,
                717646691535162,
                663366796076914,
                318459064945314,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2657789238608841,
                1960452633787082,
                2919148848086913,
                3744474074452359,
                1451061489880786,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                947085906234007,
                323284730494107,
                1485778563977200,
                728576821512394,
                901584347702286,
            ]),
            xy2d: FieldElement51::from_limbs([
                1575783124125742,
                2126210792434375,
                1569430791264065,
                1402582372904727,
                1891780248341114,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3090232019245924,
                4249503325136911,
                3270591693593114,
                1662001808174330,
                2330127946643001,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                739152638255629,
                2074935399403557,
                505483666745895,
                1611883356514088,
                628654635394878,
            ]),
            xy2d: FieldElement51::from_limbs([
                1822054032121349,
                643057948186973,
                7306757352712,
                577249257962099,
                284735863382083,
            ]),
        },
    ]),
    LookupTable([
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3618358370049178,
                1448606567552085,
                3730680834630016,
                2417602993041145,
                1115718458123497,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                204146226972102,
                1630511199034723,
                2215235214174763,
                174665910283542,
                956127674017216,
            ]),
            xy2d: FieldElement51::from_limbs([
                1562934578796716,
                1070893489712745,
                11324610642270,
                958989751581897,
                2172552325473805,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1770564423056008,
                2987323445349813,
                1326060113795288,
                1509650369341127,
                2317692235267932,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                623682558650637,
                1337866509471512,
                990313350206649,
                1314236615762469,
                1164772974270275,
            ]),
            xy2d: FieldElement51::from_limbs([
                223256821462517,
                723690150104139,
                1000261663630601,
                933280913953265,
                254872671543046,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1969087237026022,
                2876595539132372,
                1335555107635968,
                2069986355593023,
                3963899963027150,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1236103475266979,
                1837885883267218,
                1026072585230455,
                1025865513954973,
                1801964901432134,
            ]),
            xy2d: FieldElement51::from_limbs([
                1115241013365517,
                1712251818829143,
                2148864332502771,
                2096001471438138,
                2235017246626125,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3551068012286861,
                2047148477845620,
                2165648650132450,
                1612539282026145,
                2765997725314138,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                118352772338543,
                1067608711804704,
                1434796676193498,
                1683240170548391,
                230866769907437,
            ]),
            xy2d: FieldElement51::from_limbs([
                1850689576796636,
                1601590730430274,
                1139674615958142,
                1954384401440257,
                76039205311,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1723387471374172,
                3249101280723658,
                2785727448808904,
                2272728458379212,
                1756575222802512,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2146711623855116,
                503278928021499,
                625853062251406,
                1109121378393107,
                1033853809911861,
            ]),
            xy2d: FieldElement51::from_limbs([
                571005965509422,
                2005213373292546,
                1016697270349626,
                56607856974274,
                914438579435146,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1346698876211176,
                2076651707527589,
                3336561384795453,
                2517134292513653,
                1068954492309670,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1769967932677654,
                1695893319756416,
                1151863389675920,
                1781042784397689,
                400287774418285,
            ]),
            xy2d: FieldElement51::from_limbs([
                1851867764003121,
                403841933237558,
                820549523771987,
                761292590207581,
                1743735048551143,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                410915148140008,
                2107072311871739,
                3256167275561751,
                2351484709082008,
                1180818713503223,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                285945406881439,
                648174397347453,
                1098403762631981,
                1366547441102991,
                1505876883139217,
            ]),
            xy2d: FieldElement51::from_limbs([
                672095903120153,
                1675918957959872,
                636236529315028,
                1569297300327696,
                2164144194785875,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1902708175321798,
                3287143344600686,
                1178560808893262,
                2552895497743394,
                1280977479761117,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1615357281742403,
                404257611616381,
                2160201349780978,
                1160947379188955,
                1578038619549541,
            ]),
            xy2d: FieldElement51::from_limbs([
                2013087639791217,
                822734930507457,
                1785668418619014,
                1668650702946164,
                389450875221715,
            ]),
        },
    ]),
    LookupTable([
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2705718263383616,
                2358206633614248,
                2072540975937134,
                308588860670238,
                1304394580755385,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1295082798350326,
                2091844511495996,
                1851348972587817,
                3375039684596,
                789440738712837,
            ]),
            xy2d: FieldElement51::from_limbs([
                2083069137186154,
                848523102004566,
                993982213589257,
                1405313299916317,
                1532824818698468,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3747761112537659,
                1397203457344778,
                4026750030752190,
                2391102557240943,
                2318403398028034,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1782411379088302,
                1096724939964781,
                27593390721418,
                542241850291353,
                1540337798439873,
            ]),
            xy2d: FieldElement51::from_limbs([
                693543956581437,
                171507720360750,
                1557908942697227,
                1074697073443438,
                1104093109037196,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                345288228393400,
                3351443383432420,
                2386681722088990,
                1740551994106739,
                2500011992985018,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                231429562203065,
                1526290236421172,
                2021375064026423,
                1520954495658041,
                806337791525116,
            ]),
            xy2d: FieldElement51::from_limbs([
                1079623667189886,
                872403650198613,
                766894200588288,
                2163700860774109,
                2023464507911816,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                854645372543796,
                1936406001954827,
                2403260476226501,
                3077125552956802,
                1554306377287555,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1497138821904622,
                1044820250515590,
                1742593886423484,
                1237204112746837,
                849047450816987,
            ]),
            xy2d: FieldElement51::from_limbs([
                667962773375330,
                1897271816877105,
                1399712621683474,
                1143302161683099,
                2081798441209593,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2378947665252234,
                1936114012888109,
                1704424366552046,
                3108474694401560,
                2968403435020606,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1072409664800960,
                2146937497077528,
                1508780108920651,
                935767602384853,
                1112800433544068,
            ]),
            xy2d: FieldElement51::from_limbs([
                333549023751292,
                280219272863308,
                2104176666454852,
                1036466864875785,
                536135186520207,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2625466093568366,
                2398257055215356,
                2555916080813104,
                2667888562832962,
                3510376944868638,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1186115062588401,
                2251609796968486,
                1098944457878953,
                1153112761201374,
                1791625503417267,
            ]),
            xy2d: FieldElement51::from_limbs([
                1870078460219737,
                2129630962183380,
                852283639691142,
                292865602592851,
                401904317342226,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1361070124828016,
                815664541425524,
                3278598711049919,
                1951790935390646,
                2807674705520038,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1546301003424277,
                459094500062839,
                1097668518375311,
                1780297770129643,
                720763293687608,
            ]),
            xy2d: FieldElement51::from_limbs([
                1212405311403990,
                1536693382542438,
                61028431067459,
                1863929423417129,
                1223219538638038,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1294303766540260,
                3435357279640341,
                3134071170918340,
                2315654383110622,
                2213283684565086,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                339050984211414,
                601386726509773,
                413735232134068,
                966191255137228,
                1839475899458159,
            ]),
            xy2d: FieldElement51::from_limbs([
                235605972169408,
                2174055643032978,
                1538335001838863,
                1281866796917192,
                1815940222628465,
            ]),
        },
    ]),
    LookupTable([
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1632352921721536,
                1833328609514701,
                2092779091951987,
                4175756015558474,
                2210068022482918,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                35271216625062,
                1712350667021807,
                983664255668860,
                98571260373038,
                1232645608559836,
            ]),
            xy2d: FieldElement51::from_limbs([
                1998172393429622,
                1798947921427073,
                784387737563581,
                1589352214827263,
                1589861734168180,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1733739258725305,
                2283515530744786,
                2453769758904107,
                3243892858242237,
                1194308773174555,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                846415389605137,
                746163495539180,
                829658752826080,
                592067705956946,
                957242537821393,
            ]),
            xy2d: FieldElement51::from_limbs([
                1758148849754419,
                619249044817679,
                168089007997045,
                1371497636330523,
                1867101418880350,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2578433797894864,
                2513559319756263,
                1700682323676192,
                1577907266349064,
                3469447477068264,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1714182387328607,
                1477856482074168,
                574895689942184,
                2159118410227270,
                1555532449716575,
            ]),
            xy2d: FieldElement51::from_limbs([
                853828206885131,
                998498946036955,
                1835887550391235,
                207627336608048,
                258363815956050,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2392941288336925,
                3488528558590503,
                2894901233585134,
                1646615130509172,
                1208239602291765,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1501663228068911,
                1354879465566912,
                1444432675498247,
                897812463852601,
                855062598754348,
            ]),
            xy2d: FieldElement51::from_limbs([
                714380763546606,
                1032824444965790,
                1774073483745338,
                1063840874947367,
                1738680636537158,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1640635546696233,
                2884968766877360,
                2212651044092395,
                2282390772269100,
                2620315074574625,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1171650314802029,
                1567085444565577,
                1453660792008405,
                757914533009261,
                1619511342778196,
            ]),
            xy2d: FieldElement51::from_limbs([
                420958967093237,
                971103481109486,
                2169549185607107,
                1301191633558497,
                1661514101014240,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3158923465503550,
                1332556122804145,
                4075855067109735,
                3619414031128206,
                1982558335973171,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1121533090144639,
                1021251337022187,
                110469995947421,
                1511059774758394,
                2110035908131662,
            ]),
            xy2d: FieldElement51::from_limbs([
                303213233384524,
                2061932261128138,
                352862124777736,
                40828818670255,
                249879468482660,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                856559257852200,
                2760317478634258,
                3629993581580163,
                3975258940632376,
                1962275756614520,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1445691340537320,
                40614383122127,
                402104303144865,
                485134269878232,
                1659439323587426,
            ]),
            xy2d: FieldElement51::from_limbs([
                20057458979482,
                1183363722525800,
                2140003847237215,
                2053873950687614,
                2112017736174909,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2228654250927986,
                3735391177100515,
                1368661293910955,
                3328311098862539,
                526650682059607,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                709481497028540,
                531682216165724,
                316963769431931,
                1814315888453765,
                258560242424104,
            ]),
            xy2d: FieldElement51::from_limbs([
                1053447823660455,
                1955135194248683,
                1010900954918985,
                1182614026976701,
                1240051576966610,
            ]),
        },
    ]),
    LookupTable([
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1957943897155478,
                1788667368028035,
                2389492723714354,
                2252839333292309,
                3078204576998275,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1848942433095597,
                1582009882530495,
                1849292741020143,
                1068498323302788,
                2001402229799484,
            ]),
            xy2d: FieldElement51::from_limbs([
                1528282417624269,
                2142492439828191,
                2179662545816034,
                362568973150328,
                1591374675250271,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2411826493119617,
                2484141002903963,
                2149181472355544,
                598041771119831,
                2435658815595421,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2013278155187349,
                662660471354454,
                793981225706267,
                411706605985744,
                804490933124791,
            ]),
            xy2d: FieldElement51::from_limbs([
                2051892037280204,
                488391251096321,
                2230187337030708,
                930221970662692,
                679002758255210,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1530723630438670,
                875873929577927,
                2593359947955236,
                2701702933216000,
                1055551308214178,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1461835919309432,
                1955256480136428,
                180866187813063,
                1551979252664528,
                557743861963950,
            ]),
            xy2d: FieldElement51::from_limbs([
                359179641731115,
                1324915145732949,
                902828372691474,
                294254275669987,
                1887036027752957,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                4295071423139571,
                2038225437857463,
                1317528426475850,
                1398989128982787,
                2027639881006861,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2072902725256516,
                312132452743412,
                309930885642209,
                996244312618453,
                1590501300352303,
            ]),
            xy2d: FieldElement51::from_limbs([
                1397254305160710,
                695734355138021,
                2233992044438756,
                1776180593969996,
                1085588199351115,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2692366865016258,
                2506694600041928,
                2745669038615469,
                1556322069683365,
                3819256354004466,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1950722461391320,
                1907845598854797,
                1822757481635527,
                2121567704750244,
                73811931471221,
            ]),
            xy2d: FieldElement51::from_limbs([
                387139307395758,
                2058036430315676,
                1220915649965325,
                1794832055328951,
                1230009312169328,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1765973779329498,
                2911143873132225,
                2271621715291913,
                3553728154996461,
                3368065817761132,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1127572801181483,
                1224743760571696,
                1276219889847274,
                1529738721702581,
                1589819666871853,
            ]),
            xy2d: FieldElement51::from_limbs([
                2181229378964934,
                2190885205260020,
                1511536077659137,
                1246504208580490,
                668883326494241,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2689666469258543,
                2920826224880015,
                2333696811665585,
                523874406393177,
                2496851874620484,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1975438052228868,
                1071801519999806,
                594652299224319,
                1877697652668809,
                1489635366987285,
            ]),
            xy2d: FieldElement51::from_limbs([
                958592545673770,
                233048016518599,
                851568750216589,
                567703851596087,
                1740300006094761,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2014540178270324,
                192672779514432,
                2465676996326778,
                2194819933853410,
                1716422829364835,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1540769606609725,
                2148289943846077,
                1597804156127445,
                1230603716683868,
                815423458809453,
            ]),
            xy2d: FieldElement51::from_limbs([
                1738560251245018,
                1779576754536888,
                1783765347671392,
                1880170990446751,
                1088225159617541,
            ]),
        },
    ]),
    LookupTable([
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2911103727614740,
                1956447718227572,
                1830568515922666,
                3092868863429656,
                1669607124206367,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1143465490433355,
                1532194726196059,
                1093276745494697,
                481041706116088,
                2121405433561163,
            ]),
            xy2d: FieldElement51::from_limbs([
                1686424298744462,
                1451806974487153,
                266296068846582,
                1834686947542675,
                1720762336132256,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3141016840074207,
                3295090436969907,
                3107924901237156,
                1669272323124635,
                1603340330827879,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1206396181488998,
                333158148435054,
                1402633492821422,
                1120091191722026,
                1945474114550509,
            ]),
            xy2d: FieldElement51::from_limbs([
                766720088232571,
                1512222781191002,
                1189719893490790,
                2091302129467914,
                2141418006894941,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2671463460991841,
                1998875112167986,
                3678399683938955,
                3406728169064757,
                2738338345823434,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                938160078005954,
                1421776319053174,
                1941643234741774,
                180002183320818,
                1414380336750546,
            ]),
            xy2d: FieldElement51::from_limbs([
                398001940109652,
                1577721237663248,
                1012748649830402,
                1540516006905144,
                1011684812884559,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1653276489969611,
                2257881638852872,
                1921777941170835,
                1604139841794531,
                3113010867325889,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                996661541407379,
                1455877387952927,
                744312806857277,
                139213896196746,
                1000282908547789,
            ]),
            xy2d: FieldElement51::from_limbs([
                1450817495603008,
                1476865707053229,
                1030490562252053,
                620966950353376,
                1744760161539058,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2811528223687828,
                2288856475326432,
                2038622963352005,
                1637244893271723,
                3278365165924196,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                962165956135846,
                1116599660248791,
                182090178006815,
                1455605467021751,
                196053588803284,
            ]),
            xy2d: FieldElement51::from_limbs([
                796863823080135,
                1897365583584155,
                420466939481601,
                2165972651724672,
                932177357788289,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                877047233620613,
                1375632631944375,
                2895573425567369,
                2911822552533124,
                2271153746017078,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2216943882299338,
                394841323190322,
                2222656898319671,
                558186553950529,
                1077236877025190,
            ]),
            xy2d: FieldElement51::from_limbs([
                801118384953213,
                1914330175515892,
                574541023311511,
                1471123787903705,
                1526158900256288,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3201417702772463,
                2207116611267330,
                3164719852826535,
                2752958352884036,
                2314162374456719,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1474518386765335,
                1760793622169197,
                1157399790472736,
                1622864308058898,
                165428294422792,
            ]),
            xy2d: FieldElement51::from_limbs([
                1961673048027128,
                102619413083113,
                1051982726768458,
                1603657989805485,
                1941613251499678,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1401939116319247,
                2587106153588320,
                2323846009771033,
                862423201496005,
                3102318568216632,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1234706593321979,
                1083343891215917,
                898273974314935,
                1640859118399498,
                157578398571149,
            ]),
            xy2d: FieldElement51::from_limbs([
                1143483057726416,
                1992614991758919,
                674268662140796,
                1773370048077526,
                674318359920189,
            ]),
        },
    ]),
    LookupTable([
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1835401379538542,
                173900035308392,
                818247630716732,
                4013900225838034,
                1021506399448290,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1506632088156630,
                2127481795522179,
                513812919490255,
                140643715928370,
                442476620300318,
            ]),
            xy2d: FieldElement51::from_limbs([
                2056683376856736,
                219094741662735,
                2193541883188309,
                1841182310235800,
                556477468664293,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3566819241596075,
                1049075855992602,
                4318372866671791,
                2518704280870781,
                2040482348591519,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                94096246544434,
                922482381166992,
                24517828745563,
                2139430508542503,
                2097139044231004,
            ]),
            xy2d: FieldElement51::from_limbs([
                537697207950515,
                1399352016347350,
                1563663552106345,
                2148749520888918,
                549922092988516,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1747985413252415,
                680511052635695,
                1809559829982725,
                2846074064615302,
                2453472984431229,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                323583936109569,
                1973572998577657,
                1192219029966558,
                79354804385273,
                1374043025560347,
            ]),
            xy2d: FieldElement51::from_limbs([
                213277331329947,
                416202017849623,
                1950535221091783,
                1313441578103244,
                2171386783823658,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2440888617915079,
                993969372859109,
                3147669935222235,
                3799101348983503,
                1477373024911349,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1620578418245010,
                541035331188469,
                2235785724453865,
                2154865809088198,
                1974627268751826,
            ]),
            xy2d: FieldElement51::from_limbs([
                1346805451740245,
                1350981335690626,
                942744349501813,
                2155094562545502,
                1012483751693409,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2107080134091762,
                1132567062788208,
                1824935377687210,
                769194804343737,
                1857941799971888,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1074666112436467,
                249279386739593,
                1174337926625354,
                1559013532006480,
                1472287775519121,
            ]),
            xy2d: FieldElement51::from_limbs([
                1872620123779532,
                1892932666768992,
                1921559078394978,
                1270573311796160,
                1438913646755037,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3089190001333428,
                3264053113908846,
                989780015893986,
                1351393287739814,
                2580427560230798,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1028328827183114,
                1711043289969857,
                1350832470374933,
                1923164689604327,
                1495656368846911,
            ]),
            xy2d: FieldElement51::from_limbs([
                1900828492104143,
                430212361082163,
                687437570852799,
                832514536673512,
                1685641495940794,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3094432661621646,
                605670026766215,
                290836444839585,
                2415010588577604,
                2213815011799644,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1176336383453996,
                1725477294339771,
                12700622672454,
                678015708818208,
                162724078519879,
            ]),
            xy2d: FieldElement51::from_limbs([
                1448049969043497,
                1789411762943521,
                385587766217753,
                90201620913498,
                832999441066823,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2767886146978542,
                2240508292484615,
                3603469341851756,
                3475055379001735,
                3002035638112385,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1263624896582495,
                1102602401673328,
                526302183714372,
                2152015839128799,
                1483839308490010,
            ]),
            xy2d: FieldElement51::from_limbs([
                442991718646863,
                1599275157036458,
                1925389027579192,
                899514691371390,
                350263251085160,
            ]),
        },
    ]),
    LookupTable([
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1689713572022124,
                2845654372939621,
                3229894858477217,
                1985127338729498,
                3927868934032873,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1557207018622683,
                340631692799603,
                1477725909476187,
                614735951619419,
                2033237123746766,
            ]),
            xy2d: FieldElement51::from_limbs([
                968764929340557,
                1225534776710944,
                662967304013036,
                1155521416178595,
                791142883466590,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1487081286167458,
                3244839255500182,
                1792378982844639,
                2950452258685122,
                2153908693179753,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1123181311102823,
                685575944875442,
                507605465509927,
                1412590462117473,
                568017325228626,
            ]),
            xy2d: FieldElement51::from_limbs([
                560258797465417,
                2193971151466401,
                1824086900849026,
                579056363542056,
                1690063960036441,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1918407319222397,
                2605567366745211,
                1930426334528098,
                1564816146005724,
                4113142195393344,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2131325168777276,
                1176636658428908,
                1756922641512981,
                1390243617176012,
                1966325177038383,
            ]),
            xy2d: FieldElement51::from_limbs([
                2063958120364491,
                2140267332393533,
                699896251574968,
                273268351312140,
                375580724713232,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2024297515263178,
                2668759143407935,
                3330814048702549,
                2423412039258430,
                1031677520051052,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2033900009388450,
                1744902869870788,
                2190580087917640,
                1949474984254121,
                231049754293748,
            ]),
            xy2d: FieldElement51::from_limbs([
                343868674606581,
                550155864008088,
                1450580864229630,
                481603765195050,
                896972360018042,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2151139328380127,
                2566545695770176,
                2311556639460451,
                1676664391494650,
                2048348075599360,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1528930066340597,
                1605003907059576,
                1055061081337675,
                1458319101947665,
                1234195845213142,
            ]),
            xy2d: FieldElement51::from_limbs([
                830430507734812,
                1780282976102377,
                1425386760709037,
                362399353095425,
                2168861579799910,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3407562046415562,
                980662895504005,
                2053766700883521,
                2742766027762854,
                2762205690726604,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1683750316716132,
                652278688286128,
                1221798761193539,
                1897360681476669,
                319658166027343,
            ]),
            xy2d: FieldElement51::from_limbs([
                618808732869972,
                72755186759744,
                2060379135624181,
                1730731526741822,
                48862757828238,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3714971784278753,
                3394840525452699,
                614590986558882,
                1409210575145591,
                1882816996436803,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2230133264691131,
                563950955091024,
                2042915975426398,
                827314356293472,
                672028980152815,
            ]),
            xy2d: FieldElement51::from_limbs([
                264204366029760,
                1654686424479449,
                2185050199932931,
                2207056159091748,
                506015669043634,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1784446333136550,
                1973746527984364,
                334856327359575,
                3408569589569858,
                3275749938360725,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2065270940578383,
                31477096270353,
                306421879113491,
                181958643936686,
                1907105536686083,
            ]),
            xy2d: FieldElement51::from_limbs([
                1496516440779464,
                1748485652986458,
                872778352227340,
                818358834654919,
                97932669284220,
            ]),
        },
    ]),
    LookupTable([
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2723435829455580,
                2924255216478824,
                1804995246884102,
                1842309243470804,
                3753662318666930,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1013216974933691,
                538921919682598,
                1915776722521558,
                1742822441583877,
                1886550687916656,
            ]),
            xy2d: FieldElement51::from_limbs([
                2094270000643336,
                303971879192276,
                40801275554748,
                649448917027930,
                1818544418535447,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2241737709499146,
                549397817447461,
                838180519319392,
                1725686958520781,
                3957438894582995,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1216074541925116,
                50120933933509,
                1565829004133810,
                721728156134580,
                349206064666188,
            ]),
            xy2d: FieldElement51::from_limbs([
                948617110470858,
                346222547451945,
                1126511960599975,
                1759386906004538,
                493053284802266,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1454933046815146,
                3126495827951610,
                1467170975468587,
                1432316382418897,
                2111710746366763,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2105387117364450,
                1996463405126433,
                1303008614294500,
                851908115948209,
                1353742049788635,
            ]),
            xy2d: FieldElement51::from_limbs([
                750300956351719,
                1487736556065813,
                15158817002104,
                1511998221598392,
                971739901354129,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1874648163531674,
                2124487685930551,
                1810030029384882,
                918400043048335,
                2838148440985898,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1235084464747900,
                1166111146432082,
                1745394857881591,
                1405516473883040,
                4463504151617,
            ]),
            xy2d: FieldElement51::from_limbs([
                1663810156463827,
                327797390285791,
                1341846161759410,
                1964121122800605,
                1747470312055380,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                660005247548214,
                2071860029952887,
                3610548013635355,
                911703252219106,
                3266179736709079,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2206641276178231,
                1690587809721504,
                1600173622825126,
                2156096097634421,
                1106822408548216,
            ]),
            xy2d: FieldElement51::from_limbs([
                1344788193552206,
                1949552134239140,
                1735915881729557,
                675891104100469,
                1834220014427292,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1920949492387945,
                2410685102072778,
                2322108077349280,
                2877838278583064,
                3719881539786256,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                622221042073383,
                1210146474039168,
                1742246422343683,
                1403839361379025,
                417189490895736,
            ]),
            xy2d: FieldElement51::from_limbs([
                22727256592983,
                168471543384997,
                1324340989803650,
                1839310709638189,
                504999476432775,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3565040332441556,
                1721896294296941,
                2304063388272514,
                2065069734239231,
                3056710287109878,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1337466662091884,
                1287645354669772,
                2018019646776184,
                652181229374245,
                898011753211715,
            ]),
            xy2d: FieldElement51::from_limbs([
                1969792547910734,
                779969968247557,
                2011350094423418,
                1823964252907487,
                1058949448296945,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2459143550747250,
                1118176942430252,
                3010694408233412,
                806764629546265,
                1157700123092949,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1273565321399022,
                1638509681964574,
                759235866488935,
                666015124346707,
                897983460943405,
            ]),
            xy2d: FieldElement51::from_limbs([
                1717263794012298,
                1059601762860786,
                1837819172257618,
                1054130665797229,
                680893204263559,
            ]),
        },
    ]),
    LookupTable([
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2237039662793603,
                2249022333361206,
                2058613546633703,
                2401253908530527,
                2215176649164581,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                79472182719605,
                1851130257050174,
                1825744808933107,
                821667333481068,
                781795293511946,
            ]),
            xy2d: FieldElement51::from_limbs([
                755822026485370,
                152464789723500,
                1178207602290608,
                410307889503239,
                156581253571278,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3669985309815545,
                2736319981413860,
                3898537095128197,
                3653287498355512,
                1349185550126960,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1495380034400429,
                325049476417173,
                46346894893933,
                1553408840354856,
                828980101835683,
            ]),
            xy2d: FieldElement51::from_limbs([
                1280337889310282,
                2070832742866672,
                1640940617225222,
                2098284908289951,
                450929509534434,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2659503167684029,
                2378371955168899,
                2537839641198868,
                1999255076709337,
                2030511179441770,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1254958221100483,
                1153235960999843,
                942907704968834,
                637105404087392,
                1149293270147267,
            ]),
            xy2d: FieldElement51::from_limbs([
                894249020470196,
                400291701616810,
                406878712230981,
                1599128793487393,
                1145868722604026,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3749755063888563,
                2361916158338507,
                1128535642171975,
                1900106496009660,
                2381592531146157,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                452487513298665,
                1352120549024569,
                1173495883910956,
                1999111705922009,
                367328130454226,
            ]),
            xy2d: FieldElement51::from_limbs([
                1717539401269642,
                1475188995688487,
                891921989653942,
                836824441505699,
                1885988485608364,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3493583935107776,
                2439136865632830,
                3370281625921440,
                2680547565621609,
                2282158712612572,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2022432361201842,
                1088816090685051,
                1977843398539868,
                1854834215890724,
                564238862029357,
            ]),
            xy2d: FieldElement51::from_limbs([
                938868489100585,
                1100285072929025,
                1017806255688848,
                1957262154788833,
                152787950560442,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3119119231364171,
                2872271776627789,
                2477832016990963,
                2593801257642876,
                1761675818237335,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1295072362439987,
                931227904689414,
                1355731432641687,
                922235735834035,
                892227229410209,
            ]),
            xy2d: FieldElement51::from_limbs([
                1680989767906154,
                535362787031440,
                2136691276706570,
                1942228485381244,
                1267350086882274,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2617818047455756,
                2684460443440843,
                2378209521329782,
                1973842949591661,
                2897427157127624,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                535509430575217,
                546885533737322,
                1524675609547799,
                2138095752851703,
                1260738089896827,
            ]),
            xy2d: FieldElement51::from_limbs([
                1159906385590467,
                2198530004321610,
                714559485023225,
                81880727882151,
                1484020820037082,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1377485731340769,
                2046328105512000,
                1802058637158797,
                2313945950453421,
                1356993908853900,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2013612215646735,
                1830770575920375,
                536135310219832,
                609272325580394,
                270684344495013,
            ]),
            xy2d: FieldElement51::from_limbs([
                1237542585982777,
                2228682050256790,
                1385281931622824,
                593183794882890,
                493654978552689,
            ]),
        },
    ]),
    LookupTable([
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2299141301692989,
                1891414891220256,
                983894663308928,
                2427961581972066,
                3378060928864955,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1694030170963455,
                502038567066200,
                1691160065225467,
                949628319562187,
                275110186693066,
            ]),
            xy2d: FieldElement51::from_limbs([
                1124515748676336,
                1661673816593408,
                1499640319059718,
                1584929449166988,
                558148594103306,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1784525599998356,
                1619698033617383,
                2097300287550715,
                2510065271789004,
                1905684794832757,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1288941072872766,
                931787902039402,
                190731008859042,
                2006859954667190,
                1005931482221702,
            ]),
            xy2d: FieldElement51::from_limbs([
                1465551264822703,
                152905080555927,
                680334307368453,
                173227184634745,
                666407097159852,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2111017076203943,
                3630560299479595,
                1248583954016455,
                3604089008549670,
                1895180776543895,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                171348223915638,
                662766099800389,
                462338943760497,
                466917763340314,
                656911292869115,
            ]),
            xy2d: FieldElement51::from_limbs([
                488623681976577,
                866497561541722,
                1708105560937768,
                1673781214218839,
                1506146329818807,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2412225278142205,
                950394373239688,
                2682296937026182,
                711676555398831,
                320964687779005,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                988979367990485,
                1359729327576302,
                1301834257246029,
                294141160829308,
                29348272277475,
            ]),
            xy2d: FieldElement51::from_limbs([
                1434382743317910,
                100082049942065,
                221102347892623,
                186982837860588,
                1305765053501834,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2205916462268190,
                2751663643476068,
                961960554686615,
                2409862576442233,
                1841471168298304,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1191737341426592,
                1847042034978363,
                1382213545049056,
                1039952395710448,
                788812858896859,
            ]),
            xy2d: FieldElement51::from_limbs([
                1346965964571152,
                1291881610839830,
                2142916164336056,
                786821641205979,
                1571709146321039,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                787164375951248,
                2454669019058437,
                3608390234717387,
                1431233331032509,
                786341368775957,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                492448143532951,
                304105152670757,
                1761767168301056,
                233782684697790,
                1981295323106089,
            ]),
            xy2d: FieldElement51::from_limbs([
                665807507761866,
                1343384868355425,
                895831046139653,
                439338948736892,
                1986828765695105,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3007896024559801,
                1721699973539148,
                2510565115413133,
                1390588532210644,
                1212530909934781,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                852891097972275,
                1816988871354562,
                1543772755726524,
                1174710635522444,
                202129090724628,
            ]),
            xy2d: FieldElement51::from_limbs([
                1205281565824323,
                22430498399418,
                992947814485516,
                1392458699738672,
                688441466734558,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3302427242100220,
                1955849529137134,
                2171162376368357,
                2343545681983462,
                447733118757825,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1287181461435438,
                622722465530711,
                880952150571872,
                741035693459198,
                311565274989772,
            ]),
            xy2d: FieldElement51::from_limbs([
                1003649078149734,
                545233927396469,
                1849786171789880,
                1318943684880434,
                280345687170552,
            ]),
        },
    ]),
]);

/// Odd multiples of the basepoint `[B, 3B, 5B, 7B, 9B, 11B, 13B, 15B, ..., 127B]`.
#[cfg(feature = "precomputed-tables")]
#[allow(dead_code)]
pub(crate) const AFFINE_ODD_MULTIPLES_OF_BASEPOINT: NafLookupTable8<AffineNielsPoint> =
    NafLookupTable8([
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3540182452943730,
                2497478415033846,
                2521227595762870,
                1462984067271729,
                2389212253076811,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                62697248952638,
                204681361388450,
                631292143396476,
                338455783676468,
                1213667448819585,
            ]),
            xy2d: FieldElement51::from_limbs([
                301289933810280,
                1259582250014073,
                1422107436869536,
                796239922652654,
                1953934009299142,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1601611775252272,
                1720807796594148,
                1132070835939856,
                3512254832574799,
                2147779492816910,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                316559037616741,
                2177824224946892,
                1459442586438991,
                1461528397712656,
                751590696113597,
            ]),
            xy2d: FieldElement51::from_limbs([
                1850748884277385,
                1200145853858453,
                1068094770532492,
                672251375690438,
                1586055907191707,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                769950342298400,
                2384754244604994,
                3095885746880802,
                3225892188161580,
                2977876099231263,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                425251763115706,
                608463272472562,
                442562545713235,
                837766094556764,
                374555092627893,
            ]),
            xy2d: FieldElement51::from_limbs([
                1086255230780037,
                274979815921559,
                1960002765731872,
                929474102396301,
                1190409889297339,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2916800678241215,
                2065379846933858,
                2622030924071124,
                2602788184473875,
                1233371373142984,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2019367628972465,
                676711900706637,
                110710997811333,
                1108646842542025,
                517791959672113,
            ]),
            xy2d: FieldElement51::from_limbs([
                965130719900578,
                247011430587952,
                526356006571389,
                91986625355052,
                2157223321444601,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1802695059464988,
                1664899123557221,
                2845359304426105,
                2160434469266658,
                3179370264440279,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1725674970513508,
                1933645953859181,
                1542344539275782,
                1767788773573747,
                1297447965928905,
            ]),
            xy2d: FieldElement51::from_limbs([
                1381809363726107,
                1430341051343062,
                2061843536018959,
                1551778050872521,
                2036394857967624,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                4222693909998302,
                2779866139518454,
                1619374932191226,
                2207306624415883,
                1169170329061080,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2070390218572616,
                1458919061857835,
                624171843017421,
                1055332792707765,
                433987520732508,
            ]),
            xy2d: FieldElement51::from_limbs([
                893653801273833,
                1168026499324677,
                1242553501121234,
                1306366254304474,
                1086752658510815,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2465253816303469,
                3191571337672685,
                1159882208056013,
                2569188183312765,
                621213314200686,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1971678598905747,
                338026507889165,
                762398079972271,
                655096486107477,
                42299032696322,
            ]),
            xy2d: FieldElement51::from_limbs([
                177130678690680,
                1754759263300204,
                1864311296286618,
                1180675631479880,
                1292726903152791,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1913163449625248,
                2712579013977241,
                2193883288642313,
                1008900146920800,
                1721983679009502,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1070401523076875,
                1272492007800961,
                1910153608563310,
                2075579521696771,
                1191169788841221,
            ]),
            xy2d: FieldElement51::from_limbs([
                692896803108118,
                500174642072499,
                2068223309439677,
                1162190621851337,
                1426986007309901,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1819621230288238,
                2735700366193240,
                1755134670739586,
                3080648199451191,
                4172807995775876,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                992069868904071,
                799011518185730,
                1777586403832768,
                1134820506145684,
                1999461475558530,
            ]),
            xy2d: FieldElement51::from_limbs([
                425204543703124,
                2040469794090382,
                1651690622153809,
                1500530168597569,
                1253908377065966,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2105824306960939,
                1387520302709358,
                3633176580451016,
                2211816663841753,
                1629085891776489,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1485201376284999,
                1022406647424656,
                504181009209019,
                962621520820995,
                590876713147230,
            ]),
            xy2d: FieldElement51::from_limbs([
                265873406365287,
                1192742653492898,
                88553098803050,
                525037770869640,
                1266933811251234,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3552316659826612,
                1254279525791875,
                1609927932077699,
                3578654071679972,
                3750681296069893,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                37186803519861,
                1404297334376301,
                578519728836650,
                1740727951192592,
                2095534282477028,
            ]),
            xy2d: FieldElement51::from_limbs([
                833234263154399,
                2023862470013762,
                1854137933982069,
                853924318090959,
                1589812702805850,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3679150557957763,
                1319179453661745,
                497496853611112,
                2665464286942351,
                1208137952365560,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1654513078530905,
                907489875842908,
                126098711296368,
                1726320004173677,
                28269495058173,
            ]),
            xy2d: FieldElement51::from_limbs([
                114436686957443,
                532739313025996,
                115428841215897,
                2191499400074366,
                370280402676434,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1111146849833253,
                2016430049079759,
                1860522747477948,
                3537164738290194,
                4137142824844184,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                429069864577128,
                975327637149449,
                237881983565075,
                1654761232378630,
                2122527599091807,
            ]),
            xy2d: FieldElement51::from_limbs([
                2093793463548278,
                754827233241879,
                1420389751719629,
                1829952782588138,
                2011865756773717,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                676293365438898,
                2850296017886344,
                1205350322490195,
                2763699392265669,
                2133931188538142,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                48340340349120,
                1299261101494832,
                1137329686775218,
                1534848106674340,
                1351662218216799,
            ]),
            xy2d: FieldElement51::from_limbs([
                1904520614137939,
                1590301001714014,
                215781420985270,
                2043534301034629,
                1970888949300424,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2365217962409710,
                2061307169694064,
                1887478590157603,
                2169639621284316,
                2373810867477200,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1020052624656948,
                1260412094216707,
                366721640607121,
                585331442306596,
                345876457758061,
            ]),
            xy2d: FieldElement51::from_limbs([
                975390299880933,
                1066555195234642,
                12651997758352,
                1184252205433068,
                1058378155074223,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1431537716602643,
                2024827957433813,
                3746434518400495,
                1087794891033550,
                2156817571680455,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                929288033346881,
                255179964546973,
                711057989588035,
                208899572612840,
                185348357387383,
            ]),
            xy2d: FieldElement51::from_limbs([
                823689746424808,
                47266130989546,
                209403309368097,
                1100966895202707,
                710792075292719,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2311213117823762,
                3296668540922318,
                2004276520649823,
                1861500579441125,
                3148029033359833,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1563693677475261,
                1843782073741194,
                1950700654453170,
                911540858113949,
                2085151496302359,
            ]),
            xy2d: FieldElement51::from_limbs([
                1427880892005482,
                106216431121745,
                42608394782284,
                1217295886989793,
                1514235272796882,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3544335535746750,
                2367994491347456,
                2567261456502612,
                1854058085060971,
                2263545563461076,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                787426011300053,
                2105981035769060,
                1130476291127206,
                1748659348100075,
                53470983013756,
            ]),
            xy2d: FieldElement51::from_limbs([
                553548273865386,
                5927805718390,
                65184587381926,
                633576679686953,
                576048559439973,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                993787326657446,
                3868807161609258,
                1615796046728943,
                2514644292681953,
                2059021068660907,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                251010270518880,
                1681684095763484,
                1521949356387564,
                431593457045116,
                1855308922422910,
            ]),
            xy2d: FieldElement51::from_limbs([
                618490909691959,
                1257497595618257,
                202952467594088,
                35577762721238,
                1494883566841973,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1673474571932262,
                2409784519770613,
                2636095316260487,
                2761112584601925,
                3333713288149876,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1600640202645197,
                1019569075331823,
                1041916487915822,
                1680448171313267,
                2126903137527901,
            ]),
            xy2d: FieldElement51::from_limbs([
                894964745143659,
                106116880092678,
                1009869382959477,
                317866368542032,
                1986983122763912,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1765281781276487,
                2863247187455184,
                2589075472439062,
                1386435905543054,
                2182338478845320,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1144730936996693,
                2213315231278180,
                1489676672185125,
                665039429138074,
                1131283313040268,
            ]),
            xy2d: FieldElement51::from_limbs([
                2004734176670602,
                1738311085075235,
                418866995976618,
                1050782508034394,
                577747313404652,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2185209688340293,
                1309276076461009,
                2514740038571278,
                3994889904012999,
                3018098826231021,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1405936970888515,
                1754621155316654,
                1211862168554999,
                1813045702919083,
                997853418197172,
            ]),
            xy2d: FieldElement51::from_limbs([
                82037622045021,
                1646398333621944,
                613095452763466,
                1312329542583705,
                81014679202721,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2389287991277873,
                403851022333257,
                1597473361477193,
                2953351602509212,
                2135174663049062,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1826548187201150,
                302299893734126,
                1475477168615781,
                842617616347376,
                1438600873676130,
            ]),
            xy2d: FieldElement51::from_limbs([
                663049852468609,
                1649295727846569,
                1048009692742781,
                628866177992421,
                1914360327429204,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1795645928096646,
                306878154408959,
                2924901319092394,
                2801261341654799,
                1653782432983523,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2077597317438627,
                212642017882064,
                674844477518888,
                875487498687554,
                2060550250171182,
            ]),
            xy2d: FieldElement51::from_limbs([
                1420448018683809,
                1032663994771382,
                1341927003385267,
                1340360916546159,
                1988547473895228,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1082660122598844,
                2545055705583789,
                3888919679589007,
                1670283344995811,
                3403239134794618,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                90430593339788,
                1838338032241275,
                571293238480915,
                1639938867416883,
                257378872001111,
            ]),
            xy2d: FieldElement51::from_limbs([
                1528535658865034,
                1516636853043960,
                787000569996728,
                1464531394704506,
                1684822625133795,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                811329918113934,
                2783463529007378,
                1769095754634835,
                2970819621866866,
                881037178164325,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1784566501964517,
                433890943689325,
                1186055625589419,
                1496077405487512,
                1731807117886548,
            ]),
            xy2d: FieldElement51::from_limbs([
                424909811816304,
                1355993963741797,
                409606483251841,
                455665350637068,
                1617009023642808,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2478728492077816,
                2780289048655501,
                2328687177473769,
                4107341333582032,
                1316147724308250,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1617420574301156,
                1741273341070467,
                667135503486508,
                2100436564640123,
                1032223920000865,
            ]),
            xy2d: FieldElement51::from_limbs([
                1753947659404033,
                247279202390193,
                1819288880178945,
                737334285670249,
                1037873664856104,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1762568490530034,
                673742465299012,
                2054571050635888,
                2040165159255111,
                3040123733327257,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1627187989987422,
                1686331580821752,
                1309895873498183,
                719718719104086,
                300063199808722,
            ]),
            xy2d: FieldElement51::from_limbs([
                238176707016164,
                1440454788877048,
                203336037573144,
                1437789888677072,
                101522256664211,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1895216760098480,
                1934324337975022,
                3677350688973167,
                2536415965456176,
                714678003308640,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                508185358728815,
                1691320535341855,
                2168887448239256,
                1035124393070661,
                1936603999698584,
            ]),
            xy2d: FieldElement51::from_limbs([
                390562831571647,
                1390223890708972,
                1383183990676371,
                435998174196410,
                1882086414390730,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3747620842612921,
                2081794785291195,
                3284594056262745,
                2090090346797895,
                2581692978935809,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                244144781251265,
                1290834426417077,
                1888701171101942,
                1233922456644870,
                241117402207491,
            ]),
            xy2d: FieldElement51::from_limbs([
                1266169390045455,
                1148042013187970,
                878921907853942,
                1815738019658093,
                908920199341621,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2521768507305118,
                953557056811112,
                2015863732865770,
                1358382511861315,
                2835421647899992,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2239837206240498,
                330928973149665,
                422268062913642,
                1481280019493032,
                619879520439841,
            ]),
            xy2d: FieldElement51::from_limbs([
                1360166735366017,
                1770556573948510,
                1395061284191031,
                1814003148068126,
                522781147076884,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2611794802645686,
                707234844948070,
                1314059396506491,
                2919250341703934,
                2161831667832785,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                934831784182383,
                433734253968318,
                1660867106725771,
                1968393082772831,
                873946300968490,
            ]),
            xy2d: FieldElement51::from_limbs([
                26306827827554,
                430884999378685,
                1504310424376419,
                1761358720837522,
                542195685418530,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1762131062631725,
                3123952634417535,
                3619918390837537,
                2909990877347294,
                1411594230004385,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                538272372224622,
                1425714779586199,
                588313661410172,
                1497062084392578,
                1602174047128512,
            ]),
            xy2d: FieldElement51::from_limbs([
                907490361939255,
                1963620338391363,
                626927432296975,
                1250748516081414,
                959901171882527,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1335066153744413,
                2887804660779657,
                2653073855954038,
                2765226981667422,
                938831784476763,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                296699434737224,
                2047543711075683,
                2076451038937139,
                227783599906901,
                1602062110967627,
            ]),
            xy2d: FieldElement51::from_limbs([
                1574834773194203,
                1384279952062839,
                393652417255803,
                2166968242848859,
                1552890441390820,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1619646774410947,
                1576090644023562,
                3035228391320965,
                1735328519940543,
                2355324535937066,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1024074573633446,
                957088456885874,
                1690425531356997,
                2102187380180052,
                1082544623222033,
            ]),
            xy2d: FieldElement51::from_limbs([
                1871906170635853,
                1719383891167200,
                1584032250247862,
                823764804192117,
                2244048510084261,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                642147846489775,
                3334304977145699,
                305205716788147,
                2589176626729533,
                2224680511484174,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1734162377166545,
                260713621840346,
                157174591942595,
                952544272517991,
                222818702471733,
            ]),
            xy2d: FieldElement51::from_limbs([
                1213115494182947,
                286778704335711,
                2130189536016490,
                308349182281342,
                1217623948685491,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3360052266973635,
                1843486583624091,
                1561693837124349,
                1084041964025479,
                1866270922024009,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                460705465481210,
                1968151453817859,
                497005926994844,
                625618055866751,
                2176893440866887,
            ]),
            xy2d: FieldElement51::from_limbs([
                1655800250476757,
                2036588542300609,
                666447448675243,
                1615721995750683,
                1508669225186765,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2245948203759141,
                1058306669699396,
                1452898014240582,
                3961024141962768,
                1633235287338608,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                986647273684279,
                1507266907811370,
                1260572633649005,
                2071672342077446,
                695976026010857,
            ]),
            xy2d: FieldElement51::from_limbs([
                1312356620823495,
                1635278548098567,
                901946076841033,
                585120475533168,
                1240667113237384,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2313723935779695,
                1506054666773895,
                996040223525031,
                636592914999692,
                1497801917020297,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                292042016419794,
                1158932298133044,
                2062611870323738,
                1946058478962569,
                1749165808126286,
            ]),
            xy2d: FieldElement51::from_limbs([
                654683942212830,
                1526897351349087,
                2006818439922838,
                2194919327350361,
                1451960776874416,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3015041017808905,
                2951823141773809,
                2584865668253675,
                2508192032998563,
                2582137700042019,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1628123495344283,
                2072923641214546,
                1647225812023982,
                855655925244679,
                1758126430071140,
            ]),
            xy2d: FieldElement51::from_limbs([
                1615895096489599,
                275295258643784,
                937665541219916,
                1313496726746346,
                1186468946422626,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1603070202850694,
                2072127623773242,
                1692648737212158,
                2493373404187852,
                1248948672117105,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                11167836031898,
                596565174397990,
                2196351068723859,
                314744641791907,
                1102014997250781,
            ]),
            xy2d: FieldElement51::from_limbs([
                1409047922401191,
                69960384467966,
                688103515547600,
                1309746102488044,
                150292892873778,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1986083055103168,
                691715819340300,
                1361811659746933,
                3459052030333434,
                1063594696046061,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1201987338414749,
                2198784582460616,
                1203335513981498,
                489243077045066,
                2205278143582433,
            ]),
            xy2d: FieldElement51::from_limbs([
                2034744376624534,
                2077387101466387,
                148448542974969,
                1502697574577258,
                473186584705655,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                472016956315960,
                720786972252993,
                2840633661190043,
                3150798753357827,
                2816563335499153,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                253464247569755,
                168314237403057,
                511780806170295,
                1058862316549135,
                1646858476817137,
            ]),
            xy2d: FieldElement51::from_limbs([
                595092995922219,
                1491311840717691,
                291581784452778,
                1569186646367854,
                1031385061400544,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3483137021572755,
                1526955102024322,
                2778006642704458,
                457549634924205,
                1097420237736736,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1246991699537710,
                81367319519439,
                530844036072196,
                163656863755855,
                1950742455979290,
            ]),
            xy2d: FieldElement51::from_limbs([
                191532664076407,
                539378506082089,
                1021612562876554,
                1026603384732632,
                1773368780410653,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                4144620731387879,
                590179521333342,
                4034023318016108,
                2255745030335426,
                2699746851701250,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2206599697359952,
                553895797384417,
                181689161933786,
                1153123447919104,
                778568064152659,
            ]),
            xy2d: FieldElement51::from_limbs([
                1706307000059211,
                1885601289314487,
                889758608505788,
                550131729999853,
                1006862664714268,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3210197754285058,
                2048500453422630,
                3403309827888207,
                927154428508963,
                4199813798872019,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                992058915374933,
                476120535358775,
                1973648780784340,
                2025282643598818,
                2182318983793230,
            ]),
            xy2d: FieldElement51::from_limbs([
                1343440812005821,
                1316045839091795,
                1884951299078063,
                1765919609219175,
                2197567554627988,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3129247779382818,
                4415026969054274,
                1900265885969643,
                1528796215447059,
                2172730393748688,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1773355092297603,
                64654329538271,
                1332124041660957,
                748492100858001,
                895500006200535,
            ]),
            xy2d: FieldElement51::from_limbs([
                2000840647851980,
                546565968824914,
                420633283457524,
                195470736374507,
                1958689297569520,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                743138980705446,
                3411117504637167,
                2591389959690621,
                2380042066577202,
                3022267940115114,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                165947002229363,
                115186103724967,
                1068573292121517,
                1842565776920938,
                1969395681111987,
            ]),
            xy2d: FieldElement51::from_limbs([
                553322266190633,
                234265665613185,
                484544650202821,
                1238773526575826,
                2017991917953668,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2581954631514051,
                1245093644265357,
                3537016673825374,
                1834216551713857,
                923978372152807,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1855378315339552,
                890045579230758,
                1764718173975590,
                197904186055854,
                1718129022310327,
            ]),
            xy2d: FieldElement51::from_limbs([
                1278162928734862,
                1894118254109862,
                987503995465517,
                177406744098996,
                781538103127693,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1996603431230215,
                1191888797552937,
                1207440075928499,
                2765853449051137,
                2525314961343288,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                808903879370889,
                990820108751280,
                1084429472258867,
                1078562781312589,
                254514692695625,
            ]),
            xy2d: FieldElement51::from_limbs([
                615855140068469,
                586046731175395,
                693470779212674,
                1964537100203868,
                1350330550265229,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3344544372023708,
                720386671449874,
                2480841360702110,
                2036034126860286,
                2015744690201389,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1337446193390478,
                1984110761311871,
                746489405020285,
                407347127604128,
                1740475330360596,
            ]),
            xy2d: FieldElement51::from_limbs([
                140840424783613,
                1063284623568331,
                1136446106453878,
                372042229029799,
                442607248430694,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2330781679120937,
                376801425148230,
                2032603686676107,
                1488926293635130,
                1317278311532959,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1290116731380016,
                2166899563471713,
                831997001838078,
                870954980505220,
                2108537278055823,
            ]),
            xy2d: FieldElement51::from_limbs([
                1912719171026343,
                846194720551034,
                2043988124740726,
                993234269653961,
                421229796383281,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2651184584992902,
                2775702557638963,
                2539786009779572,
                2575974880015305,
                2122619079836732,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1154054290132562,
                931753998725577,
                1647742001778052,
                865765466488226,
                1083816107290025,
            ]),
            xy2d: FieldElement51::from_limbs([
                986341121095108,
                1522330369638573,
                1990880546211047,
                501525962272123,
                198539304862139,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1496414019192687,
                3991034436173951,
                3380311659062196,
                2854747485359158,
                3346958036643152,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                805612068303425,
                1891790027761335,
                1587008567571549,
                722120737390201,
                378156757163816,
            ]),
            xy2d: FieldElement51::from_limbs([
                1588994517921951,
                977362751042302,
                1329302387067714,
                2069348224564088,
                1586007159625211,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2490539421551682,
                1985699850375015,
                2331762317128172,
                4145097393776678,
                2521049460190674,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                615817553313996,
                2245962768078178,
                482564324326173,
                2101336843140780,
                1240914880829407,
            ]),
            xy2d: FieldElement51::from_limbs([
                1438242482238189,
                874267817785463,
                1620810389770625,
                866155221338671,
                1040426546798301,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                2403083624110300,
                2548561409802975,
                2492699136535911,
                2358289519456539,
                3203964320363148,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1913986535403097,
                1977163223054199,
                1972905914623196,
                1650122133472502,
                1905849310819035,
            ]),
            xy2d: FieldElement51::from_limbs([
                858174816360838,
                614595356564037,
                1099584959044836,
                636998087084906,
                1070393269058348,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3666695924830668,
                3585640662737501,
                2372994528684236,
                2628565977288995,
                3482812783469694,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1994161359147952,
                2198039369802658,
                62790022842537,
                1522306785848169,
                951223194802833,
            ]),
            xy2d: FieldElement51::from_limbs([
                852296621440717,
                431889737774209,
                370755457746189,
                437604073958073,
                627857326892757,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1794955764684156,
                2586904290013612,
                1322647643615887,
                856117964085888,
                2652432778663153,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                933592377399646,
                78031722952813,
                926049890685253,
                1471649501316246,
                33789909190376,
            ]),
            xy2d: FieldElement51::from_limbs([
                1479319468832059,
                203906207621608,
                659828362330083,
                44358398435755,
                1273573524210803,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1592342143350813,
                3227219208247713,
                2345240352078765,
                2577750109932929,
                2933512841197243,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2184946892642995,
                1517382324576002,
                1557940277419806,
                2170635134813213,
                747314658627002,
            ]),
            xy2d: FieldElement51::from_limbs([
                1823193620577742,
                1135817878516419,
                1731253819308581,
                1031652967267804,
                2123506616999453,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1346190246005805,
                2052692552023851,
                1718128041785940,
                2491557332978474,
                3474370880388305,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                424776012994573,
                281050757243423,
                626466040846420,
                990194703866532,
                38571969885982,
            ]),
            xy2d: FieldElement51::from_limbs([
                192408346595466,
                1054889725292349,
                584097975693004,
                1447909807397749,
                2134645004369136,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3169895788615063,
                3503097743181446,
                601598510029975,
                1422812237223371,
                2121009661378329,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1603348391996783,
                2066143816131699,
                1789627290363958,
                2145705961178118,
                1985578641438222,
            ]),
            xy2d: FieldElement51::from_limbs([
                352633958653380,
                856927627345554,
                793925083122702,
                93551575767286,
                1222010153634215,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1756866499986349,
                911731956999969,
                2707505543214075,
                4006920335263786,
                822501008147910,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1094036422864347,
                1897208881572508,
                1503607738246960,
                1901060196071406,
                294068411105729,
            ]),
            xy2d: FieldElement51::from_limbs([
                587776484399576,
                1116861711228807,
                343398777436088,
                936544065763093,
                1643746750211060,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                3477749685790410,
                267997399528836,
                2953780922004404,
                3252368924080907,
                3787792887348381,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                2042368155872443,
                41662387210459,
                1676313264498480,
                1333968523426810,
                1765708383352310,
            ]),
            xy2d: FieldElement51::from_limbs([
                1453394896690938,
                1585795827439909,
                1469309456804303,
                1294645324464404,
                2042954198665899,
            ]),
        },
        AffineNielsPoint {
            y_plus_x: FieldElement51::from_limbs([
                1810069207599881,
                1358344669503239,
                1989371257548167,
                2316270051121225,
                3019675451276507,
            ]),
            y_minus_x: FieldElement51::from_limbs([
                1866114438287676,
                1663420339568364,
                1437691317033088,
                538298302628038,
                1212711449614363,
            ]),
            xy2d: FieldElement51::from_limbs([
                1769235035677897,
                1562012115317882,
                31277513664750,
                536198657928416,
                1976134212537183,
            ]),
        },
    ]);
