package exhaustive

import (
	"go/parser"
	"go/token"
	"testing"
)

func TestContainsGeneratedComment(t *testing.T) {
	cases := []struct {
		comment string
		want    bool
	}{
		{"// Code generated DO NOT EDIT.", false},
		{"/* Code generated by some program. DO NOT EDIT.*/", false},
		{"/* Code generated by some program. DO NOT EDIT. */", false},
		{"// // Code generated by some program. DO NOT EDIT.", false},
		{"// Code generated by some program. DO NOT EDIT. ", false},
		{"// Code generated .", false},
		{" DO NOT EDIT.", false},
		{"Code generated by some program. DO NOT EDIT.", false},
		{"w", false},
		{"", false},
		{"//", false},
		{"// ", false},

		{"// Code generated by some program. DO NOT EDIT.", true},
		{"// Code generated  DO NOT EDIT.", true},
	}

	for _, tt := range cases {
		got := isGeneratedFileComment(tt.comment)
		if tt.want != got {
			t.Errorf("%s: want %v, got %v", tt.comment, tt.want, got)
		}
	}
}

func TestIsGeneratedFile(t *testing.T) {
	t.Run("is generated", func(t *testing.T) {
		const source = `
	// Code generated by some thing. DO NOT EDIT.

package foo
func f() {
}`
		f, err := parser.ParseFile(token.NewFileSet(), "", source, parser.ParseComments)
		assertNoError(t, err)
		if !isGeneratedFile(f) {
			t.Errorf("unexpectedly false")
		}
	})

	t.Run("is not generated", func(t *testing.T) {
		const source = `package foo
func f() {
}`
		f, err := parser.ParseFile(token.NewFileSet(), "", source, parser.ParseComments)
		assertNoError(t, err)
		if isGeneratedFile(f) {
			t.Errorf("unexpectedly true")
		}
	})
}
