﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/apigatewayv2/ApiGatewayV2Request.h>
#include <aws/apigatewayv2/ApiGatewayV2_EXPORTS.h>
#include <aws/apigatewayv2/model/Cors.h>
#include <aws/apigatewayv2/model/IpAddressType.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace ApiGatewayV2 {
namespace Model {

/**
 * <p>Updates an Api.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/apigatewayv2-2018-11-29/UpdateApiRequest">AWS
 * API Reference</a></p>
 */
class UpdateApiRequest : public ApiGatewayV2Request {
 public:
  AWS_APIGATEWAYV2_API UpdateApiRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateApi"; }

  AWS_APIGATEWAYV2_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The API identifier.</p>
   */
  inline const Aws::String& GetApiId() const { return m_apiId; }
  inline bool ApiIdHasBeenSet() const { return m_apiIdHasBeenSet; }
  template <typename ApiIdT = Aws::String>
  void SetApiId(ApiIdT&& value) {
    m_apiIdHasBeenSet = true;
    m_apiId = std::forward<ApiIdT>(value);
  }
  template <typename ApiIdT = Aws::String>
  UpdateApiRequest& WithApiId(ApiIdT&& value) {
    SetApiId(std::forward<ApiIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An API key selection expression. Supported only for WebSocket APIs. See <a
   * href="https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-websocket-api-selection-expressions.html#apigateway-websocket-api-apikey-selection-expressions">API
   * Key Selection Expressions</a>.</p>
   */
  inline const Aws::String& GetApiKeySelectionExpression() const { return m_apiKeySelectionExpression; }
  inline bool ApiKeySelectionExpressionHasBeenSet() const { return m_apiKeySelectionExpressionHasBeenSet; }
  template <typename ApiKeySelectionExpressionT = Aws::String>
  void SetApiKeySelectionExpression(ApiKeySelectionExpressionT&& value) {
    m_apiKeySelectionExpressionHasBeenSet = true;
    m_apiKeySelectionExpression = std::forward<ApiKeySelectionExpressionT>(value);
  }
  template <typename ApiKeySelectionExpressionT = Aws::String>
  UpdateApiRequest& WithApiKeySelectionExpression(ApiKeySelectionExpressionT&& value) {
    SetApiKeySelectionExpression(std::forward<ApiKeySelectionExpressionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A CORS configuration. Supported only for HTTP APIs.</p>
   */
  inline const Cors& GetCorsConfiguration() const { return m_corsConfiguration; }
  inline bool CorsConfigurationHasBeenSet() const { return m_corsConfigurationHasBeenSet; }
  template <typename CorsConfigurationT = Cors>
  void SetCorsConfiguration(CorsConfigurationT&& value) {
    m_corsConfigurationHasBeenSet = true;
    m_corsConfiguration = std::forward<CorsConfigurationT>(value);
  }
  template <typename CorsConfigurationT = Cors>
  UpdateApiRequest& WithCorsConfiguration(CorsConfigurationT&& value) {
    SetCorsConfiguration(std::forward<CorsConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>This property is part of quick create. It specifies the credentials required
   * for the integration, if any. For a Lambda integration, three options are
   * available. To specify an IAM Role for API Gateway to assume, use the role's
   * Amazon Resource Name (ARN). To require that the caller's identity be passed
   * through from the request, specify arn:aws:iam::*:user/ *. To use resource-based
   * permissions on supported AWS services, don't specify this parameter. Currently,
   * this property is not used for HTTP integrations. If provided, this value
   * replaces the credentials associated with the quick create integration. Supported
   * only for HTTP APIs.</p>
   */
  inline const Aws::String& GetCredentialsArn() const { return m_credentialsArn; }
  inline bool CredentialsArnHasBeenSet() const { return m_credentialsArnHasBeenSet; }
  template <typename CredentialsArnT = Aws::String>
  void SetCredentialsArn(CredentialsArnT&& value) {
    m_credentialsArnHasBeenSet = true;
    m_credentialsArn = std::forward<CredentialsArnT>(value);
  }
  template <typename CredentialsArnT = Aws::String>
  UpdateApiRequest& WithCredentialsArn(CredentialsArnT&& value) {
    SetCredentialsArn(std::forward<CredentialsArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the API.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  UpdateApiRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Avoid validating models when creating a deployment. Supported only for
   * WebSocket APIs.</p>
   */
  inline bool GetDisableSchemaValidation() const { return m_disableSchemaValidation; }
  inline bool DisableSchemaValidationHasBeenSet() const { return m_disableSchemaValidationHasBeenSet; }
  inline void SetDisableSchemaValidation(bool value) {
    m_disableSchemaValidationHasBeenSet = true;
    m_disableSchemaValidation = value;
  }
  inline UpdateApiRequest& WithDisableSchemaValidation(bool value) {
    SetDisableSchemaValidation(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies whether clients can invoke your API by using the default
   * execute-api endpoint. By default, clients can invoke your API with the default
   * https://{api_id}.execute-api.{region}.amazonaws.com endpoint. To require that
   * clients use a custom domain name to invoke your API, disable the default
   * endpoint.</p>
   */
  inline bool GetDisableExecuteApiEndpoint() const { return m_disableExecuteApiEndpoint; }
  inline bool DisableExecuteApiEndpointHasBeenSet() const { return m_disableExecuteApiEndpointHasBeenSet; }
  inline void SetDisableExecuteApiEndpoint(bool value) {
    m_disableExecuteApiEndpointHasBeenSet = true;
    m_disableExecuteApiEndpoint = value;
  }
  inline UpdateApiRequest& WithDisableExecuteApiEndpoint(bool value) {
    SetDisableExecuteApiEndpoint(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The IP address types that can invoke your API or domain name.</p>
   */
  inline IpAddressType GetIpAddressType() const { return m_ipAddressType; }
  inline bool IpAddressTypeHasBeenSet() const { return m_ipAddressTypeHasBeenSet; }
  inline void SetIpAddressType(IpAddressType value) {
    m_ipAddressTypeHasBeenSet = true;
    m_ipAddressType = value;
  }
  inline UpdateApiRequest& WithIpAddressType(IpAddressType value) {
    SetIpAddressType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the API.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  UpdateApiRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>This property is part of quick create. If not specified, the route created
   * using quick create is kept. Otherwise, this value replaces the route key of the
   * quick create route. Additional routes may still be added after the API is
   * updated. Supported only for HTTP APIs.</p>
   */
  inline const Aws::String& GetRouteKey() const { return m_routeKey; }
  inline bool RouteKeyHasBeenSet() const { return m_routeKeyHasBeenSet; }
  template <typename RouteKeyT = Aws::String>
  void SetRouteKey(RouteKeyT&& value) {
    m_routeKeyHasBeenSet = true;
    m_routeKey = std::forward<RouteKeyT>(value);
  }
  template <typename RouteKeyT = Aws::String>
  UpdateApiRequest& WithRouteKey(RouteKeyT&& value) {
    SetRouteKey(std::forward<RouteKeyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The route selection expression for the API. For HTTP APIs, the
   * routeSelectionExpression must be ${request.method} ${request.path}. If not
   * provided, this will be the default for HTTP APIs. This property is required for
   * WebSocket APIs.</p>
   */
  inline const Aws::String& GetRouteSelectionExpression() const { return m_routeSelectionExpression; }
  inline bool RouteSelectionExpressionHasBeenSet() const { return m_routeSelectionExpressionHasBeenSet; }
  template <typename RouteSelectionExpressionT = Aws::String>
  void SetRouteSelectionExpression(RouteSelectionExpressionT&& value) {
    m_routeSelectionExpressionHasBeenSet = true;
    m_routeSelectionExpression = std::forward<RouteSelectionExpressionT>(value);
  }
  template <typename RouteSelectionExpressionT = Aws::String>
  UpdateApiRequest& WithRouteSelectionExpression(RouteSelectionExpressionT&& value) {
    SetRouteSelectionExpression(std::forward<RouteSelectionExpressionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>This property is part of quick create. For HTTP integrations, specify a fully
   * qualified URL. For Lambda integrations, specify a function ARN. The type of the
   * integration will be HTTP_PROXY or AWS_PROXY, respectively. The value provided
   * updates the integration URI and integration type. You can update a quick-created
   * target, but you can't remove it from an API. Supported only for HTTP APIs.</p>
   */
  inline const Aws::String& GetTarget() const { return m_target; }
  inline bool TargetHasBeenSet() const { return m_targetHasBeenSet; }
  template <typename TargetT = Aws::String>
  void SetTarget(TargetT&& value) {
    m_targetHasBeenSet = true;
    m_target = std::forward<TargetT>(value);
  }
  template <typename TargetT = Aws::String>
  UpdateApiRequest& WithTarget(TargetT&& value) {
    SetTarget(std::forward<TargetT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A version identifier for the API.</p>
   */
  inline const Aws::String& GetVersion() const { return m_version; }
  inline bool VersionHasBeenSet() const { return m_versionHasBeenSet; }
  template <typename VersionT = Aws::String>
  void SetVersion(VersionT&& value) {
    m_versionHasBeenSet = true;
    m_version = std::forward<VersionT>(value);
  }
  template <typename VersionT = Aws::String>
  UpdateApiRequest& WithVersion(VersionT&& value) {
    SetVersion(std::forward<VersionT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_apiId;
  bool m_apiIdHasBeenSet = false;

  Aws::String m_apiKeySelectionExpression;
  bool m_apiKeySelectionExpressionHasBeenSet = false;

  Cors m_corsConfiguration;
  bool m_corsConfigurationHasBeenSet = false;

  Aws::String m_credentialsArn;
  bool m_credentialsArnHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  bool m_disableSchemaValidation{false};
  bool m_disableSchemaValidationHasBeenSet = false;

  bool m_disableExecuteApiEndpoint{false};
  bool m_disableExecuteApiEndpointHasBeenSet = false;

  IpAddressType m_ipAddressType{IpAddressType::NOT_SET};
  bool m_ipAddressTypeHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::String m_routeKey;
  bool m_routeKeyHasBeenSet = false;

  Aws::String m_routeSelectionExpression;
  bool m_routeSelectionExpressionHasBeenSet = false;

  Aws::String m_target;
  bool m_targetHasBeenSet = false;

  Aws::String m_version;
  bool m_versionHasBeenSet = false;
};

}  // namespace Model
}  // namespace ApiGatewayV2
}  // namespace Aws
