﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/connect/model/EncryptionType.h>
#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>
#include <aws/core/utils/HashingUtils.h>

using namespace Aws::Utils;

namespace Aws {
namespace Connect {
namespace Model {
namespace EncryptionTypeMapper {

static const int KMS_HASH = HashingUtils::HashString("KMS");

EncryptionType GetEncryptionTypeForName(const Aws::String& name) {
  int hashCode = HashingUtils::HashString(name.c_str());
  if (hashCode == KMS_HASH) {
    return EncryptionType::KMS;
  }
  EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
  if (overflowContainer) {
    overflowContainer->StoreOverflow(hashCode, name);
    return static_cast<EncryptionType>(hashCode);
  }

  return EncryptionType::NOT_SET;
}

Aws::String GetNameForEncryptionType(EncryptionType enumValue) {
  switch (enumValue) {
    case EncryptionType::NOT_SET:
      return {};
    case EncryptionType::KMS:
      return "KMS";
    default:
      EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
      if (overflowContainer) {
        return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
      }

      return {};
  }
}

}  // namespace EncryptionTypeMapper
}  // namespace Model
}  // namespace Connect
}  // namespace Aws
