﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>
#include <aws/core/utils/HashingUtils.h>
#include <aws/customer-profiles/model/Type.h>

using namespace Aws::Utils;

namespace Aws {
namespace CustomerProfiles {
namespace Model {
namespace TypeMapper {

static const int ALL_HASH = HashingUtils::HashString("ALL");
static const int ANY_HASH = HashingUtils::HashString("ANY");
static const int NONE_HASH = HashingUtils::HashString("NONE");

Type GetTypeForName(const Aws::String& name) {
  int hashCode = HashingUtils::HashString(name.c_str());
  if (hashCode == ALL_HASH) {
    return Type::ALL;
  } else if (hashCode == ANY_HASH) {
    return Type::ANY;
  } else if (hashCode == NONE_HASH) {
    return Type::NONE;
  }
  EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
  if (overflowContainer) {
    overflowContainer->StoreOverflow(hashCode, name);
    return static_cast<Type>(hashCode);
  }

  return Type::NOT_SET;
}

Aws::String GetNameForType(Type enumValue) {
  switch (enumValue) {
    case Type::NOT_SET:
      return {};
    case Type::ALL:
      return "ALL";
    case Type::ANY:
      return "ANY";
    case Type::NONE:
      return "NONE";
    default:
      EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
      if (overflowContainer) {
        return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
      }

      return {};
  }
}

}  // namespace TypeMapper
}  // namespace Model
}  // namespace CustomerProfiles
}  // namespace Aws
