package ntlmssp

import (
	"bytes"
	"crypto/rc4"
	"errors"
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestNonce(t *testing.T) {
	tables := []struct {
		got, want int
		err       error
	}{
		{8, 8, nil},
	}

	for _, table := range tables {
		got, err := nonce(table.got)
		assert.Equal(t, table.want, len(got))
		assert.Equal(t, table.err, err)

		dup, err := nonce(table.got)
		assert.Equal(t, table.want, len(dup))
		assert.Equal(t, table.err, err)

		assert.NotEqual(t, got, dup)
	}
}

func TestCreateDESKey(t *testing.T) {
	tables := []struct {
		got, want []byte
		err       error
	}{
		// Examples from Davenport
		{
			[]byte{
				0x53, 0x45, 0x43, 0x52, 0x45, 0x54, 0x30,
			},
			[]byte{
				0x52, 0xa2, 0x51, 0x6b, 0x25, 0x2a, 0x51, 0x61,
			},
			nil,
		},
		{
			[]byte{
				0x31, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			[]byte{
				0x31, 0x80, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			},
			nil,
		},
		// Some bad inputs
		{
			[]byte{0x01},
			nil,
			errors.New("need at least 7 bytes"),
		},
		{
			nil,
			nil,
			errors.New("need at least 7 bytes"),
		},
	}

	for _, table := range tables {
		got, err := createDESKey(table.got)
		assert.Equal(t, table.want, got)
		assert.Equal(t, table.err, err)
	}
}

func TestEncryptDES(t *testing.T) {
	tables := []struct {
		k, d, want []byte
		err        error
	}{
		{
			[]byte{
				0x53, 0x45, 0x43, 0x52, 0x45, 0x54, 0x30,
			},
			[]byte(lmCiphertext),
			[]byte{
				0xff, 0x37, 0x50, 0xbc, 0xc2, 0xb2, 0x24, 0x12,
			},
			nil,
		},
		{
			[]byte{
				0x31, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			[]byte(lmCiphertext),
			[]byte{
				0xc2, 0x26, 0x5b, 0x23, 0x73, 0x4e, 0x0d, 0xac,
			},
			nil,
		},
		// Bad inputs
		{
			[]byte{0x01},
			[]byte(lmCiphertext),
			nil,
			errors.New("need at least 7 bytes"),
		},
	}

	for _, table := range tables {
		got, err := encryptDES(table.k, table.d)
		assert.Equal(t, table.want, got)
		assert.Equal(t, table.err, err)
	}
}

func TestEncryptDESL(t *testing.T) {
	tables := []struct {
		k, d, want []byte
		err        error
	}{
		{
			[]byte{
				0xff, 0x37, 0x50, 0xbc, 0xc2, 0xb2, 0x24, 0x12,
				0xc2, 0x26, 0x5b, 0x23, 0x73, 0x4e, 0x0d, 0xac,
			},
			[]byte{
				0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
			},
			[]byte{
				0xc3, 0x37, 0xcd, 0x5c, 0xbd, 0x44, 0xfc, 0x97,
				0x82, 0xa6, 0x67, 0xaf, 0x6d, 0x42, 0x7c, 0x6d,
				0xe6, 0x7c, 0x20, 0xc2, 0xd3, 0xe7, 0x7c, 0x56,
			},
			nil,
		},
		{
			[]byte{
				0xcd, 0x06, 0xca, 0x7c, 0x7e, 0x10, 0xc9, 0x9b,
				0x1d, 0x33, 0xb7, 0x48, 0x5a, 0x2e, 0xd8, 0x08,
			},
			[]byte{
				0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
			},
			[]byte{
				0x25, 0xa9, 0x8c, 0x1c, 0x31, 0xe8, 0x18, 0x47,
				0x46, 0x6b, 0x29, 0xb2, 0xdf, 0x46, 0x80, 0xf3,
				0x99, 0x58, 0xfb, 0x8c, 0x21, 0x3a, 0x9c, 0xc6,
			},
			nil,
		},
	}

	for _, table := range tables {
		got, err := encryptDESL(table.k, table.d)
		assert.Equal(t, table.want, got)
		assert.Equal(t, table.err, err)
	}
}

func TestEncryptRC4(t *testing.T) {
	tables := []struct {
		key, first, second []byte
	}{
		// 40-bit
		{
			[]byte{
				0x01, 0x02, 0x03, 0x04, 0x05,
			},
			[]byte{
				0xb2, 0x39, 0x63, 0x05, 0xf0, 0x3d, 0xc0, 0x27,
				0xcc, 0xc3, 0x52, 0x4a, 0x0a, 0x11, 0x18, 0xa8,
			},
			[]byte{
				0x69, 0x82, 0x94, 0x4f, 0x18, 0xfc, 0x82, 0xd5,
				0x89, 0xc4, 0x03, 0xa4, 0x7a, 0x0d, 0x09, 0x19,
			},
		},
		// 56-bit
		{
			[]byte{
				0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			},
			[]byte{
				0x29, 0x3f, 0x02, 0xd4, 0x7f, 0x37, 0xc9, 0xb6,
				0x33, 0xf2, 0xaf, 0x52, 0x85, 0xfe, 0xb4, 0x6b,
			},
			[]byte{
				0xe6, 0x20, 0xf1, 0x39, 0x0d, 0x19, 0xbd, 0x84,
				0xe2, 0xe0, 0xfd, 0x75, 0x20, 0x31, 0xaf, 0xc1,
			},
		},
		// 128-bit
		{
			[]byte{
				0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
				0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
			},
			[]byte{
				0x9a, 0xc7, 0xcc, 0x9a, 0x60, 0x9d, 0x1e, 0xf7,
				0xb2, 0x93, 0x28, 0x99, 0xcd, 0xe4, 0x1b, 0x97,
			},
			[]byte{
				0x52, 0x48, 0xc4, 0x95, 0x90, 0x14, 0x12, 0x6a,
				0x6e, 0x8a, 0x84, 0xf1, 0x1d, 0x1a, 0x9e, 0x1c,
			},
		},
	}

	for _, table := range tables {
		// Encryption
		handle, err := initRC4(table.key)
		assert.Nil(t, err)
		assert.Equal(t, table.first, encryptRC4(handle, bytes.Repeat([]byte{0x00}, 16)))
		assert.Equal(t, table.second, encryptRC4(handle, bytes.Repeat([]byte{0x00}, 16)))

		// Decryption
		handle, err = initRC4(table.key)
		assert.Nil(t, err)
		assert.Equal(t, bytes.Repeat([]byte{0x00}, 16), encryptRC4(handle, table.first))
		assert.Equal(t, bytes.Repeat([]byte{0x00}, 16), encryptRC4(handle, table.second))
	}
}

func TestEncryptRC4K(t *testing.T) {
	tables := []struct {
		k, d, want []byte
		err        error
	}{
		{
			[]byte{
				0x3f, 0x37, 0x3e, 0xa8, 0xe4, 0xaf, 0x95, 0x4f,
				0x14, 0xfa, 0xa5, 0x06, 0xf8, 0xee, 0xbd, 0xc4,
			},
			[]byte{
				0xf0, 0xf0, 0xaa, 0xbb, 0x00, 0x11, 0x22, 0x33,
				0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xaa, 0xbb,
			},
			[]byte{
				0x1d, 0x33, 0x55, 0xeb, 0x71, 0xc8, 0x28, 0x50,
				0xa9, 0xa2, 0xd6, 0x5c, 0x29, 0x52, 0xe6, 0xf3,
			},
			nil,
		},
		// Bad key
		{
			nil,
			[]byte{
				0xf0, 0xf0, 0xaa, 0xbb, 0x00, 0x11, 0x22, 0x33,
				0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xaa, 0xbb,
			},
			[]byte{
				0x1d, 0x33, 0x55, 0xeb, 0x71, 0xc8, 0x28, 0x50,
				0xa9, 0xa2, 0xd6, 0x5c, 0x29, 0x52, 0xe6, 0xf3,
			},
			rc4.KeySizeError(0),
		},
	}

	for _, table := range tables {
		got, err := encryptRC4K(table.k, table.d)
		assert.Equal(t, table.err, err)
		if err == nil {
			assert.Equal(t, table.want, got)
		}
	}
}

func TestHashMD4(t *testing.T) {
	tables := []struct {
		got, want []byte
	}{
		{
			[]byte{
				0x53, 0x00, 0x65, 0x00, 0x63, 0x00, 0x52, 0x00,
				0x45, 0x00, 0x74, 0x00, 0x30, 0x00, 0x31, 0x00,
			},
			[]byte{
				0xcd, 0x06, 0xca, 0x7c, 0x7e, 0x10, 0xc9, 0x9b,
				0x1d, 0x33, 0xb7, 0x48, 0x5a, 0x2e, 0xd8, 0x08,
			},
		},
	}

	for _, table := range tables {
		assert.Equal(t, table.want, hashMD4(table.got))
	}
}

func TestHashMD5(t *testing.T) {
	tables := []struct {
		got, want []byte
	}{
		{
			[]byte{
				0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
				0xff, 0xff, 0xff, 0x00, 0x11, 0x22, 0x33, 0x44,
			},
			[]byte{
				0xbe, 0xac, 0x9a, 0x1b, 0xc5, 0xa9, 0x86, 0x7c,
				0x15, 0x19, 0x2b, 0x31, 0x05, 0xd5, 0xbe, 0xb1,
			},
		},
	}

	for _, table := range tables {
		assert.Equal(t, table.want, hashMD5(table.got))
	}
}

func TestHmacMD5(t *testing.T) {
	tables := []struct {
		k, m, want []byte
	}{
		{
			[]byte{
				0xcd, 0x06, 0xca, 0x7c, 0x7e, 0x10, 0xc9, 0x9b,
				0x1d, 0x33, 0xb7, 0x48, 0x5a, 0x2e, 0xd8, 0x08,
			},
			[]byte{
				0x55, 0x00, 0x53, 0x00, 0x45, 0x00, 0x52, 0x00,
				0x44, 0x00, 0x4f, 0x00, 0x4d, 0x00, 0x41, 0x00,
				0x49, 0x00, 0x4e, 0x00,
			},
			[]byte{
				0x04, 0xb8, 0xe0, 0xba, 0x74, 0x28, 0x9c, 0xc5,
				0x40, 0x82, 0x6b, 0xab, 0x1d, 0xee, 0x63, 0xae,
			},
		},
	}

	for _, table := range tables {
		assert.Equal(t, table.want, hmacMD5(table.k, table.m))
	}
}

func TestHashCRC32(t *testing.T) {
	tables := []struct {
		got, want []byte
	}{
		{
			[]byte{
				0x50, 0x00, 0x6c, 0x00, 0x61, 0x00, 0x69, 0x00,
				0x6e, 0x00, 0x74, 0x00, 0x65, 0x00, 0x78, 0x00,
				0x74, 0x00,
			},
			[]byte{
				0x7d, 0x84, 0xaa, 0x93,
			},
		},
	}

	for _, table := range tables {
		assert.Equal(t, table.want, hashCRC32(table.got))
	}
}
