/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import { TokenInfo } from "@gnu-taler/taler-util";
import { useTranslationContext } from "@gnu-taler/web-util/browser";
import { format } from "date-fns";
import { Fragment, h, VNode } from "preact";
import { StateUpdater, useState } from "preact/hooks";
import {
  datetimeFormatForSettings,
  usePreference,
} from "../../../../hooks/preference.js";

const TALER_SCREEN_ID = 32;

type Entity = TokenInfo;

interface Props {
  tokens: Entity[];
  onDelete: (e: Entity) => void;
  onCreate: () => void;
  onLoadMoreBefore?: () => void;
  onLoadMoreAfter?: () => void;
}

export function CardTable({
  tokens,
  onCreate,
  onDelete,
  onLoadMoreAfter,
  onLoadMoreBefore,
}: Props): VNode {
  const [rowSelection, rowSelectionHandler] = useState<string[]>([]);

  const { i18n } = useTranslationContext();

  return (
    <div class="card has-table">
      <header class="card-header">
        <p class="card-header-title">
          <span class="icon">
            <i class="mdi mdi-key" />
          </span>
          <i18n.Translate>Access tokens</i18n.Translate>
        </p>
        <div class="card-header-icon" aria-label="more options">
          <span
            class="has-tooltip-left"
            data-tooltip={i18n.str`Create access token`}
          >
            <button
              class="button is-info"
              type="button"
              accessKey="+"
              onClick={onCreate}
            >
              <span class="icon is-small">
                <i class="mdi mdi-plus mdi-36px" />
              </span>
            </button>
          </span>
        </div>
      </header>
      <div class="card-content">
        <div class="b-table has-pagination">
          <div class="table-wrapper has-mobile-cards">
            {tokens.length > 0 ? (
              <Table
                tokens={tokens}
                onDelete={onDelete}
                rowSelection={rowSelection}
                rowSelectionHandler={rowSelectionHandler}
                onLoadMoreAfter={onLoadMoreAfter}
                onLoadMoreBefore={onLoadMoreBefore}
              />
            ) : (
              <EmptyTable />
            )}
          </div>
        </div>
      </div>
    </div>
  );
}
interface TableProps {
  rowSelection: string[];
  tokens: Entity[];
  onDelete: (e: Entity) => void;
  rowSelectionHandler: StateUpdater<string[]>;
  onLoadMoreBefore?: () => void;
  onLoadMoreAfter?: () => void;
}

function Table({
  tokens,
  onDelete,
  onLoadMoreAfter,
  onLoadMoreBefore,
}: TableProps): VNode {
  const { i18n } = useTranslationContext();
  const [settings] = usePreference();
  return (
    <Fragment>
      <div class="table-container">
        {onLoadMoreBefore && (
          <button
            type="button"
            class="button is-fullwidth"
            data-tooltip={i18n.str`Load more devices before the first one`}
            onClick={onLoadMoreBefore}
          >
            <i18n.Translate>Load first page</i18n.Translate>
          </button>
        )}

        <table class="table is-fullwidth is-striped is-hoverable is-fullwidth">
          <thead>
            <tr>
              <th>
                <i18n.Translate>Scope</i18n.Translate>
              </th>
              <th>
                <i18n.Translate>Created at</i18n.Translate>
              </th>
              <th>
                <i18n.Translate>Expires at</i18n.Translate>
              </th>
              <th>
                <i18n.Translate>Description</i18n.Translate>
              </th>
              <th />
            </tr>
          </thead>
          <tbody>
            {tokens.map((t, idx) => {
              return (
                <tr key={idx}>
                  <td
                  // onClick={(): void => onSelect(t)}
                  // style={{ cursor: "pointer" }}
                  >
                    {t.scope}
                  </td>
                  <td
                  // onClick={(): void => onSelect(t)}
                  // style={{ cursor: "pointer" }}
                  >
                    {t.creation_time.t_s === "never"
                      ? i18n.str`Never`
                      : format(
                          new Date(t.creation_time.t_s * 1000),
                          datetimeFormatForSettings(settings),
                        )}
                  </td>
                  <td
                  // onClick={(): void => onSelect(t)}
                  // style={{ cursor: "pointer" }}
                  >
                    {t.expiration.t_s === "never"
                      ? i18n.str`Never`
                      : format(
                          new Date(t.expiration.t_s * 1000),
                          datetimeFormatForSettings(settings),
                        )}
                  </td>
                  <td
                  // onClick={(): void => onSelect(t)}
                  // style={{ cursor: "pointer" }}
                  >
                    {t.description}
                  </td>
                  <td class="is-actions-cell right-sticky">
                    <div class="buttons is-right">
                      <span
                        class="has-tooltip-left"
                        data-tooltip={i18n.str`Remove this access token`}
                      >
                        <button
                          class="button is-small is-danger"
                          type="button"
                          onClick={(): void => onDelete(t)}
                        >
                          <i18n.Translate>Delete</i18n.Translate>
                        </button>
                      </span>
                    </div>
                  </td>
                </tr>
              );
            })}
          </tbody>
        </table>
        {onLoadMoreAfter && (
          <button
            type="button"
            class="button is-fullwidth"
            data-tooltip={i18n.str`Load more devices after the last one`}
            onClick={onLoadMoreAfter}
          >
            <i18n.Translate>Load next page</i18n.Translate>
          </button>
        )}
      </div>
    </Fragment>
  );
}

function EmptyTable(): VNode {
  const { i18n } = useTranslationContext();
  return (
    <div class="content has-text-grey has-text-centered">
      <p>
        <span class="icon is-large">
          <i class="mdi mdi-magnify mdi-48px" />
        </span>
      </p>
      <p>
        <i18n.Translate>
          There are no active sessions yet, add one by pressing the + sign
        </i18n.Translate>
      </p>
    </div>
  );
}
