use crate::conventional::changelog::release::Release;
use crate::conventional::changelog::renderer::Renderer;

use crate::conventional::changelog::error::ChangelogError;
use crate::conventional::changelog::template::{MonoRepoContext, PackageContext, Template};

use std::fs;
use std::path::Path;

pub mod error;
pub(crate) mod release;
pub(crate) mod renderer;
pub(crate) mod serde;
pub mod template;

const CHANGELOG_SEPARATOR: &str = "- - -";

const DEFAULT_HEADER: &str =
    "# Changelog\nAll notable changes to this project will be documented in this file. \
See [conventional commits](https://www.conventionalcommits.org/) for commit guidelines.\n\n- - -\n";

const DEFAULT_FOOTER: &str =
    "Changelog generated by [cocogitto](https://github.com/cocogitto/cocogitto).";

pub enum ReleaseType<'a> {
    Standard,
    MonoRepo(MonoRepoContext<'a>),
    Package(PackageContext<'a>),
}

impl Release<'_> {
    pub fn into_markdown(
        self,
        template: Template,
        context: ReleaseType,
    ) -> Result<String, tera::Error> {
        let renderer = Renderer::try_new(template)?;

        let mut renderer = match context {
            ReleaseType::Standard => renderer,
            ReleaseType::MonoRepo(context) => renderer.with_monorepo_context(context),
            ReleaseType::Package(context) => renderer.with_package_context(context),
        };

        renderer.render(self)
    }

    pub fn write_to_file<S: AsRef<Path>>(
        self,
        path: S,
        template: Template,
        kind: ReleaseType,
    ) -> Result<(), ChangelogError> {
        let mut changelog_content = fs::read_to_string(path.as_ref())
            .unwrap_or_else(|_| [DEFAULT_HEADER, DEFAULT_FOOTER].join(""));

        let changelog = self.into_markdown(template, kind)?;
        let separator_idx = changelog_content.find(CHANGELOG_SEPARATOR);

        if let Some(idx) = separator_idx {
            changelog_content.insert(idx + CHANGELOG_SEPARATOR.len(), '\n');
            changelog_content.insert_str(idx + CHANGELOG_SEPARATOR.len() + 1, &changelog);
            changelog_content.insert_str(
                idx + CHANGELOG_SEPARATOR.len() + 1 + changelog.len(),
                "\n- - -\n",
            );
            fs::write(path.as_ref(), changelog_content)?;

            Ok(())
        } else {
            Err(ChangelogError::SeparatorNotFound(
                path.as_ref().to_path_buf(),
            ))
        }
    }
}
